<?php
/**
* @version $Id: cbpaidsubscriptions.worldpay.php 1581 2012-12-24 02:36:44Z beat $
* @package CBSubs (TM) Community Builder Plugin for Paid Subscriptions (TM)
* @subpackage Plugin for Paid Subscriptions
* @copyright (C) 2007-2022 and Trademark of Lightning MultiCom SA, Switzerland - www.joomlapolis.com - and its licensors, all rights reserved
* @license http://www.gnu.org/licenses/old-licenses/gpl-2.0.html GNU/GPL version 2
*/

use CBLib\Registry\ParamsInterface;
use CBLib\Language\CBTxt;

/** Ensure this file is being included by a parent file */
if ( ! ( defined( '_VALID_CB' ) || defined( '_JEXEC' ) || defined( '_VALID_MOS' ) ) ) { die( 'Direct Access to this location is not allowed.' ); }

global $_CB_framework;

// Avoids errors in CB plugin edit:
/** @noinspection PhpIncludeInspection */
include_once( $_CB_framework->getCfg( 'absolute_path' ) . '/components/com_comprofiler/plugin/user/plug_cbpaidsubscriptions/cbpaidsubscriptions.class.php' );

// This gateway implements a payment handler using a hosted page at the PSP:
// Import class cbpaidHostedPagePayHandler that extends cbpaidPayHandler
// and implements all gateway-generic CBSubs methods.

/**
 * Payment handler class for this gateway: Handles all payment events and notifications, called by the parent class:
 *
 * OEM base
 * Please note that except the constructor and the API version this class does not implement any public methods.
 */
class cbpaidworldpayoem extends cbpaidHostedPagePayHandler
{
	/**
	 * Gateway API version used
	 * @var int
	 */
	public $gatewayApiVersion	=	"1.3.0";

	/**
	 * Constructor
	 *
	 * @param cbpaidGatewayAccount $account
	 */
	public function __construct( $account )
	{
		parent::__construct( $account );

		// Set gateway URLS for $this->pspUrl() results: first 2 are the main hosted payment page posting URL, next ones are gateway-specific:
		$this->_gatewayUrls	=	array(	'psp+normal'	 => $this->getAccountParam( 'psp_normal_url' ),
										'psp+test'		 => $this->getAccountParam( 'psp_test_url' ) );
	}

	/**
	 * CBSUBS HOSTED PAGE PAYMENT API METHODS:
	 */

	/**
	 * Returns single payment request parameters for gateway depending on basket (without specifying payment type)
	 *
	 * @param  cbpaidPaymentBasket  $paymentBasket   paymentBasket object
	 * @return array                                 Returns array $requestParams
	 */
	protected function getSinglePaymentRequstParams( $paymentBasket )
	{
		// build hidden form fields or redirect to gateway url parameters array:
		$requestParams	=	$this->_getBasicRequstParams( $paymentBasket );

		// sign single payment params:
		$this->_signRequestParams( $requestParams );

		return $requestParams;
	}

	/**
	 * Optional function: only needed for recurring payments:
	 * Returns subscription request parameters for gateway depending on basket (without specifying payment type)
	 *
	 * @param  cbpaidPaymentBasket  $paymentBasket   paymentBasket object
	 * @return array                                 Returns array $requestParams
	 */
	protected function getSubscriptionRequstParams( $paymentBasket )
	{
		// mandatory parameters:
		$requestParams								=	$this->_getBasicRequstParams( $paymentBasket );

		// check for subscription or if single payment:
		if ( $paymentBasket->period3 ) {
			// change type to recurring (futurepay):
			$requestParams['futurePayType']			=	'regular';
			$requestParams['option']				=	0;

			// calculate duration and type:
			list( $duration1, $type1 )				=	$this->_periodsLimits( explode( ' ', $paymentBasket->period3 ) );

			// determine if there is a trial (initial) period:
			if ( $paymentBasket->period1 ) {
				list( $duration2, $type2 )			=	$this->_periodsLimits( explode( ' ', $paymentBasket->period1 ) );

				$requestParams['amount']			=	sprintf( '%.2f', $paymentBasket->mc_amount1 );
				$requestParams['startDelayUnit']	=	$type2;
				$requestParams['startDelayMult']	=	$duration2;
			} else {
				$requestParams['startDelayUnit']	=	$type1;
				$requestParams['startDelayMult']	=	$duration1;
			}

			// add subscription specific parameters:
			$requestParams['intervalUnit']			=	$type1;
			$requestParams['intervalMult']			=	$duration1;
			$requestParams['normalAmount']			=	sprintf( '%.2f', $paymentBasket->mc_amount3 );
			$requestParams['noOfPayments']			=	$paymentBasket->recur_times;
		}

		// sign reocurring payment params:
		$this->_signRequestParams( $requestParams );

		return $requestParams;
	}


	/**
	 * The user got redirected back from the payment service provider with a success message: Let's see how successfull it was
	 *
	 * @param  cbpaidPaymentBasket  $paymentBasket  New empty object. returning: includes the id of the payment basket of this callback (strictly verified, otherwise untouched)
	 * @param  array                $postdata       _POST data for saving edited tab content as generated with getEditTab
	 * @return string                               HTML to display if frontend, FALSE if XML error (and not yet ErrorMSG generated), or NULL if nothing to display
	 */
	protected function handleReturn( $paymentBasket, $postdata )
	{
		if ( ( count( $postdata ) > 0 ) && isset( $postdata['cartId'] ) ) {
			// we prefer POST for sensitive data:
			$requestdata			=	$postdata;
		} else {
			// but if customer needs GET, we will work with it too (removing CMS/CB/CBSubs specific routing params):
			$requestdata			=	$this->_getGetParams();
		}

		// check if its a return link clicked in payment page:
		$paymentBasketId			=	(int) cbGetParam( $requestdata, 'cartId', null );

		if ( ! $paymentBasketId ) {
			$requestdata['cartId']	=	(int) cbGetParam( $_GET, 'cbpbasket', null );
		}

		return $this->_returnParamsHandler( $paymentBasket, $requestdata, 'R' );
	}

	/**
	 * The user cancelled his payment
	 *
	 * @param  cbpaidPaymentBasket  $paymentBasket  New empty object. returning: includes the id of the payment basket of this callback (strictly verified, otherwise untouched)
	 * @param  array                $postdata       _POST data for saving edited tab content as generated with getEditTab
	 * @return string                               HTML to display, FALSE if registration cancelled and ErrorMSG generated, or NULL if nothing to display
	 */
	protected function handleCancel( $paymentBasket, $postdata )
	{
		// The user cancelled his payment (and registration):
		if ( $this->hashPdtBackCheck( $this->_getReqParam( 'pdtback' ) ) ) {
			$paymentBasketId					=	(int) $this->_getReqParam( 'basket' );

			// check if cancel was from gateway:
			if ( ! $paymentBasketId ) {
				$paymentBasketId				=	(int) cbGetParam( $postdata, 'cartId', null );
			}

			$exists								=	$paymentBasket->load( (int) $paymentBasketId );

			if ( $exists && ( $this->_getReqParam( 'id' ) == $paymentBasket->shared_secret ) && ( $paymentBasket->payment_status != 'Completed' ) ) {
				$paymentBasket->payment_status	=	'RedisplayOriginalBasket';

				$this->_setErrorMSG( CBTxt::T( 'Payment cancelled.' ) );
			}
		}

		return false;
	}

	/**
	 * The payment service provider server did a server-to-server notification: Verify and handle it here:
	 *
	 * @param  cbpaidPaymentBasket  $paymentBasket  New empty object. returning: includes the id of the payment basket of this callback (strictly verified, otherwise untouched)
	 * @param  array                $postdata       _POST data for saving edited tab content as generated with getEditTab
	 * @return string                              Text to return to gateway if notification, or NULL if nothing to display
	 */
	protected function handleNotification( $paymentBasket, $postdata )
	{
		if ( ( count( $postdata ) > 0 ) && isset( $postdata['cartId'] ) ) {
			// we prefer POST for sensitive data:
			$requestdata	=	$postdata;
		} else {
			// but if gateway needs GET, we will work with it too:
			$requestdata	=	$this->_getGetParams();
		}

		if ( cbGetParam( $requestdata, 'transStatus' ) == 'C' ) {
			return $this->handleCancel( $paymentBasket, $postdata );
		} else {
			return $this->_returnParamsHandler( $paymentBasket, $requestdata, 'I' );
		}
	}

	/**
	 * Cancels an existing recurring subscription
	 *
	 * @param  cbpaidPaymentBasket  $paymentBasket  paymentBasket object
	 * @param  cbpaidPaymentItem[]  $paymentItems   redirect immediately instead of returning HTML for output
	 * @return boolean|string                       TRUE if unsubscription done successfully, STRING if error
	 */
	protected function handleStopPaymentSubscription( $paymentBasket, $paymentItems )
	{
		global $_CB_framework;

		$return									=	false;

		// only for recurring subscriptions:
		if ( $paymentBasket->mc_amount3 ) {
			// only if an actual subscription id exists:
			if ( $paymentBasket->subscr_id ) {
				$remoteid						=	$this->getAccountParam( 'pspremoteid' );

				// mandatory parameters:
				$requestParams					=	array();
				$requestParams['instId']		=	( $remoteid ? $remoteid : $this->getAccountParam( 'pspid' ) );
				$requestParams['authPW']		=	$this->getAccountParam( 'pspremotepsw' );
				$requestParams['futurePayId']	=	$paymentBasket->subscr_id;
				$requestParams['op-cancelFP']	=	null;

				$response						=	null;
				$status							=	null;
				$error							=	$this->_httpsRequest( str_replace( 'purchase', 'iadmin', $this->gatewayUrl( 'psp' ) ), $requestParams, 45, $response, $status, 'post' );

				if ( $error || ( $status != 200 ) || ( ! $response ) ) {
					$this->_setLogErrorMSG( 3, null, $this->getPayName() . ': unsubscribe API response failed', CBTxt::T( 'Submitted unsubscription failed on-site.' ) . ' ' . CBTxt::T( 'Please contact site administrator to check error log.' ) );
				} else {
					if ( substr( $response, 0, 1 ) == 'Y' ) {
						$ipn					=	$this->_prepareIpn( 'R', $paymentBasket->payment_status, $paymentBasket->payment_type, 'Unsubscribe', $_CB_framework->now(), 'utf-8' );
						$ipn->test_ipn			=	$paymentBasket->test_ipn;
						$ipn->raw_result		=	'SUCCESS';
						$ipn->raw_data			=	'$message_type="STOP_PAYMENT_SUBSCRIPTION"' . ";\n"
												.	/* cbGetParam() not needed: we want raw info */ '$response=' . var_export( $response, true ) . ";\n"
												.	/* cbGetParam() not needed: we want raw info */ '$_GET=' . var_export( $_GET, true ) . ";\n"
												.	/* cbGetParam() not needed: we want raw info */ '$_POST=' . var_export( $_POST, true ) . ";\n";

						$ipn->bindBasket( $paymentBasket );

						$ipn->sale_id			=	$paymentBasket->id;
						$ipn->txn_id			=	$paymentBasket->txn_id;
						$ipn->subscr_id			=	$paymentBasket->subscr_id;
						$ipn->txn_type			=	'subscr_cancel';

						$this->_storeIpnResult( $ipn, 'SUCCESS' );
						$this->_bindIpnToBasket( $ipn, $paymentBasket );

						$return					=	true;
					} else {
						$this->_setLogErrorMSG( 3, null, $this->getPayName() . ': ' . $response, CBTxt::T( 'Sorry, the payment server replied with an error.' ) . ' ' . CBTxt::T( 'Please contact site administrator to check payment status and error log.' ) );
					}
				}
			} else {
				$this->_setLogErrorMSG( 3, null, $this->getPayName() . ': unsubscribe failed from missing subscr_id in payment basket', CBTxt::T( 'Submitted unsubscription failed on-site.' ) . ' ' . CBTxt::T( 'Please contact site administrator to check error log.' ) );
			}
		}

		return $return;
	}

	/**
	 * sign payment request $requestParams with validation code added to $requestParams array
	 *
	 * @param  array     $requestParams        The keyed parameters array to add generate validation to $requestParams['signature']
	 */
	private function _signRequestParams( &$requestParams )
	{
		// List of parameters to concatenate if not empty:
		$listOfParams						=	array(	'$pspsecret',	// Concatenate security code generated at gateway: equivalent to: $this->getAccountParam( 'pspsecret' )
														'INSTID',		// installation id (pspid)
														'CARTID',		// item code (basket id)
														'AMOUNT',		// purchase price (10.01, etc..)
														'CURRENCY',		// currency code
														'MC_USERID'		// user id
													);

		// Concatenate them using this payments concatenation function with $caseInsensitiveKeys = true:
		$string								=	$this->_concatVars( $requestParams, $listOfParams, null, ':', '', false, false, true, false );

		// add validation code doing md5 of the string without uppercasing:
		$requestParams['signatureFields']	=	'instId:cartId:amount:currency:MC_userId';
		$requestParams['signature']			=	$this->_hashString( $string, 'md5', false );
		$requestParams['M_signature']		=	$requestParams['signature'];
	}

	/**
	 * Validate a reply in $requestParams using md5check value, and then also through the API if it is available
	 *
	 * @param  array   $requestParams  Returned in case of valid and verified through API too: API reply, raw format
	 * @param  boolean $rawParams      $requestParams are raw $_POST or $_GET input and should be sanitized and unescaped if needed
	 * @param  boolean $recurring      If the validation is for a recurring payment or not
	 * @return boolean                 TRUE: valid, FALSE: invalid
	 */
	private function _pspVerifySignature( $requestParams, $rawParams = false, $recurring = false )
	{
		// IPNs don't return custom parameters; so IPN validation needs to be callback password only:
		if ( $recurring ) {
			// confirm response password:
			$valid			=	( cbGetParam( $requestParams, 'callbackPW' ) == $this->getAccountParam( 'pspresponsepsw' ) );
		} else {
			// List of parameters to concatenate if not empty:
			$listOfParams	=	array(	'$pspsecret',	// Concatenate security code generated at gateway: equivalent to: $this->getAccountParam( 'pspsecret' )
										'INSTID',		// installation id (pspid)
										'CARTID',		// item code (basket id)
										'AMOUNT',		// purchase price (10.01, etc..)
										'CURRENCY',		// currency code
										'MC_USERID'		// user id
									);

			// Concatenate them using this payments concatenation function with $caseInsensitiveKeys = true:
			$string			=	$this->_concatVars( $requestParams, $listOfParams, null, ':', '', false, false, true, false, $rawParams );

			// confirm validation:
			$valid			=	( cbGetParam( $requestParams, 'M_signature' ) == $this->_hashString( $string, 'md5', false ) );

			if ( $valid ) {
				// confirm response password:
				$valid		=	( cbGetParam( $requestParams, 'callbackPW' ) == $this->getAccountParam( 'pspresponsepsw' ) );
			}
		}

		return $valid;
	}

	/**
	 * Compute the CBSubs payment_status based on gateway's reply in $postdata:
	 *
	 * @param  array   $postdata  raw POST data received from the payment gateway
	 * @param  string  $reason    OUT: reason_code
     * @return string             CBSubs payment status
	 */
	private function _paymentStatus( $postdata, &$reason )
	{
		$status			=	cbGetParam( $postdata, 'transStatus' );

		switch ( $status ) {
			case 'Y':
				$reason	=	null;
				$status	=	'Completed';
				break;
			case 'C':
				$reason	=	'Payment cancelled';
				$status	=	'Denied';
				break;
		}

		return $status;
	}

	/**
	 * Compute the CBSubs payment_type based on gateway's reply $postdata:
	 *
	 * @param  array   $postdata raw POST data received from the payment gateway
	 * @return string  Human-readable string
	 */
	private function _getPaymentType( $postdata )
	{
		$type			=	cbGetParam( $postdata, 'cardType' );

		switch ( strtolower( $type ) ) {
			case 'amex':
				$type	=	'Amex';
                break;
			case 'dins':
				$type	=	'Diners Card';
                break;
            case 'elv':
				$type	=	'ELV';
                break;
            case 'jcb':
				$type	=	'JCB Card';
                break;
            case 'mscd':
				$type	=	'MasterCard';
                break;
            case 'solo':
				$type	=	'Solo Card';
                break;
            case 'swit':
				$type	=	'Maestro';
                break;
            case 'visa':
				$type	=	'Visa Card';
                break;
            case 'visd':
				$type	=	'Visa Delta';
                break;
            case 'vied':
				$type	=	'Visa Electron';
                break;
            case 'vsip':
				$type	=	'Visa Purchasing';
                break;
            default:
				break;
		}

		return $type;
	}

	/**
	 * parse period into period and type then calculate new period limitations:
	 *
	 * @param array $periodTypeArray
	 * @return array
	 */
	private function _periodsLimits( $periodTypeArray )
	{
		// parse period into period and type:
		$p		=	$periodTypeArray[0];
		$t		=	$periodTypeArray[1];

		// change single letter type to digit identifier and calculate start date:
		if ( $t == 'Y' ) {
			$t	=	4;
		} elseif ( $t == 'M' ) {
			$t	=	3;
		} elseif  ( $t == 'W' ) {
			$t	=	2;
		} elseif ( $t == 'D' ) {
			$t	=	1;
		}

		return array( $p, $t );
	}

	/**
	 * Popoulates basic request parameters for gateway depending on basket (without specifying payment type)
	 *
	 * @param  cbpaidPaymentBasket  $paymentBasket   paymentBasket object
	 * @return array                                 Returns array $requestParams
	 */
	private function _getBasicRequstParams( $paymentBasket )
	{
		// mandatory parameters:
		$requestParams							=	array();
		$requestParams['instId']				=	$this->getAccountParam( 'pspid' );
		$requestParams['lang']					=	$this->getAccountParam( 'language' );
		$requestParams['cartId']				=	$paymentBasket->id;
		$requestParams['amount']				=	sprintf( '%.2f', $paymentBasket->mc_gross );
		$requestParams['currency']				=	$paymentBasket->mc_currency;

		// optional parameters:
		$requestParams['MC_userId']				=	$paymentBasket->user_id;
		$requestParams['desc']					=	$paymentBasket->item_name;
		$requestParams['testMode']				=	( (int) $this->getAccountParam( 'normal_gateway' ) == 0 ? 100 : 0 );

		// recommended anti-fraud fields:
		$requestParams['name']					=	$paymentBasket->address_name;

		if ( $this->getAccountParam( 'givehiddenemail' ) && ( strlen( $paymentBasket->payer_email ) <= 50 ) ) {
			$requestParams['email']				=	$paymentBasket->payer_email;
		}

		if ( $this->getAccountParam( 'givehiddenbilladdress' ) ) {
			cbimport( 'cb.tabs' ); // needed for cbIsoUtf_substr()

			$addressFields						=	array(	'address1' => array( $paymentBasket->address_street, 30 ),
															'postcode' => array( $paymentBasket->address_zip, 10 ),
															'town' => array( $paymentBasket->address_city, 30 ),
															'country' => array( $paymentBasket->getInvoiceCountry( 2 ), 3 ),
															'countryString' => array( $paymentBasket->address_country, 30 ),
															'region' => array( $paymentBasket->getInvoiceState(), 30 )
														);

			foreach ( $addressFields as $k => $value_maxlength ) {
				$adrField						=	cbIsoUtf_substr( $value_maxlength[0], 0, $value_maxlength[1] );

				if ( $adrField ) {
					$requestParams[$k]			=	$adrField;
				}
			}
		}

		if ( $this->getAccountParam( 'givehiddenshipaddress' ) ) {
			cbimport( 'cb.tabs' ); // needed for cbIsoUtf_substr()

			$addressFields						=	array(	'delvAddress1' => array( $paymentBasket->address_street, 30 ),
															'delvPostcode' => array( $paymentBasket->address_zip, 10 ),
															'delvTown' => array( $paymentBasket->address_city, 30 ),
															'delvCountry' => array( $paymentBasket->getInvoiceCountry( 2 ), 3 ),
															'delvCountryString' => array( $paymentBasket->address_country, 30 ),
															'delvRegion' => array( $paymentBasket->getInvoiceState(), 30 )
														);

			foreach ( $addressFields as $k => $value_maxlength ) {
				$adrField						=	cbIsoUtf_substr( $value_maxlength[0], 0, $value_maxlength[1] );

				if ( $adrField ) {
					$requestParams[$k]			=	$adrField;
				}
			}

			$requestParams['withDelivery']		=	null;
		}

		if ( $this->getAccountParam( 'givehiddentelno' ) && ( strlen( $paymentBasket->contact_phone ) <= 50 ) ) {
			$requestParams['tel']				=	$paymentBasket->contact_phone;
		}

		$requestParams['hideCurrency']			=	null;

		return $requestParams;
	}


	/**
	 * The user got redirected back from the payment service provider with a success message: let's see how successfull it was
	 *
	 * @param  cbpaidPaymentBasket  $paymentBasket       New empty object. returning: includes the id of the payment basket of this callback (strictly verified, otherwise untouched)
	 * @param  array                $requestdata         Data returned by gateway
	 * @param  string               $type                Type of return ('R' for PDT, 'I' for INS, 'A' for Autorecurring payment (Vault) )
     * @param  array                $additionalLogData   Additional strings to log with IPN
     * @return string                                    HTML to display if frontend, text to return to gateway if notification, FALSE if registration cancelled and ErrorMSG generated, or NULL if nothing to display
	 */
	private function _returnParamsHandler( $paymentBasket, $requestdata, $type, $additionalLogData = null )
	{
		global $_CB_framework, $_GET, $_POST;

		$ret													=	null;
		$paymentBasketId										=	(int) cbGetParam( $requestdata, 'cartId', null );

		if ( $paymentBasketId ) {
			$exists												=	$paymentBasket->load( (int) $paymentBasketId );

			if ( $exists && ( ( cbGetParam( $requestdata, $this->_getPagingParamName( 'id' ), 0 ) == $paymentBasket->shared_secret ) && ( ! ( ( $type == 'R' ) && ( $paymentBasket->payment_status == 'Completed' ) ) ) ) ) {
				// Log the return record:
				$log_type										=	$type;
				$reason											=	null;
				$paymentStatus									=	$this->_paymentStatus( $requestdata, $reason );
				$paymentType									=	$this->_getPaymentType( $requestdata );
				$paymentTime									=	$_CB_framework->now();

				if ( $paymentStatus == 'Error' ) {
					$errorTypes									=	array( 'I' => 'D', 'R' => 'E' );

					if ( isset( $errorTypes[$type] ) ) {
						$log_type								=	$errorTypes[$type];
					}
				}

				$ipn											=	$this->_prepareIpn( $log_type, $paymentStatus, $paymentType, $reason, $paymentTime, 'utf-8' );

				if ( $paymentStatus == 'Refunded' ) {
					// in case of refund we need to log the payment as it has same TnxId as first payment: so we need payment_date for discrimination:
					$ipn->payment_date							=	gmdate( 'H:i:s M d, Y T', $paymentTime ); // paypal-style
				}

				$ipn->test_ipn									=	( (int) cbGetParam( $requestdata, 'testmode' ) > 0 ? 1 : 0 );
				$ipn->raw_data									=	'$message_type="' . ( $type == 'R' ? 'RETURN_TO_SITE' : ( $type == 'I' ? 'NOTIFICATION' : 'UNKNOWN' ) ) . '";' . "\n";

				if ( $additionalLogData ) {
					foreach ( $additionalLogData as $k => $v ) {
						$ipn->raw_data							.=	'$' . $k . '="' . var_export( $v, true ) . '";' . "\n";
					}
				}

				$ipn->raw_data									.=	/* cbGetParam() not needed: we want raw info */ '$requestdata=' . var_export( $requestdata, true ) . ";\n"
																.	/* cbGetParam() not needed: we want raw info */ '$_GET=' . var_export( $_GET, true ) . ";\n"
																.	/* cbGetParam() not needed: we want raw info */ '$_POST=' . var_export( $_POST, true ) . ";\n";

				if ( $paymentStatus == 'Error' ) {
					$paymentBasket->reason_code					=	$reason;

					$this->_storeIpnResult( $ipn, 'ERROR:' . $reason );
					$this->_setLogErrorMSG( 4, $ipn, $this->getPayName() . ': ' . $reason, CBTxt::T( 'Sorry, the payment server replied with an error.' ) . ' ' . CBTxt::T( 'Please contact site administrator to check payment status and error log.' ) );

					$ret										=	false;
				} else {
					$ipn->bindBasket( $paymentBasket );

					$ipn->sale_id								=	$paymentBasketId;

					$insToIpn									=	array(	'txn_id' => 'transId',
																			'mc_gross' => 'amount',
																			'mc_currency' => 'currency',
																			'address_name' => 'name',
																			'address_street' => 'address1',
																			'address_zip' => 'postcode',
																			'address_city' => 'town',
																			'address_country' => 'countryString',
																			'address_country_code' => 'country',
																			'address_state' => 'region',
																			'contact_phone' => 'tel',
																			'payer_email' => 'email',
																			'payer_id' => 'MC_userId',
																			'residence_country' => 'countryString',
																			'business' => 'instId',
																			'receiver_id' => 'instId'
																		);

					foreach ( $insToIpn as $k => $v ) {
						$ipn->$k								=	cbGetParam( $requestdata, $v );
					}

					if ( ! $ipn->payer_id ) {
						// IPNs don't send custom parameters so lets set the payer id back to user id from basket:
						$ipn->payer_id							=	$paymentBasket->user_id;
					}

					// try to build first and last name from name:
					$name										=	cbGetParam( $requestdata, 'name' );

					if ( $name ) {
						$card_names								=	explode( ' ', $name, 2 );

						if ( count( $card_names ) < 2 ) {
							$card_names							=	array( $name, '' );
						}

						$ipn->first_name						=	$card_names[0];
						$ipn->last_name							=	$card_names[1];
					}

					$card_match									=	(int) substr( cbGetParam( $requestdata, 'AVS' ), 0, 1 );

					if ( $card_match == 2 ) {
						$ipn->payer_status						=	'verified';
					}

					$postcode_match								=	(int) substr( cbGetParam( $requestdata, 'AVS' ), 1, 1 );
					$address_match								=	(int) substr( cbGetParam( $requestdata, 'AVS' ), 2, 1 );
					$country_match								=	(int) substr( cbGetParam( $requestdata, 'AVS' ), 3, 1 );

					if ( in_array( $postcode_match, array( 0, 1, 2 ) ) && in_array( $address_match, array( 0, 1, 2 ) ) && in_array( $country_match, array( 0, 1, 2 ) ) ) {
						$ipn->address_status					=	'confirmed';
					} else {
						$ipn->address_status					=	'unconfirmed';
					}

					$ipn->user_id								=	(int) $paymentBasket->user_id;

					// check what type of purchase this is:
					$recurring									=	( cbGetParam( $requestdata, 'futurePayId' ) ? true : false );

					// handle recurring subscriptions properly or default to single payment:
					if ( $recurring ) {
						if ( ( $paymentStatus == 'Completed' ) && ( ! $paymentBasket->subscr_id ) ) {
							$ipn->txn_type						=	'subscr_signup';
							$ipn->subscr_id						=	cbGetParam( $requestdata, 'futurePayId' );
							$ipn->subscr_date					=	$ipn->payment_date;
						} elseif ( $paymentStatus == 'Denied' ) {
							if ( ( $paymentBasket->reattempts_tried + 1 ) <= cbpaidScheduler::getInstance( $this )->retries ) {
								$ipn->txn_type					=	'subscr_failed';
							} else {
								$ipn->txn_type					=	'subscr_cancel';
							}
						} elseif ( in_array( $paymentStatus, array( 'Completed', 'Processed', 'Pending' ) ) ) {
							$ipn->txn_type						=	'subscr_payment';
						}
					} else {
						$ipn->txn_type							=	'web_accept';
					}

					// validate payment from PDT or IPN
					if ( $this->_pspVerifySignature( $requestdata, true, ( $ipn->txn_type == 'subscr_payment' ) ) ) {
						if ( ( $paymentBasketId == cbGetParam( $requestdata, 'cartId', null ) ) && ( ( sprintf( '%.2f', $paymentBasket->mc_gross ) == $ipn->mc_gross ) || ( $ipn->payment_status == 'Refunded' ) ) && ( $paymentBasket->mc_currency == $ipn->mc_currency ) ) {
							if ( in_array( $ipn->payment_status, array( 'Completed', 'Processed', 'Pending', 'Refunded', 'Denied' ) ) ) {
								$this->_storeIpnResult( $ipn, 'SUCCESS' );
								$this->_bindIpnToBasket( $ipn, $paymentBasket );

								// add the gateway to the basket:
								$paymentBasket->payment_method	=	$this->getPayName();
								$paymentBasket->gateway_account	=	$this->getAccountParam( 'id' );

								// 0: not auto-recurring, 1: auto-recurring without payment processor notifications, 2: auto-renewing with processor notifications updating $expiry_date:
								$autorecurring_type				=	( in_array( $ipn->txn_type, array( 'subscr_payment', 'subscr_signup', 'subscr_modify', 'subscr_eot', 'subscr_cancel', 'subscr_failed' ) ) ? 2 : 0 );

								// 0: not auto-renewing (manual renewals), 1: asked for by user, 2: mandatory by configuration:
								$autorenew_type					=	( $autorecurring_type ? ( ( $this->getAccountParam( 'enabled', 0 ) == 3 ) && ( $paymentBasket->isAnyAutoRecurring() == 2 ) ? 1 : 2 ) : 0 );

								if ( $recurring ) {
									$paymentBasket->reattempt	=	1; // we want to reattempt auto-recurring payment in case of failure
								}

								$this->updatePaymentStatus( $paymentBasket, $ipn->txn_type, $ipn->payment_status, $ipn, 1, $autorecurring_type, $autorenew_type, false );

								if ( in_array( $ipn->payment_status, array( 'Completed', 'Processed', 'Pending' ) ) ) {
									$ret						=	true;
								}
							} else {
								$this->_storeIpnResult( $ipn, 'FAILED' );

								$paymentBasket->payment_status	=	$ipn->payment_status;

								$this->_setErrorMSG( '<div class="alert alert-info">' . $this->getTxtNextStep( $paymentBasket ) . '</div>' );

								$paymentBasket->payment_status	=	'RedisplayOriginalBasket';
								$ret							=	false;
							}
						} else {
							$this->_storeIpnResult( $ipn, 'MISMATCH' );
							$this->_setLogErrorMSG( 3, $ipn, $this->getPayName() . ': amount or currency missmatch', CBTxt::T( 'Sorry, the payment does not match the basket.' ) . ' ' . CBTxt::T( 'Please contact site administrator to check error log.' ) );

							$ret								=	false;
						}
					} else {
						$this->_storeIpnResult( $ipn, 'SIGNERROR' );
						$this->_setLogErrorMSG( 3, $ipn, $this->getPayName() . ': md5check or transaction does not match with gateway. Please check Secret Key setting', CBTxt::T( 'The Secret Key signature is incorrect.' ) . ' ' . CBTxt::T( 'Please contact site administrator to check error log.' ) );

						$ret									=	false;
					}
				}
			}
		} else {
			$rawData											=	/* cbGetParam() not needed: we want raw info */ '$requestdata=' . var_export( $requestdata, true ) . ";\n"
																.	/* cbGetParam() not needed: we want raw info */ '$_GET=' . var_export( $_GET, true ) . ";\n"
																.	/* cbGetParam() not needed: we want raw info */ '$_POST=' . var_export( $_POST, true ) . ";\n";

			$this->_setLogErrorMSG( 3, null, $this->getPayName() . ': cartId is missing in the return URL: ' . $rawData, null );
		}

		if ( ( $type == 'I' ) && $ret ) {
			$paymentPage										=	$this->getAccountParam( 'payment_page_custom' );

			 if ( trim( strip_tags( $paymentPage ) ) != '' ) {
				$cbUser											=	CBuser::getInstance( (int) $paymentBasket->user_id, false );

				$extraStrings									=	$paymentBasket->substitutionStrings( true );
				$extraStrings['WORLDPAY_BANNER']				=	'<WPDISPLAY ITEM=banner>';
				$extraStrings['LINK_BACK_TO_SITE']				=	'<a href="' . $this->getSuccessUrl( $paymentBasket ) . '">' . CBTxt::T( 'Click here to return to [sitename].', null, array( '[sitename]' => $_CB_framework->getCfg( 'sitename' ) ) ) . '</a>';

				$paymentPageCSS									=	$this->getAccountParam( 'payment_page_custom_css' );

				$ret											=	( $paymentPageCSS ? '<style type="text/css">' . $paymentPageCSS. '</style>' : null )
																.	$cbUser->replaceUserVars( $paymentPage, false, true, $extraStrings, true );
			}
		}

		return  $ret;
	}
}

/**
 * Payment account class for this gateway: Stores the settings for that gateway instance, and is used when editing and storing gateway parameters in the backend.
 *
 * OEM base
 * No methods need to be implemented or overriden in this class, except to implement the private-type params used specifically for this gateway:
 */
class cbpaidGatewayAccountworldpayoem extends cbpaidGatewayAccounthostedpage
{

	/**
	 * USED by XML interface ONLY !!! Renders URL for successful returns
	 *
	 * @param  string              $value   Variable value ( 'successurl', 'cancelurl', 'notifyurl' )
	 * @param  ParamsInterface     $params
	 * @param  string              $name    The name of the form element
	 * @param  CBSimpleXMLElement  $node    The xml element for the parameter
	 * @return string                       HTML to display
	 */
	public function renderUrl( $value, $params, $name, $node ) {
		return $this->getPayMean()->adminUrlRender( $node->attributes( 'value' ) );
	}
}
/**
 * Payment handler class for this gateway: Handles all payment events and notifications, called by the parent class:
 *
 * Gateway-specific
 * Please note that except the constructor and the API version this class does not implement any public methods.
 */
class cbpaidworldpay extends cbpaidworldpayoem
{
}

/**
 * Payment account class for this gateway: Stores the settings for that gateway instance, and is used when editing and storing gateway parameters in the backend.
 *
 * Gateway-specific
 * No methods need to be implemented or overriden in this class, except to implement the private-type params used specifically for this gateway:
 */
class cbpaidGatewayAccountworldpay extends cbpaidGatewayAccountworldpayoem
{
}
