<?php
/**
* @version $Id: cbpaidsubscriptions.twocheckout_salesapi.php 1556 2012-12-20 14:47:51Z beat $
* @package CBSubs (TM) Community Builder Plugin for Paid Subscriptions (TM)
* @subpackage CBSubs 2checkout gateway - 2co API
* @copyright (C) 2007-2022 and Trademark of Lightning MultiCom SA, Switzerland - www.joomlapolis.com - and its licensors, all rights reserved
* @license http://www.gnu.org/licenses/old-licenses/gpl-2.0.html GNU/GPL version 2
*/

use CBLib\Xml\SimpleXMLElement;

/** ensure this file is being included by a parent file */
if ( ! ( defined( '_VALID_CB' ) || defined( '_JEXEC' ) || defined( '_VALID_MOS' ) ) ) { die( 'Direct Access to this location is not allowed.' ); }

/**
* Paid Subscriptions 2co API class
*/
class cbpaidtwocheckoutAPI
{
	public $apiUrl		=	'https://www.2checkout.com/api';
	protected $username;
	protected $password;

	/**
	 * Constructor
	 *
	 * @param  string  $username
	 * @param  string  $password
	 */
	public function __construct( $username, $password )
	{
		$this->username		=	$username;
		$this->password		=	$password;
	}

	/**
	 * Posts a POST form by https if available, otherwise by http and gets result.
	 *
	 * @param  string        $urlNoHttpsPrefix   URL without https:// in front
	 * @param  array|string  $formvars           Variables in form to post
	 * @param  int           $timeout            Timeout of the access
	 * @param  string        $result             RETURNING: the fetched access
	 * @param  int           $status             RETURNING: the status of the access (e.g. 200 is normal)
	 * @param  string        $getPostType        'post' (default) or 'get'
	 * @param  string        $contentType        'normal' (default) or 'xml' ($formvars['xml']=xml content) or 'json' (application/json)
	 * @param  string        $acceptType         '* / *' (default) or 'application/xml' or 'application/json'
	 * @param  boolean       $https              SSL protocol (default: true)
	 * @param  int           $port               port number (default: 443)
	 * @param  string        $username           HTTP username authentication
	 * @param  string        $password           HTTP password authentication
	 * @param  boolean       $allowHttpFallback  Allow fallback to http if https not available locally (default: false)
	 * @param  string        $referer            referrer
	 * @return int                               error-code of access (0 for ok)
	 */
	private function _httpsRequest( $urlNoHttpsPrefix, $formvars, $timeout, &$result, &$status, $getPostType = 'post', $contentType='normal', $acceptType='*/*', $https = true, $port = 443, $username = '', $password = '', $allowHttpFallback = false, $referer = null )
	{
		return cbpaidWebservices::httpsRequest( $urlNoHttpsPrefix, $formvars, $timeout, $result, $status, $getPostType, $contentType, $acceptType, $https, $port, $username, $password, $allowHttpFallback, $referer );
	}

	/**
	 * Adds raw data $result to $rawData as api_reply PHP-language string
	 *
	 * @param  string       $result   data string to add
	 * @param  string|null  $rawData  IN+OUT: string to append to
	 */
	private function _addRawData( $result, &$rawData )
	{
		if ( $rawData ) {
			$rawData						.=	'$api_reply="'
											.	str_replace( '"', "\\\"", $result )
											.	"\"\n";
		}
	}

	/**
	 * Executes the remote call to the PSP API
	 *
	 * @param  array   $formVars            Name-Value pairs to post
	 * @param  string  $getPostType         'post' or 'get
	 * @param  string  $apiSubtree          the API call "subdirectory"
	 * @param  string  $resultSubArrayName  Expected reply's main XML tag
	 * @param  string  $raw_data            IN+OUTPUT: raw data returned by the gateway is added there for notification logging
	 * @return SimpleXMLElement|string      OK: SimpleXMLElement, KO: string with error
	 */
	private function _api_xml_fetch( $formVars, $getPostType, $apiSubtree, $resultSubArrayName, &$raw_data )
	{
		$result								=	null;
		$status								=	null;
		$error	=	$this->_httpsRequest( $this->apiUrl . '/' . $apiSubtree, $formVars, 60, $result, $status, $getPostType, 'xml', 'application/xml', true, 443, $this->username, $this->password, false, null );
		if ( ( $error == 0 ) && in_array( $status, array( 200, 400 ) ) ) {
			if ( isset( $result[0] ) && ( $result[0] == '<' ) ) {
				// fix API bug: non-XML-compliant tag name:
				$result						=	iconv("UTF-8","UTF-8//IGNORE",$result );
				$this->_addRawData( $result, $raw_data );
				$result						=	str_replace( array( '<2co_fees>', '</2co_fees>'),array( '<fees_2co>', '</fees_2co>'), $result );
				$xdoc						=	@new SimpleXMLElement( $result, LIBXML_NONET | ( defined('LIBXML_COMPACT') ? LIBXML_COMPACT : 0 ) );
				if ( $xdoc && count( $xdoc->children() ) > 0 ) {
 					if ( $status == 200 ) {
						$xresp_code			=	$xdoc->getChildByNameAttributes( 'response_code' );
						if ( ( $xresp_code !== false ) && ( $xresp_code->data() == 'OK' ) ) {
							if ( isset( $xdoc->$resultSubArrayName ) ) {
								$return		=	$xdoc->$resultSubArrayName;
							} else {
								$return		=	sprintf( '2checkout: API %s fetch error: returned XML is missing required element "%s" : ', $apiSubtree, $resultSubArrayName, $result );
							}
						} else {
							$return			=	sprintf( '2checkout: API %s fetch error: returned XML has response code : "%s" and response message: %s', $apiSubtree, $xresp_code, $xdoc->getChildByNameAttributes( 'response_message' ) );
						}
 					} elseif ( $status == 400 ) {
						$xresp_errors		=	$xdoc->getChildByNameAttributes( 'errors' );
						if ( ( $xresp_errors !== false ) && isset( $xresp_errors->code ) && isset( $xresp_errors->message ) ) {
							$return			=	sprintf( '2checkout: API %s error: code: %s, message: %s with URL: %s and posted variables: %s', $apiSubtree, $xresp_errors->code, $xresp_errors->message, $this->apiUrl . '/' . $apiSubtree, http_build_query( $formVars, '', '&' ) );
						} else {
							$return			=	sprintf( '2checkout: API %s fetch error: error status: %d, unknown error record.', $apiSubtree, $status );
						}
 					} else {
						 $return			=	sprintf( '2checkout: API %s fetch error: error status: %d, unknown error record.', $apiSubtree, $status );
					 }
				} else {
					$return					=	sprintf( '2checkout: API %s fetch error: returned XML incomplete : %s', $apiSubtree, $result );
				}
			} else {
				$return						=	sprintf( '2checkout: API %s fetch error: returned result seems not XML : %s', $apiSubtree, $result );
			}
		} else {
			$return							=	sprintf( '2checkout: API %s fetch error: error status: %d.', $apiSubtree, $status );
		}
		if ( is_string( $return ) ) {
			global $_CB_database;
			$logObject						=	new cbpaidHistory( $_CB_database );
			$null							=	null;
			$logObject->logError( 3, $return, $null );
		}
		return $return;
	}

	/**
	 * Gets Company details with API acct/detail_company_info
	 *
	 * @param  string                   $vendor_id  Vendor id
	 * @param  string                   $raw_data   IN+OUT: Raw data for notification logging
	 * @return SimpleXMLElement|string              OK: SimpleXMLElement, KO: string with error
	 */
	public function _api_detail_company_info( /** @noinspection PhpUnusedParameterInspection */ $vendor_id, &$raw_data )
	{
		$formVars					=	array(); 	// 'vendor_id' => $vendor_id );
		return $this->_api_xml_fetch( $formVars, 'get/1.0', 'acct/detail_company_info', 'vendor_company_info', $raw_data );
		// return $this->_api_fetch( $formVars, 'get', 'acct/detail_company_info', 'vendor_company_info' );
	}

	/**
	 * 2co API interface for: products/detail_product
	 *
	 * @param  string                   $vendor_id   Vendor id
	 * @param  string                   $product_id  Product id
	 * @param  string                   $raw_data    IN+OUT: Raw data for notification logging
	 * @return SimpleXMLElement|string               OK: SimpleXMLElement, KO: string with error
	 */
	public function _api_detail_product( $vendor_id, $product_id, &$raw_data )
	{
		$formVars					=	array(); 	// 'vendor_id' => $vendor_id );
		$get						=	array();
		if ( $vendor_id ) {
			$get[]					=	'vendor_id=' .urlencode( $vendor_id );
		}
		if ( $product_id ) {
			$get[]					=	'product_id=' . urlencode( $product_id );
		}
		$get						=	implode( '&', $get );
		return $this->_api_xml_fetch( $formVars, 'get/1.0', 'products/detail_product' . ( $get ? '?' . $get : '' ), 'product', $raw_data );
		// return $this->_api_fetch( $formVars, 'get', 'acct/detail_company_info', 'vendor_company_info' );
	}

	/**
	 * 2co API interface for: sales/detail_sale
	 *
	 * @param  string                   $sale_id     Sale id
	 * @param  string                   $invoice_id  Invoice id
	 * @param  string                   $raw_data    IN+OUT: Raw data for notification logging
	 * @return SimpleXMLElement|string               OK: SimpleXMLElement, KO: string with error
	 */
	public function _api_detail_sale( $sale_id, $invoice_id, &$raw_data )
	{
		$formVars					=	array();
		$get						=	array();
		if ( $sale_id ) {
			$get[]					=	'sale_id=' . urlencode( $sale_id );
		}
		if ( $invoice_id ) {
			$get[]					=	'invoice_id=' . urlencode( $invoice_id );
		}
		$get						=	implode( '&', $get );
		return $this->_api_xml_fetch( $formVars, 'get/1.0', 'sales/detail_sale' . ( $get ? '?' . $get : '' ), 'sale', $raw_data );
	}

	/**
	 * 2co API interface for: sales/stop_lineitem_recurring
	 *
	 * @param $lineitem_id
	 * @param  string                   $raw_data   IN+OUT: Raw data for notification logging
	 * @return SimpleXMLElement|string              OK: SimpleXMLElement, KO: string with error
	 */
	public function _api_stop_lineitem_recurring( $lineitem_id, &$raw_data )
	{
		$get						=	'lineitem_id=' . urlencode( $lineitem_id );
		return $this->_api_xml_fetch( array(), 'post/1.0', 'sales/stop_lineitem_recurring' . '?' . $get, 'response_code', $raw_data );
	}

	/**
	 * 2co API interface for: sales/refund_invoice
	 *
	 * @param  string                   $invoice_id  Invoice id
	 * @param  string                   $category    Category id
	 * @param  float|int                $amount      Amount to refund the invoice by
	 * @param  string                   $comment     Comment for the refund passed to PSP
	 * @param  string                   $raw_data    IN+OUT: Raw data for notification logging
	 * @return SimpleXMLElement|string               OK: SimpleXMLElement, KO: string with error
	 */
	public function _api_refund_invoice( $invoice_id, $category, $amount, $comment, &$raw_data )
	{
		$formVars					=	array();
		$get						=	array();
		$get[]						=	'invoice_id=' . urlencode( $invoice_id );
		$get[]						=	'category=' . urlencode( $category );
		if ( $amount ) {
			$get[]					=	'currency=vendor';
			$get[]					=	'amount=' . urlencode( $amount );
		}
		if ( $comment ) {
			$get[]					=	'comment=' . urlencode( $comment );
		}
		$get						=	implode( '&', $get );
		return $this->_api_xml_fetch( $formVars, 'post/1.0', 'sales/refund_invoice' . ( $get ? '?' . $get : '' ), 'response_code', $raw_data );
	}

	/**
	 * 2co API interface for: sales/refund_lineitem
	 *
	 * @param  string                   $lineitem_id  Line item id
	 * @param  string                   $category     Category id
	 * @param  string                   $comment      Comment for the refund passed to PSP
	 * @param  string                   $raw_data     IN+OUT: Raw data for notification logging
	 * @return SimpleXMLElement|string                OK: SimpleXMLElement, KO: string with error
	 */
	public function _api_refund_lineitem( $lineitem_id, $category, $comment, &$raw_data )
	{
		$get						=	array();
		$get[]						=	'lineitem_id=' . urlencode( $lineitem_id );
		$get[]						=	'category=' . urlencode( $category );
		if ( $comment ) {
			$get[]					=	'comment=' . urlencode( $comment );
		}
		$get						=	implode( '&', $get );
		return $this->_api_xml_fetch( array(), 'post/1.0', 'sales/refund_lineitem' . '?' . $get, 'response_code', $raw_data );
	}
}
