<?php
/**
* @version $Id: cbpaidsubscriptions.telemoneyeasypay.php 1581 2012-12-24 02:36:44Z beat $
* @package CBSubs (TM) Community Builder Plugin for Paid Subscriptions (TM)
* @subpackage Plugin for Paid Subscriptions
* @copyright (C) 2007-2022 and Trademark of Lightning MultiCom SA, Switzerland - www.joomlapolis.com - and its licensors, all rights reserved
* @license http://www.gnu.org/licenses/old-licenses/gpl-2.0.html GNU/GPL version 2
*/

use CBLib\Language\CBTxt;

/** Ensure this file is being included by a parent file */
if ( ! ( defined( '_VALID_CB' ) || defined( '_JEXEC' ) || defined( '_VALID_MOS' ) ) ) { die( 'Direct Access to this location is not allowed.' ); }

global $_CB_framework;

// Avoids errors in CB plugin edit:
/** @noinspection PhpIncludeInspection */
include_once( $_CB_framework->getCfg( 'absolute_path' ) . '/components/com_comprofiler/plugin/user/plug_cbpaidsubscriptions/cbpaidsubscriptions.class.php' );

// This gateway implements a payment handler using a hosted page at the PSP:
// Import class cbpaidHostedPagePayHandler that extends cbpaidPayHandler
// and implements all gateway-generic CBSubs methods.

/**
 * To debug shared secret: Keep 0 for security reasons!
 * @var int
 */
define( 'TELEMONEY_DEBUG_SHARED', 0 );

/**
 * Payment handler class for this gateway: Handles all payment events and notifications, called by the parent class:
 * Please note that except the constructor and the API version this class does not implement any public methods.
 */
class cbpaidtelemoneyeasypayoem extends cbpaidHostedPagePayHandler
{
	/**
	 * Gateway API version used
	 * @var int
	 */
	public $gatewayApiVersion	=	"1.3.0";

	/**
	 * Constructor
	 *
	 * @param cbpaidGatewayAccount $account
	 */
	public function __construct( $account )
	{
		parent::__construct( $account );

		// Set gateway URLS for $this->pspUrl() results: first 2 are the main hosted payment page posting URL, next ones are gateway-specific:
		$this->_gatewayUrls	=	array(	'psp+normal'	 => $this->getAccountParam( 'psp_normal_url' ),
										'psp+test'		 => $this->getAccountParam( 'psp_test_url' ) );
	}

	/**
	 * CBSUBS HOSTED PAGE PAYMENT API METHODS:
	 */

	/**
	 * Returns single payment request parameters for gateway depending on basket (without specifying payment type)
	 *
	 * @param  cbpaidPaymentBasket  $paymentBasket   paymentBasket object
	 * @return array                                 Returns array $requestParams
	 */
	protected function getSinglePaymentRequstParams( $paymentBasket )
	{
		// build hidden form fields or redirect to gateway url parameters array:
		$requestParams	=	$this->_getBasicRequstParams( $paymentBasket );

		// sign single payment params:
		$this->_signRequestParams( $requestParams );

		return $requestParams;
	}

	/**
	 * Optional function: only needed for recurring payments:
	 * Returns subscription request parameters for gateway depending on basket (without specifying payment type)
	 *
	 * @param  cbpaidPaymentBasket  $paymentBasket   paymentBasket object
	 * @return array                                 Returns array $requestParams
	 */
	protected function getSubscriptionRequstParams( $paymentBasket )
	{
		return $this->getSinglePaymentRequstParams( $paymentBasket );
	}

	/**
	 * The user got redirected back from the payment service provider with a success message: Let's see how successfull it was
	 *
	 * @param  cbpaidPaymentBasket  $paymentBasket  New empty object. returning: includes the id of the payment basket of this callback (strictly verified, otherwise untouched)
	 * @param  array                $postdata       _POST data for saving edited tab content as generated with getEditTab
	 * @return string                               HTML to display if frontend, FALSE if XML error (and not yet ErrorMSG generated), or NULL if nothing to display
	 */
	protected function handleReturn( $paymentBasket, $postdata )
	{
		if ( ( count( $postdata ) > 0 ) && isset( $postdata['TM_RefNo'] ) ) {
			// we prefer POST for sensitive data:
			$requestdata				=	$postdata;
		} else {
			// but if gateway needs GET, we will work with it too:
			$requestdata				=	$this->_getGetParams();
		}

		$paymentBasketId				=	(int) cbGetParam( $requestdata, 'TM_RefNo', null );
		if ( ! $paymentBasketId ) {
			$paymentBasketId			=	(int) cbGetParam( $_GET, 'cbpbasket', null );
		}

		// EasyPay doesn't return any info as it's all through IPN before PDT, so display a good hint to user that IPN is not reaching:
		if ( $paymentBasketId ) {
			$exists						=	$paymentBasket->load( (int) $paymentBasketId );
		} else {
			$exists						=	false;
		}

		if ( ( ! $paymentBasketId ) || ( ! $exists ) ) {
			$errorMsg					=	( $paymentBasketId ? CBTxt::T("Sorry, payment gateway protocol error.") : CBTxt::T("Sorry, your order can not be found.") );
			$this->_setLogErrorMSG( 3, $paymentBasket, $this->getPayName() . ': No TM_RefNo received or payment basket deleted.', $errorMsg . ' ' . CBTxt::T( 'Please contact site administrator to check error log.' ) );
			return false;
		}
		if ( $paymentBasket->payment_status == 'NotInitiated' ) {
			// Ok, we either have a Denied payment, which did not affect the basket, but stored an IPN notification for it:

			// Let's find if a latest IPN exists for this basket for this gateway:
			$ipn						=	new cbpaidPaymentNotification();
			$whereArray					=	array(	'payment_basket_id'	=> (int) $paymentBasket->id,
													'gateway_account'	=> (int) $this->getAccountParam( 'id' ),
													'log_type'			=> 'I',
													'raw_result'		=> 'SUCCESS' );
			$ipnEntryExists				=	$ipn->loadThisMatching( $whereArray, array( 'id' => 'DESC' ) );

			if ( $ipnEntryExists ) {
				if ( $ipn->payment_status == 'Denied' ) {

					$newMsg				=	CBTxt::Th("The payment has been denied and therefore not executed.")
										.	' '
										.	CBTxt::Th( htmlspecialchars( $ipn->reason_code ) )
										.	'. '
										.	CBTxt::Th("Please choose another payment method.");
				} else {

					$newMsg				=	CBTxt::Th("Your transaction is not cleared and has currently following status:")
										.	' <strong>' . htmlspecialchars( $ipn->translatedPaymentStatus() ) . '</strong>. '
										.	CBTxt::Th( htmlspecialchars( $ipn->reason_code ) );
				}
				$this->_setErrorMSG( $newMsg );

			} else {

				// Or we simply did not receive a notification at all:
				$this->_setLogErrorMSG( 3, $paymentBasket, $this->getPayName() . ': No Notification received from PSP. Your server must be reachable from public network, or set notification host correspondingly.', CBTxt::T("Sorry no response for your payment from payment server yet. Please refresh page in a minute or check your email and status later.") . ' ' . CBTxt::T( 'Please contact site administrator to check error log.' ) );
			}
			return false;
		}
		return null;
	}

	/**
	 * The user cancelled his payment
	 *
	 * @param  cbpaidPaymentBasket  $paymentBasket  New empty object. returning: includes the id of the payment basket of this callback (strictly verified, otherwise untouched)
	 * @param  array                $postdata       _POST data for saving edited tab content as generated with getEditTab
	 * @return string                               HTML to display, FALSE if registration cancelled and ErrorMSG generated, or NULL if nothing to display
	 */
	protected function handleCancel( $paymentBasket, $postdata )
	{
		// The user cancelled his payment (and registration):
		if ( $this->hashPdtBackCheck( $this->_getReqParam( 'pdtback' ) ) ) {
			$paymentBasketId					=	(int) $this->_getReqParam( 'basket' );

			// check if cancel was from gateway:
			if ( ! $paymentBasketId ) {
				$paymentBasketId				=	(int) cbGetParam( $postdata, 'TM_RefNo', null );
			}

			$exists								=	$paymentBasket->load( (int) $paymentBasketId );

			if ( $exists && ( $this->_getReqParam( 'id' ) == $paymentBasket->shared_secret ) && ( $paymentBasket->payment_status != 'Completed' ) ) {
				$paymentBasket->payment_status	=	'RedisplayOriginalBasket';

				$this->_setErrorMSG( CBTxt::T( 'Payment cancelled.' ) );
			}
		}
		return false;
	}

	/**
	 * The payment service provider server did a server-to-server notification: Verify and handle it here:
	 *
	 * @param  cbpaidPaymentBasket  $paymentBasket  New empty object. returning: includes the id of the payment basket of this callback (strictly verified, otherwise untouched)
	 * @param  array                $postdata       _POST data for saving edited tab content as generated with getEditTab
	 * @return string                              Text to return to gateway if notification, or NULL if nothing to display
	 */
	protected function handleNotification( $paymentBasket, $postdata )
	{
		if ( $this->_checkOriginIp() ) {
			if ( ( count( $postdata ) > 0 ) && isset( $postdata['TM_RefNo'] ) ) {
				// we prefer POST for sensitive data:
				$requestdata	=	$postdata;
			} else {
				// but if gateway needs GET, we will work with it too:
				$requestdata	=	$this->_getGetParams();
			}
			return $this->_returnParamsHandler( $paymentBasket, $requestdata, 'I' );
		} else {


			// Returns error 500 to TeleMoney, so they know we could not handle this notification:
			header('HTTP/1.0 500 Internal Server Error');
			echo CBTxt::T( 'Please contact site administrator to check payment status and error log.' );
			exit();
		}
	}

	/**
	 * OVERRIDE OF CBSUBS API:
	 * Needed for the additional shared secret security parameter in the notification URL.
	 */

	/**
	 * Returns the notification URL for the callbacks/IPNs (not htmlspecialchared)
	 * Uses getAccountParam( 'notifications_host' ) if it is defined
	 *
	 * @param  cbpaidPaymentBasket  $paymentBasket
	 * @param  boolean              $noAccount      TRUE: do not include account number, but account method, FALSE (default): include account number
	 * @return string               URL
	 */
	protected function getNotifyUrl( $paymentBasket, $noAccount = false )
	{
		$url					=	parent::getNotifyUrl( $paymentBasket, $noAccount );

		if ( ! $this->getAccountParam( 'send_status_url_to_gateway', '1' ) ) {
			$tmep_shared_secret	=	$this->getAccountParam( 'tmep_shared_secret' );
			if ( $tmep_shared_secret ) {
				$url			.=	'&tmepshkey=' . urlencode( $tmep_shared_secret );
			}
		}
		return $url;
	}

	/**
	 * GATEWAY-INTERNAL SPECIFIC PRIVATE METHODS:
	 */

	/**
	 * Check IP address for allowed ones in notifications:
	 * Basic security as there is no signature here: check if IP address comes from TeleMoney.
	 *
	 * @return boolean
	 */
	private function _checkOriginIp( )
	{
		$allowedIps			=	$this->getAccountParam( 'allowed_notification_ips' );
		if ( $allowedIps == '' ) {
			return true;
		}
		$allowedIps			=	explode( ',', $allowedIps );
		$ips				=	cbpaidRequest::getIParray();
		foreach ( $ips as $ip ) {
			if ( in_array( $ip, $allowedIps ) ) {
				return true;
			}
		}

		$errorMsg			=	sprintf( 'Notification: The originating IP address (%s) does not correspond to the configured allowed IP addresses (%s).', implode( ',', $ips ), implode( ',', $allowedIps ) );
		// Log notification with failed IP:
		$ipn				=	$this->_prepareIpn( 'I', null, null, null, null, 'utf-8' );
		$rawLogData			=	/* cbGetParam() not needed: we want raw info */ '$_GET=' . var_export( $_GET, true ) . ";\n"
							.	/* cbGetParam() not needed: we want raw info */ '$_POST=' . var_export( $_POST, true ) . ";\n"
							.	'$errormsg="' . addslashes( $errorMsg ) . ";\n";
		$ipn->raw_data		=	'$message_type="NOTIFICATION";' . "\n" . $rawLogData;
		$this->_storeIpnResult( $ipn, 'ERROR: ' . 'GATEWAY IP ADDRESS MISMATCH' );

		// Log error:
		$this->_setLogErrorMSG( 4, $ipn, $this->getPayName() . ': ' . $errorMsg , CBTxt::T( 'Sorry, the payment server notification origin could not be verified.' ) . ' ' . CBTxt::T( 'Please contact site administrator to check payment status and error log.' ) );
		return false;
	}

	/**
	 * sign payment request $requestParams with validation code added to $requestParams array
	 *
	 * @param  array     $requestParams        The keyed parameters array to add generate validation to $requestParams['signature']
	 */
	private function _signRequestParams( &$requestParams )
	{
		// List of parameters to concatenate if not empty:
		$listOfParams					=	array(	'$pspid',	// merchant id (pspid)
													'REF',		// item code (basket id)
													'CUR',		// purchase currency (USD, etc..)
													'AMT'		// purchase price (10.01, etc..)
												);

		// Concatenate them using this payments concatenation function with $caseInsensitiveKeys = true:
		$string							=	$this->_concatVars( $requestParams, $listOfParams, null, '', '', false, false, true, false, false );

		// add validation code doing md5 of the string without uppercasing:
		$requestParams['userfield5']	=	strtolower( md5( $string ) );
	}

	/**
	 * Validate a reply in $requestParams using md5check value, and then also through the API if it is available
	 *
	 * @param  array    $requestParams
	 * @return boolean                  TRUE: valid, FALSE: invalid
	 */
	private function _pspVerifySignature( $requestParams )
	{
		// List of parameters to concatenate if not empty:
		$listOfParams	=	array(	'TM_MCODE',		// merchant id (pspid)
									'TM_REFNO',		// item code (basket id)
									'TM_CURRENCY',	// purchase currency (USD, etc..)
									'TM_DEBITAMT'	// purchase price (10.01, etc..)
								);

		// Concatenate them using this payments concatenation function with $caseInsensitiveKeys = true:
		$string			=	$this->_concatVars( $requestParams, $listOfParams, null, '', '', false, false, true, false, false );

		return ( cbGetParam( $requestParams, 'TM_UserField5' ) == strtolower( md5( $string ) ) );
	}

	/**
	 * Compute the CBSubs payment_status based on gateway's reply in $postdata:
	 *
	 * @param  array   $postdata  raw POST data received from the payment gateway
	 * @param  string  $reason    OUTPUT: reason_code
	 * @return string
	 */
	private function _paymentStatus( $postdata, &$reason )
	{
		if ( ! cbGetParam( $postdata, 'TM_Error' ) ) {
			$reason	=	null;
			$status	=	'Completed';
		} else {
			$reason	=	stripslashes( cbGetParam( $postdata, 'TM_ErrorMsg' ) );
			$status	=	'Denied';
		}

		return $status;
	}

	/**
	 * Compute the CBSubs payment_type based on gateway's reply $postdata:
	 *
	 * @param  array   $postdata  raw POST data received from the payment gateway
	 * @return string             Human-readable string
	 */
	private function _getPaymentType( $postdata )
	{
		switch ( (int) cbGetParam( $postdata, 'TM_PaymentType' ) ) {
			case '2':
				$type	=	'Master Credit Card';
				break;
			case '3':
				$type	=	'VISA Credit Card';
				break;
			case '5':
				$type	=	'Amex Credit Card';
				break;
			case '22':
				$type	=	'Diners Credit Card';
				break;
			case '23':
				$type	=	'JCB Credit Card';
				break;
			case '25':
				$type	=	'China UnionPay Card';
				break;
			case '41':
				$type	=	'ENets';
				break;
			default:
				$type	=	'Credit Card';
				break;
		}

		return $type;
	}

	/**
	 * Popoulates basic request parameters for gateway depending on basket (without specifying payment type)
	 *
	 * @param  cbpaidPaymentBasket  $paymentBasket   paymentBasket object
	 * @return array                                 Returns array $requestParams
	 */
	private function _getBasicRequstParams( $paymentBasket )
	{
		// mandatory parameters:
		$requestParams						=	array();
		$requestParams['mid']				=	$this->getAccountParam( 'pspid' );
		$requestParams['ref']				=	$paymentBasket->id;
		$requestParams['cur']				=	$paymentBasket->mc_currency;
		$requestParams['amt']				=	sprintf( '%.2f', $paymentBasket->mc_gross );

		// urls for return and cancel:
		$requestParams['returnurl']			=	$this->getSuccessUrl( $paymentBasket );

		if ( TELEMONEY_DEBUG_SHARED || $this->getAccountParam( 'send_status_url_to_gateway', '1' ) ) {
			$requestParams['statusurl']		=	$this->getNotifyUrl( $paymentBasket );
		}

		// payment page status:
		$requestParams['skipstatuspage']	=	'Y';

		$requestParams['userfield4']		=	$paymentBasket->shared_secret;
		return $requestParams;
	}

	/**
	 * The user got redirected back from the payment service provider with a success message: let's see how successfull it was
	 *
	 * @param  cbpaidPaymentBasket  $paymentBasket      New empty object. returning: includes the id of the payment basket of this callback (strictly verified, otherwise untouched)
	 * @param  array                $requestdata        Data returned by gateway
	 * @param  string               $type               Type of return ('R' for PDT, 'I' for INS, 'A' for Autorecurring payment (Vault) )
	 * @param  array                $additionalLogData  Additional data to log with notification
	 * @return string                                   HTML to display if frontend, text to return to gateway if notification, FALSE if registration cancelled and ErrorMSG generated, or NULL if nothing to display
	 */
	private function _returnParamsHandler( $paymentBasket, $requestdata, $type, $additionalLogData = null )
	{
		global $_CB_framework, $_GET, $_POST;

		$ret													=	null;
		$paymentBasketId										=	(int) cbGetParam( $requestdata, 'TM_RefNo', null );

		if ( $paymentBasketId ) {
			$exists												=	$paymentBasket->load( (int) $paymentBasketId );

			$rawLogData											=	'';
			if ( $additionalLogData ) {
				foreach ( $additionalLogData as $k => $v ) {
					$rawLogData									.=	'$' . $k . '="' . var_export( $v, true ) . '";' . "\n";
				}
			}
			$rawLogData											.=	/* cbGetParam() not needed: we want raw info */ '$requestdata=' . var_export( $requestdata, true ) . ";\n"
																.	/* cbGetParam() not needed: we want raw info */ '$_GET=' . var_export( $_GET, true ) . ";\n"
																.	/* cbGetParam() not needed: we want raw info */ '$_POST=' . var_export( $_POST, true ) . ";\n";

			if ( $this->getAccountParam( 'send_status_url_to_gateway', '1' ) ) {
		  		// Normal case: statusurl is part of the form:
				$shared_secret									=	cbGetParam( $_GET, $this->_getPagingParamName( 'id' ) );
		  	} else {
		  		// Secured case: status url in not part of the form, and given offline to the PSP including the shared basket secret in parameter TM_UserField4 and the permanent gateway's shared secret in tmep_shared_secret:
				$shared_secret									=	cbGetParam( $requestdata, 'TM_UserField4' );

				// Now also check the server-server shared secret too:
				$tmep_shared_secret								=	$this->getAccountParam( 'tmep_shared_secret' );
				if ( $tmep_shared_secret ) {
					// Must be in URL:
					if ( cbGetParam( $_GET, 'tmepshkey' ) !== $tmep_shared_secret ) {

						// There is a mismatch in the shared secret:

						// Log notification:
						$ipn									=	$this->_prepareIpn( $type, null, null, null, null, 'utf-8' );
						$ipn->raw_data							=	'$message_type="NOTIFICATION";' . "\n" . $rawLogData;
						$this->_storeIpnResult( $ipn, 'ERROR: ' . 'SHARED GATEWAY SECRET MISMATCH' );

						// Log error:
						$this->_setLogErrorMSG( 4, $ipn, $this->getPayName() . ': ' . sprintf( 'Notification: The received gateway shared secret (%s) does not correspond to the configured secret (%s).', cbGetParam( $_GET, 'tmepshkey' ), $tmep_shared_secret ) , CBTxt::T( 'Sorry, the payment server notification origin could not be verified.' ) . ' ' . CBTxt::T( 'Please contact site administrator to check payment status and error log.' ) );

						// Makes the notification to be ignored:
						$exists									=	false;
					}
				}

			}

			if ( $exists ) {
				if ( $shared_secret !== $paymentBasket->shared_secret ) {
					// There is a mismatch in the shared secret:

					// Log notification:
					$ipn									=	$this->_prepareIpn( $type, null, null, null, null, 'utf-8' );
					$ipn->raw_data							=	'$message_type="NOTIFICATION";' . "\n" . $rawLogData;
					$this->_storeIpnResult( $ipn, 'ERROR: ' . 'SHARED BASKET SECRET MISMATCH' );

					// Log error:
					$this->_setLogErrorMSG( 4, $ipn, $this->getPayName() . ': ' . sprintf( 'Notification: The received basket shared secret (%s) does not correspond to the configured secret (%s).', $shared_secret, $paymentBasket->shared_secret ) , CBTxt::T( 'Sorry, the payment server notification integrity could not be verified.' ) . ' ' . CBTxt::T( 'Please contact site administrator to check payment status and error log.' ) );

					// Makes the notification to be ignored:
					$exists									=	false;
				}
			}

			if ( $exists
			&& ( ! ( ( ( $type == 'R' ) || ( $type == 'I' ) ) && ( $paymentBasket->payment_status == 'Completed' ) ) )
			&& ( $shared_secret == $paymentBasket->shared_secret ) ) {

			  	// Log the return record:
				$log_type										=	$type;
				$reason											=	null;
				$paymentStatus									=	$this->_paymentStatus( $requestdata, $reason );
				$paymentType									=	$this->_getPaymentType( $requestdata );
				$paymentTime									=	$_CB_framework->now();

				if ( $paymentStatus == 'Error' ) {
					$errorTypes									=	array( 'I' => 'D', 'R' => 'E' );
					if ( isset( $errorTypes[$type] ) ) {
						$log_type								=	$errorTypes[$type];
					}
				}

				$ipn											=	$this->_prepareIpn( $log_type, $paymentStatus, $paymentType, $reason, $paymentTime, 'utf-8' );

				if ( $paymentStatus == 'Refunded' ) {
					// in case of refund we need to log the payment as it has same TnxId as first payment: so we need payment_date for discrimination:
					$ipn->payment_date							=	gmdate( 'H:i:s M d, Y T', $paymentTime ); // paypal-style
				}

				$ipn->test_ipn									=	( $this->getAccountParam( 'normal_gateway' ) == '0' ? 1 : 0 );		// EasyPay has no param telling if it is for test or real.
				$ipn->raw_data									=	'$message_type="' . ( $type == 'R' ? 'RETURN_TO_SITE' : ( $type == 'I' ? 'NOTIFICATION' : 'UNKNOWN' ) ) . '";' . "\n"
																.	$rawLogData;

				if ( $paymentStatus == 'Error' ) {
					$paymentBasket->reason_code					=	$reason;

					$this->_storeIpnResult( $ipn, 'ERROR:' . $reason );
					$this->_setLogErrorMSG( 4, $ipn, $this->getPayName() . ': ' . $reason, CBTxt::T( 'Sorry, the payment server replied with an error.' ) . ' ' . CBTxt::T( 'Please contact site administrator to check payment status and error log.' ) );

					$ret										=	false;
				} else {
					$ipn->bindBasket( $paymentBasket );

					$ipn->sale_id								=	$paymentBasketId;

					$insToIpn									=	array(	'txn_id' => 'TM_RefNo',
																			'mc_currency' => 'TM_Currency'
																		);

					foreach ( $insToIpn as $k => $v ) {
						$ipn->$k								=	cbGetParam( $requestdata, $v );
					}

					$ipn->mc_gross								=	sprintf( '%.2f', cbGetParam( $requestdata, 'TM_DebitAmt' ) );
					$ipn->user_id								=	(int) $paymentBasket->user_id;
					$ipn->txn_type								=	'web_accept';

					// validate payment from PDT or IPN
					if ( $this->_pspVerifySignature( $requestdata ) ) {
						if ( ( $paymentBasketId == cbGetParam( $requestdata, 'TM_RefNo', null ) ) && ( ( sprintf( '%.2f', $paymentBasket->mc_gross ) == $ipn->mc_gross ) || ( $ipn->payment_status == 'Refunded' ) ) && ( $paymentBasket->mc_currency == $ipn->mc_currency ) ) {
							if ( in_array( $ipn->payment_status, array( 'Completed', 'Processed', 'Pending', 'Refunded', 'Denied' ) ) ) {
								$this->_storeIpnResult( $ipn, 'SUCCESS' );

								if ( $ipn->payment_status != 'Denied' ) {
									$this->_bindIpnToBasket( $ipn, $paymentBasket );

									// add the gateway to the basket:
									$paymentBasket->payment_method	=	$this->getPayName();
									$paymentBasket->gateway_account	=	$this->getAccountParam( 'id' );

									$this->updatePaymentStatus( $paymentBasket, $ipn->txn_type, $ipn->payment_status, $ipn, 1, 0, 0, false );

									if ( in_array( $ipn->payment_status, array( 'Completed', 'Processed', 'Pending' ) ) ) {
										$ret						=	true;
									}
								}
							} else {
								$this->_storeIpnResult( $ipn, 'FAILED' );

								$paymentBasket->payment_status	=	$ipn->payment_status;

								$this->_setErrorMSG( '<div class="alert alert-info">' . $this->getTxtNextStep( $paymentBasket ) . '</div>' );

								$paymentBasket->payment_status	=	'RedisplayOriginalBasket';
								$ret							=	false;
							}
						} else {
							$this->_storeIpnResult( $ipn, 'MISMATCH' );
							$this->_setLogErrorMSG( 3, $ipn, $this->getPayName() . ': amount or currency missmatch', CBTxt::T( 'Sorry, the payment does not match the basket.' ) . ' ' . CBTxt::T( 'Please contact site administrator to check error log.' ) );

							$ret								=	false;
						}
					} else {
						$this->_storeIpnResult( $ipn, 'SIGNERROR' );
						$this->_setLogErrorMSG( 3, $ipn, $this->getPayName() . ': md5 or transaction does not match with gateway.', CBTxt::T( 'The transaction is invalid.' ) . ' ' . CBTxt::T( 'Please contact site administrator to check error log.' ) );

						$ret									=	false;
					}
				}
			}
		} else {
			$this->_setLogErrorMSG( 3, null, $this->getPayName() . ': TM_RefNo is missing in the return URL: ' . var_export( $_GET, true ), CBTxt::T( 'Please contact site administrator to check error log.' ) );
		}

		return  $ret;
	}
}

/**
 * Payment account class for this gateway: Stores the settings for that gateway instance, and is used when editing and storing gateway parameters in the backend.
 *
 * OEM base
 * No methods need to be implemented or overriden in this class, except to implement the private-type params used specifically for this gateway:
 */
class cbpaidGatewayAccounttelemoneyeasypayoem extends cbpaidGatewayAccounthostedpage
{
}

/**
 * Payment handler class for this gateway: Handles all payment events and notifications, called by the parent class:
 *
 * Gateway-specific
 * Please note that except the constructor and the API version this class does not implement any public methods.
 */
class cbpaidtelemoneyeasypay extends cbpaidtelemoneyeasypayoem
{
}

/**
 * Payment account class for this gateway: Stores the settings for that gateway instance, and is used when editing and storing gateway parameters in the backend.
 *
 * Gateway-specific
 * No methods need to be implemented or overriden in this class, except to implement the private-type params used specifically for this gateway:
 */
class cbpaidGatewayAccounttelemoneyeasypay extends cbpaidGatewayAccounttelemoneyeasypayoem
{
}
