<?php
/**
* @version $Id: cbpaidsubscriptions.paypalpro.php 1581 2012-12-24 02:36:44Z beat $
* @package CBSubs (TM) Community Builder Plugin for Paid Subscriptions (TM)
* @subpackage Plugin for Paid Subscriptions
* @copyright (C) 2007-2022 and Trademark of Lightning MultiCom SA, Switzerland - www.joomlapolis.com - and its licensors, all rights reserved
* @license http://www.gnu.org/licenses/old-licenses/gpl-2.0.html GNU/GPL version 2
*/

use CB\Database\Table\UserTable;
use CBLib\Application\Application;
use CBLib\Registry\GetterInterface;
use CBLib\Registry\ParamsInterface;
use CBLib\Language\CBTxt;

/** Ensure this file is being included by a parent file */
if ( ! ( defined( '_VALID_CB' ) || defined( '_JEXEC' ) || defined( '_VALID_MOS' ) ) ) { die( 'Direct Access to this location is not allowed.' ); }

global $_CB_framework;

// Avoids errors in CB plugin edit:
/** @noinspection PhpIncludeInspection */
include_once( $_CB_framework->getCfg( 'absolute_path' ) . '/components/com_comprofiler/plugin/user/plug_cbpaidsubscriptions/cbpaidsubscriptions.class.php' );

// This gateway implements a payment handler using a on-site credit-card page:
// Import class cbpaidCreditCardsPayHandler that extends cbpaidPayHandler
// and implements all gateway-generic CBSubs methods.

/**
 * Payment handler class for this gateway: Handles all payment events and notifications, called by the parent class:
 *
 * OEM base
 * Please note that except the constructor and the API version this class does not implement any public methods.
 */
class cbpaidpaypalprooem extends cbpaidCreditCardsPayHandler
{
	/**
	 * Overrides base class with 1:
	 * Hash type: 1 = only if there is a basket id (default), 2 = always, 0 = never
	 * @var int
	 */
	protected $_urlHashType		=	1;

	/**
	 * Gateway API version used
	 * @var int
	 */
	public $gatewayApiVersion	=	"1.3.0";

	/**
	 * Constructor
	 *
	 * @param cbpaidGatewayAccount $account
	 */
	public function __construct( $account )
	{
		parent::__construct( $account );

		// Set gateway URLS for $this->pspUrl() results: first 2 are the main hosted payment page posting URL, next ones are gateway-specific:
		$this->_gatewayUrls	=	array(	'psp+normal'			=>	$this->getAccountParam( 'psp_normal_url' ),
										'psp+test'				=>	$this->getAccountParam( 'psp_test_url' ),
										'payflow+normal'		=>	$this->getAccountParam( 'psp_payflow_url' ),
										'payflow+test'			=>	$this->getAccountParam( 'psp_payflow_test_url' ),
										'payflow_link+normal'	=>	$this->getAccountParam( 'psp_payflow_link_url' ),
										'payflow_link+test'		=>	$this->getAccountParam( 'psp_payflow_link_test_url' )
									);
	}

	/**
	 * CBSUBS PAY HANLDER API METHODS:
	 */

	/**
	 * Draws the credit-card form
	 *
	 * @param  UserTable            $user                 User
	 * @param  cbpaidPaymentBasket  $paymentBasket        paymentBasket object
	 * @param  string               $cardType             CC-brand if no choice
	 * @param  string               $postUrl              URL for the <form>
	 * @param  string               $payButtonText        Text for payment text (if basket allows single-payments)
	 * @param  string               $subscribeButtonText  Text for subscribe button (if basket allows auto-recurring subscriptions)
	 * @param  string|null          $chosenCard
	 * @return string
	 */
	protected function _drawCCform( $user, $paymentBasket, $cardType, $postUrl, $payButtonText, $subscribeButtonText, $chosenCard = null )
	{
		global $_CB_framework;

		// We need to completely override the form inputs and behavior to POST to PayPal instead of CBSubs:
		if ( ( $this->getAccountParam( 'paypal_api' ) == 2 ) && $this->getAccountParam( 'paypal_payflow_redirect' ) ) {
			// Display any messages returned by PayPal if the payment failed for whatever reason:
			$response							=	Application::Input()->get( 'RESPMSG', null, GetterInterface::STRING );

			if ( $response ) {
				$this->_setErrorMSG( $response );
			}

			// Error responses will send us back to the form, but with cardtype prefixed, which payform usage doesn't look for so lets set it:
			if ( ! $cardType ) {
				$cardType						=	$this->_getReqParam( 'cardtype' );
			}

			$cbpaidUser							=	cbpaidUserExtension::getInstance( $user->id );
			$params								=	cbpaidApp::settingsParams();
			$sealCode							=	$params->get( 'security_logos_and_seals' );		// keep $param, it's a global setting !
			$drawCCV							=	$params->get( 'show_cc_ccv', 1 );				// keep $param, it's a global setting !
			$drawAVS							=	$this->getAccountParam( 'show_cc_avs', 0 );		// keep $param, it's a global setting !

			// We need to generate a payment token and add it to the form so PayPal knows it's a transparent redirect otherwise error:
			$txtHiddenInputs					=	$this->getPayPalTokenInputs( $paymentBasket, $cardType, $subscribeButtonText );

			if ( $payButtonText && $subscribeButtonText ) {
				// User has choice of single or recurring so lets be sure we output both tokens as we can determine which to submit:
				$txtHiddenInputs				.=	$this->getPayPalTokenInputs( $paymentBasket, $cardType );
			}

			if ( ! $txtHiddenInputs ) {
				// If we've no tokens to work with just let it output the errors and suppress displaying the form:
				return null;
			}

			$txtVisibleInputs					=	array(	'number'	=>	'<input class="w-100 form-control required creditcard" size="20" maxlength="20" type="text" autocomplete="off" name="CARDNUM" value="" />',
															'firstname'	=>	'<input class="w-100 form-control required" size="20" maxlength="50" type="text" autocomplete="off" name="BILLTOFIRSTNAME" value="' . htmlspecialchars( $paymentBasket->get( 'first_name', null, GetterInterface::STRING ) ) . '" />',
															'lastname'	=>	'<input class="w-100 form-control required" size="20" maxlength="50" type="text" autocomplete="off" name="BILLTOLASTNAME" value="' . htmlspecialchars( $paymentBasket->get( 'last_name', null, GetterInterface::STRING ) )  . '" />'
													);

			if ( $drawCCV ) {
				$txtVisibleInputs['cvv']		=	'<input class="form-control required" ' . cbValidator::getRuleHtmlAttributes( 'minlength', 3 ) . ' size="6" maxlength="4" type="text" autocomplete="off" name="CVV2" value="" />';
			}

			if ( ! $cardType ) {
				$txtVisibleInputs['cardtypes']	=	$this->_drawCCSelector( $user, $paymentBasket, $chosenCard );
				$txtVisibleInputs['cardtype']	=	null;
			} else {
				$txtVisibleInputs['cardtypes']	=	null;
				$txtVisibleInputs['cardtype']	=	$this->_renderCCimg( $cardType, false, false );
			}

			$months								=	array();
			$months[]							=	moscomprofilerHTML::makeOption( '', 'MM' );

			for ( $i=1; $i <= 12; $i++ ) {
				$months[]						=	moscomprofilerHTML::makeOption( sprintf( '%02d', $i ), sprintf( '%00d', $i ) );
			}

			$txtVisibleInputs['expmonth']		=	moscomprofilerHTML::selectList( $months, 'EXPMONTH', 'class="form-control required" size="1"', 'value', 'text', '' );

			$years								=	array();
			$years[]							=	moscomprofilerHTML::makeOption( '', 'YYYY' );

			$yearNow							=	(int) gmdate( 'Y' );
			$monthNow							=	(int) gmdate( 'm' );

			for ( $i = ( ( $monthNow == 1 ) ? -1 : 0 ) ; $i < $this->ccYearsInAdvance; $i++ ) {
				$years[]						=	moscomprofilerHTML::makeOption( $yearNow + $i, sprintf( '%0000d', $yearNow + $i ) );
			}

			$txtVisibleInputs['expyear']		=	moscomprofilerHTML::selectList( $years, 'EXPYEAR', 'class="form-control required" size="1"', 'value', 'text', '' );

			if ( $drawAVS ) {
				if ( $drawAVS >= 2 ) {
					/** @var $user cbpaidUserWithSubsFields */
					$txtVisibleInputs['address']	=	'<input class="w-100 form-control required" size="40" maxlength="60" type="text" autocomplete="off" name="BILLTOSTREET" value="' . htmlspecialchars( $user->cb_subs_inv_address_street ) . '" />';
				}

				$txtVisibleInputs['zip']			=	'<input class="form-control required" size="10" maxlength="20" type="text" autocomplete="off" name="BILLTOZIP" value="' . htmlspecialchars( $user->cb_subs_inv_address_zip ) . '" />';

				$allCountriesSelect					=	array();
				$countries							=	new cbpaidCountries();

				foreach ( $countries->twoLetterIsoCountries() as $countryName) {
					$allCountriesSelect[]			=	moscomprofilerHTML::makeOption( $countryName, CBTxt::T( $countryName ) );
				}

				$txtVisibleInputs['country']		=	moscomprofilerHTML::selectList( $allCountriesSelect, 'BILLTOCOUNTRY', 'class="form-control required" size="1"', 'value', 'text', $cbpaidUser->getInvoiceAddressField( 'cb_subs_inv_address_country' ) );
			}

			$txtButton							=	'<div class="cbregCCbutton">';

			if ( $payButtonText ) {
				$txtButton						.=	'<button type="submit" id="cbPayNow" value="pay" title="' .  htmlspecialchars( CBTxt::T("Pay now") ) . '" class="btn btn-primary btn-block text-wrap">'
												.		$payButtonText
												.	'</button>';

				$js								=	"$( '#cbPayNow' ).click( function( e ) {"
												.		"e.preventDefault();"
												.		"if ( $( '#cbsubsCCform' ).valid() ) {"
												.			"$( this ).parent().fadeOut( 'slow', function() { $( '#cbpayWheel' ).fadeIn( 'slow' ); } );"
												.			"$( '.paypalTokenRecurring' ).remove();"
												.			"$( this.form ).submit();"
												.		"}"
												.	"});";

				$_CB_framework->outputCbJQuery( $js );
			}

			if ( $payButtonText && $subscribeButtonText ) {
				$txtButton						.=	'<div class="mt-2 mb-2 row no-gutters align-items-center cbregCCbuttonDivider">'
												.		'<div class="col cbregCCbuttonDividerLine"><hr class="m-0" /></div>'
												.		'<div class="pl-2 pr-2 text-center text-muted text-small cbregCCbuttonDividerText">' . CBTxt::T( 'OR' ) . '</div>'
												.		'<div class="col cbregCCbuttonDividerLine"><hr class="m-0" /></div>'
												.	'</div>';
			}

			if ( $subscribeButtonText ) {
				$txtButton						.=	'<button type="submit" id="cbSubscribeNow" value="' . htmlspecialchars( $subscribeButtonText ) . '" title="' . htmlspecialchars( CBTxt::T("Subscribe to payments now") ) . '" class="btn btn-primary btn-block text-wrap cbPayNowWithCCbutton">'
												.		$subscribeButtonText
												.	'</button>';

				$js								=	"$( '#cbSubscribeNow' ).click( function( e ) {"
												.		"e.preventDefault();"
												.		"if ( $( '#cbsubsCCform' ).valid() ) {"
												.			"$( this ).parent().fadeOut( 'slow', function() { $( '#cbpayWheel' ).fadeIn( 'slow' ); } );"
												.			"$( '.paypalTokenSingle' ).remove();"
												.			"$( this.form ).submit();"
												.		"}"
												.	"});";

				$_CB_framework->outputCbJQuery( $js );
			}

			$txtButton							.=	'</div>'
												.	'<div id="cbpayWheel" style="display:none;"><img src="' . $this->baseClass->getPluginLIvePath() . '/icons/hot/wheel_pay.gif" alt="spinning wheel" /></div>';

			cbValidator::loadValidation();

			$return		=	'<form action="' . $this->gatewayUrl( 'payflow_link' ) . '" method="post" autocomplete="off" id="cbsubsCCform" name="cbsubsCCform" class="form-auto m-0 cb_form cbValidation">'
						.		$txtHiddenInputs;

			ob_start();
			$this->_renderCCform( $user, $paymentBasket, $cardType, $txtVisibleInputs, $txtButton );
			$return		.=		ob_get_contents();
			ob_end_clean();

			$return		.=	'</form>';

			ob_start();
			$this->_renderCCsealCode( $sealCode );
			$return		.=	ob_get_contents();
			ob_end_clean();

			return $return;
		}

		return parent::_drawCCform( $user, $paymentBasket, $cardType, $postUrl, $payButtonText, $subscribeButtonText, $chosenCard );
	}

	/**
	 * Checks that the credit-card number checkum is valid
	 *
	 * @param  array $card The credit-card ($card['number'] is used here)
	 * @return boolean     TRUE: Valid
	 */
	protected function checkCCNumber( $card )
	{
		if ( ( $this->getAccountParam( 'paypal_api' ) == 2 ) && $this->getAccountParam( 'paypal_payflow_redirect' ) ) {
			// We're using transparent redirect; PayPal has already confirmed this and we've no means of doing so our selves now:
			return true;
		}

		return parent::checkCCNumber( $card );
	}

	/**
	 * Checks the Credit-card expiry date to be in the future
	 *
	 * @param  array $card The credit-card ($card['expyear'] and $card['expmonth'] are used here)
	 * @param  int   $now  Unix time of now
	 * @return boolean     TRUE: ok
	 */
	protected function checkCCExpDate( $card, $now )
	{
		if ( ( $this->getAccountParam( 'paypal_api' ) == 2 ) && $this->getAccountParam( 'paypal_payflow_redirect' ) ) {
			// We're using transparent redirect; PayPal has already confirmed this and we've no means of doing so our selves now:
			return true;
		}

		return parent::checkCCExpDate( $card, $now );
	}

	/**
	 * Checks that the first and last names are filled-in and no longer than 50 characters each
	 *
	 * @param  array $card The credit-card ($card['firstname'] and $card['lastname'] are used here)
	 * @return boolean     TRUE: ok
	 */
	protected function checkCCName( $card )
	{
		if ( ( $this->getAccountParam( 'paypal_api' ) == 2 ) && $this->getAccountParam( 'paypal_payflow_redirect' ) ) {
			// We're using transparent redirect; PayPal has already confirmed this and we've no means of doing so our selves now:
			return true;
		}

		return parent::checkCCName( $card );
	}

	/**
	 * Attempts to validate a successful recurring payment
	 * (optional)
	 *
	 * @param  cbpaidPaymentBasket  $paymentBasket
	 * @param  string               $returnText                  RETURN param
	 * @param  boolean              $transientErrorDoReschedule  RETURN param
	 * @return boolean              TRUE: succes, FALSE: failed or unknown result
	 */
	public function processAutoRecurringPayment( $paymentBasket, &$returnText, &$transientErrorDoReschedule ) {
		global $_CB_framework;

		if ( $this->getAccountParam( 'paypal_api' ) == 2 ) {
			if ( $this->hasPaypalApi() ) {
				// https://developer.paypal.com/docs/classic/payflow/recurring-billing/
				$requestParams								=	array(	'TENDER'			=>	'C',
																		'TRXTYPE'			=>	'R',
																		'ACTION'			=>	'I',
																		'ORIGPROFILEID'		=>	$paymentBasket->subscr_id,
																		'PAYMENTHISTORY'	=>	'Y'
																	);

				$signedParams								=	$this->_signRequestParams( $requestParams );
				$formUrl									=	array();

				foreach ( $signedParams as $k => $v ) {
					$formUrl[$k]							=	$k . '[' . strlen( $v ) . ']=' . $v;
				}

				$formUrl									=	implode( '&', $formUrl );

				$results									=	array();
				$response									=	null;
				$status										=	null;
				$error										=	$this->_httpsRequest( $this->gatewayUrl( 'psp' ), $formUrl, 105, $response, $status, 'post', 'normal' );

				if ( $response ) {
					parse_str( $response, $results );
				}

				$ipn										=	$this->_prepareIpn( 'A', 'Completed', 'instant', null, $_CB_framework->now(), 'utf-8' );

				$ipn->bindBasket( $paymentBasket );

				$ipn->txn_type								=	'subscr_payment';
				$ipn->user_id								=	(int) $paymentBasket->user_id;

				$rawData									=	'$response="' . preg_replace( '/([^\s]{100})/', '$1 ', $response ) . "\"\n"
															.	'$results=' . var_export( $results, true ) . ";\n"
															.	'$requestParams=' . var_export( $requestParams, true ) . ";\n";

				$ipn->setRawData( $rawData );

				if ( $error || ( $status != 200 ) || ( ! $response ) ) {
					$ipn->setRawResult( 'FAILED' );

					$ipn->payment_status					=	'Failed';
					$ipn->txn_type							=	'subscr_failed';

					$transientErrorDoReschedule				=	true;
					$return									=	false;

					$this->_setLogErrorMSG( 3, null, $this->getPayName() . ' HTTPS POST request to payment gateway server failed.', CBTxt::T( "Submitted refund request didn't return an error but didn't complete." ) . ' ' . CBTxt::T( 'Please contact site administrator to check error log.' ) );
				} else {
					if ( cbGetParam( $results, 'RESULT' ) == '0' ) {
						$recurr								=	( $paymentBasket->recur_times_used + 1 );

						if ( in_array( cbGetParam( $results, 'P_TRANSTATE' . $recurr ), array( '6', '7', '8' ) ) ) {
							$ipn->setRawResult( 'SUCCESS' );

							$ipn->mc_gross					=	cbGetParam( $results, 'P_AMT' . $recurr );
							$ipn->txn_id					=	cbGetParam( $results, 'P_PNREF' . $recurr );

							$this->updatePaymentStatus( $paymentBasket, $ipn->txn_type, $ipn->payment_status, $ipn, 1, 2, ( ( $this->getAccountParam( 'enabled', 0 ) == 3 ) && ( $paymentBasket->isAnyAutoRecurring() == 2 ) ? 1 : 2 ), false );

							$transientErrorDoReschedule		=	false;
							$return							=	true;
						} else {
							$ipn->setRawResult( 'FAILED' );

							$ipn->payment_status			=	'Failed';
							$ipn->txn_type					=	'subscr_failed';

							switch ( cbGetParam( $results, 'P_TRANSTATE' . $recurr ) ) {
								case '11':
									$reason					=	CBTxt::T( 'Settlement Failed' );
									break;
								case '14':
									$reason					=	CBTxt::T( 'Settlement Incomplete' );
									break;
								default:
									$reason					=	CBTxt::T( 'Settlement Does Not Exist' );
									break;
							}

							$transientErrorDoReschedule		=	true;
							$return							=	false;

							$this->_setLogErrorMSG( 3, null, $this->getPayName() . ' Paypal recurring payment not complete. ERROR: ' . $reason . ' CODE: ' . cbGetParam( $results, 'P_TRANSTATE' . $recurr ), $reason . '. ' . CBTxt::T( 'Please contact site administrator to check error log.' ) );
						}
					} else {
						$ipn->setRawResult( 'FAILED' );

						$ipn->payment_status				=	'Failed';
						$ipn->txn_type						=	'subscr_failed';

						$transientErrorDoReschedule			=	true;
						$return								=	false;

						$this->_setLogErrorMSG( 3, null, $this->getPayName() . ' Paypal API error returned. ERROR: ' . cbGetParam( $results, 'RESPMSG' ) . ' CODE: ' . cbGetParam( $results, 'RESULT' ), cbGetParam( $results, 'RESPMSG' ) . '. ' . CBTxt::T( 'Please contact site administrator to check error log.' ) );
					}
				}

				$ipn->store();

				return $return;
			}

			$transientErrorDoReschedule						=	true;

			$this->_setLogErrorMSG( 3, null, $this->getPayName() . ' Needed Paypal API vendor and password not set.', CBTxt::T( 'Needed Paypal API vendor and password not set.' ) . ' ' . CBTxt::T( 'Please contact site administrator to check error log.' ) );

			return false;
		}

		return parent::processAutoRecurringPayment( $paymentBasket, $returnText, $transientErrorDoReschedule );
	}

	/**
	 * Stops a recurring payment for a basket or for specific payment items
	 *
	 * @param  cbpaidPaymentBasket  $paymentBasket  paymentBasket object
	 * @param  cbpaidPaymentItem[]  $paymentItems   redirect immediately instead of returning HTML for output
	 * @return boolean                              TRUE if unsubscription done successfully, FALSE if error
	 */
	public function stopPaymentSubscription( $paymentBasket, $paymentItems ) {
		if ( ( $this->getAccountParam( 'paypal_api' ) == 2 ) && $paymentBasket->mc_amount3 ) {
			$paymentBasket->unscheduleAutoRecurringPayments();
		}

		return parent::stopPaymentSubscription( $paymentBasket, $paymentItems );
	}

	/**
	 * Refunds a payment
	 *
	 * @param  cbpaidPaymentBasket       $paymentBasket  paymentBasket object
	 * @param  cbpaidPayment             $payment        payment object
	 * @param  cbpaidPaymentItem[]|null  $paymentItems   Array of payment items to refund completely (if no $amount)
	 * @param  boolean                   $lastRefund     Last refund for $payment ?
	 * @param  float                     $amount         Amount in currency of the payment
	 * @param  string                    $reasonText     Refund reason comment text for gateway
	 * @param  string                    $returnText     RETURN param : Text of success message or of error message
	 * @return boolean                                   true if refund done successfully, false if error
	 */
	public function refundPayment( $paymentBasket, $payment, $paymentItems, $lastRefund, $amount, $reasonText, &$returnText )
	{
		global $_CB_framework;

		$return										=	false;

		if ( $this->hasPaypalApi() ) {
			if ( $amount != $payment->mc_gross ) {
				$logType							=	'4';
				$paymentStatus						=	'Partially-Refunded';
				$refundType							=	'Partial';
			} else {
				$logType							=	'3';
				$paymentStatus						=	'Refunded';
				$refundType							=	'Full';
			}

			$endpoint								=	$this->gatewayUrl( 'psp' );

			if ( $this->getAccountParam( 'paypal_api' ) == 2 ) {
				// https://developer.paypal.com/docs/classic/payflow/recurring-billing/
				$requestParams						=	array(	'TENDER'	=>	'C',
																'TRXTYPE'	=>	'C',
																'ORIGID'	=>	$payment->txn_id
															);

				if ( $refundType != 'Full' ) {
					$requestParams['CURRENCY']		=	$payment->mc_currency;
					$requestParams['AMT']			=	sprintf( '%.02f', $amount );
				}

				if ( $reasonText ) {
					$requestParams['COMMENT1']		=	$reasonText;
				}

				$signedParams						=	$this->_signRequestParams( $requestParams );
				$formUrl							=	array();

				foreach ( $signedParams as $k => $v ) {
					$formUrl[$k]					=	$k . '[' . strlen( $v ) . ']=' . $v;
				}

				$formUrl							=	implode( '&', $formUrl );
			} else {
				// https://developer.paypal.com/docs/classic/paypal-payments-pro/integration-guide/recurring-payments/
				$requestParams						=	array(	'METHOD'		=>	'RefundTransaction',
																'TRANSACTIONID'	=>	$payment->txn_id,
																'REFUNDTYPE'	=>	$refundType
															);

				if ( $refundType != 'Full' ) {
					$requestParams['CURRENCYCODE']	=	$payment->mc_currency;
					$requestParams['AMT']			=	sprintf( '%.02f', $amount );
				}

				if ( $reasonText ) {
					$requestParams['NOTE']			=	$reasonText;
				}

				$signedParams						=	$this->_signRequestParams( $requestParams );
				$formUrl							=	$signedParams;
				$endpoint							=	str_replace( 'www', 'api-3t', $this->gatewayUrl( 'psp' ) . '/nvp' );
			}

			$results								=	array();
			$response								=	null;
			$status									=	null;
			$error									=	$this->_httpsRequest( $endpoint, $formUrl, 105, $response, $status, 'post', 'normal' );

			if ( $response ) {
				parse_str( $response, $results );
			}

			$paymentType							=	'instant';
			$reasonCode								=	null;

			$ipn									=	$this->_prepareIpn( $logType, $paymentStatus, $paymentType, $reasonCode, $_CB_framework->now(), 'utf-8' );

			$ipn->bindBasket( $paymentBasket );

			$ipn->user_id							=	(int) $paymentBasket->user_id;

			$rawData								=	'$response="' . preg_replace( '/([^\s]{100})/', '$1 ', $response ) . "\"\n"
													.	'$results=' . var_export( $results, true ) . ";\n"
													.	'$requestParams=' . var_export( $requestParams, true ) . ";\n";

			$ipn->setRawData( $rawData );

			if ( $error || ( $status != 200 ) || ( ! $response ) ) {
				$ipn->setRawResult( 'FAILED' );

				$ipn->payment_status				=	'Failed';

				$this->_setLogErrorMSG( 3, null, $this->getPayName() . ' HTTPS POST request to payment gateway server failed.', CBTxt::T( "Submitted refund request didn't return an error but didn't complete." ) . ' ' . CBTxt::T( 'Please contact site administrator to check error log.' ) );
			} else {
				if ( $this->getAccountParam( 'paypal_api' ) == 2 ) {
					if ( cbGetParam( $results, 'RESULT' ) == '0' ) {
						$ipn->setRawResult( 'SUCCESS' );

						$ipn->mc_currency			=	$payment->mc_currency;
						$ipn->mc_gross				=	( - $amount );

						$ipn->computeRefundedTax( $payment );

						$ipn->txn_id				=	cbGetParam( $results, 'PNREF' );
						$ipn->parent_txn_id			=	$payment->txn_id;

						$this->updatePaymentStatus( $paymentBasket, $ipn->txn_type, $ipn->payment_status, $ipn, 1, 0, 0, false );

						$return						=	true;
					} else {
						$ipn->setRawResult( 'FAILED' );

						$ipn->payment_status		=	'Failed';

						$this->_setLogErrorMSG( 3, null, $this->getPayName() . ' Paypal API error returned. ERROR: ' . cbGetParam( $results, 'RESPMSG' ) . ' CODE: ' . cbGetParam( $results, 'RESULT' ), cbGetParam( $results, 'RESPMSG' ) . '. ' . CBTxt::T( 'Please contact site administrator to check error log.' ) );
					}
				} else {
					if ( cbGetParam( $results, 'ACK' ) == 'Success' ) {
						$ipn->setRawResult( 'SUCCESS' );

						if ( isset( $results['REFUNDINFO'] ) ) {
							$ipn->payment_type		=	cbGetParam( $results['REFUNDINFO'], 'REFUNDSTATUS' );

							$reasonCode				=	cbGetParam( $results['REFUNDINFO'], 'PENDINGREASON' );

							if ( $reasonCode != 'None' ) {
								$ipn->reason_code	=	$reasonCode;
							}
						}

						$ipn->mc_currency			=	cbGetParam( $results, 'CURRENCYCODE' );
						$ipn->mc_gross				=	( - cbGetParam( $results, 'GROSSREFUNDAMT' ) );

						$ipn->computeRefundedTax( $payment );

						$ipn->mc_fee				=	( - cbGetParam( $results, 'FEEREFUNDAMT' ) );
						$ipn->auth_id				=	cbGetParam( $results, 'CORRELATIONID' );
						$ipn->txn_id				=	cbGetParam( $results, 'REFUNDTRANSACTIONID' );
						$ipn->parent_txn_id			=	$payment->txn_id;

						$this->updatePaymentStatus( $paymentBasket, $ipn->txn_type, $ipn->payment_status, $ipn, 1, 0, 0, false );

						$return						=	true;
					} else {
						$ipn->setRawResult( 'FAILED' );

						$ipn->payment_status		=	'Failed';

						$this->_setLogErrorMSG( 3, null, $this->getPayName() . ' Paypal API error returned. ERROR: ' . cbGetParam( $results, 'L_LONGMESSAGE0' ) . ' CODE: ' . cbGetParam( $results, 'L_ERRORCODE0' ), cbGetParam( $results, 'L_SHORTMESSAGE0' ) . '. ' . CBTxt::T( 'Please contact site administrator to check error log.' ) );
					}
				}
			}

			$ipn->store();
		} else {
			if ( $this->getAccountParam( 'paypal_api' ) == 2 ) {
				$this->_setLogErrorMSG( 3, null, $this->getPayName() . ' Needed Paypal API vendor and password not set.', CBTxt::T( 'Needed Paypal API vendor and password not set.' ) . ' ' . CBTxt::T( 'Please contact site administrator to check error log.' ) );
			} else {
				$this->_setLogErrorMSG( 3, null, $this->getPayName() . ' Needed Paypal API username, password and signature not set.', CBTxt::T( 'Needed Paypal API username, password and signature not set.' ) . ' ' . CBTxt::T( 'Please contact site administrator to check error log.' ) );
			}
		}

		return $return;
	}

	/**
	 * CBSUBS ON-SITE CREDIT-CARDS PAGES PAYMENT API METHODS:
	 */

	/**
	 * Attempts to authorize and capture a credit card for a single payment of a payment basket
	 *
	 * @param array $card                           contains type, number, firstname, lastname, expmonth, expyear, and optionally: address, zip, country
	 * @param cbpaidPaymentBasket $paymentBasket
	 * @param int $now                              unix timestamp of now
	 * @param cbpaidsubscriptionsNotification $ipn  returns the stored notification
	 * @param boolean $authnetSubscription          true if it is a subscription and amount is in mc_amount1 and not in mc_gross
	 * @return array|bool                           subscriptionId if subscription request succeeded, otherwise ARRAY( 'level' => 'spurious' or 'fatal', 'errorText', 'errorCode' => string ) of error to display
	 */
	protected function processSinglePayment( $card, $paymentBasket, $now, &$ipn, $authnetSubscription )
	{
		if ( $this->hasPaypalApi() ) {
			$countries				=	new cbpaidCountries();

			if ( $authnetSubscription > 1 ) {
				if ( $paymentBasket->period1 ) {
					$amount			=	sprintf( '%.2f', $paymentBasket->mc_amount1 );
				} else {
					$amount			=	sprintf( '%.2f', $paymentBasket->mc_amount3 );
				}
			} else {
				$amount				=	sprintf( '%.2f', $paymentBasket->mc_gross );
			}

			$ipAddresses			=	cbpaidRequest::getIParray();
			$endpoint				=	$this->gatewayUrl( 'psp' );

			if ( $this->getAccountParam( 'paypal_api' ) == 2 ) {
				if ( $this->getAccountParam( 'paypal_payflow_redirect' ) ) {
					// https://developer.paypal.com/docs/classic/payflow/integration-guide/#submitting-inquiry-transactions
					$requestParams		=	array(	'TENDER'			=>	'C',
													'TRXTYPE'			=>	'I',
													'SECURETOKEN'		=>	Application::Input()->get( 'SECURETOKEN', null, GetterInterface::STRING )
												);
				} else {
					// https://developer.paypal.com/docs/classic/payflow/integration-guide/#sending-a-simple-transaction-to-the-server
					$requestParams		=	array(	'TENDER'			=>	'C',
													'TRXTYPE'			=>	'S',
													'CUSTIP'			=>	substr( array_pop( $ipAddresses ), 0, 15 ),
													'ACCT'				=>	substr( preg_replace ( '/[^0-9]+/', '', strval( $card['number'] ) ), 0, 22 ),
													'EXPDATE'			=>	substr( sprintf( '%02d', intval( $card['expmonth'] ) ), 0, 2 ) . substr( strval( intval( $card['expyear'] ) ), 2, 2 ),
													'CVV2'				=>	substr( preg_replace ( '/[^0-9]+/', '', strval( $card['cvv'] ) ), 0, 4 ),
													'CURRENCY'			=>	$paymentBasket->mc_currency,
													'AMT'				=>	$amount,
													'EMAIL'				=>	cbIsoUtf_substr( $paymentBasket->payer_email, 0, 127 ),
													'BILLTOFIRSTNAME'	=>	cbIsoUtf_substr( $card['firstname'], 0, 30 ),
													'BILLTOLASTNAME'	=>	cbIsoUtf_substr( $card['lastname'], 0, 30 ),
													'BILLTOSTREET'		=>	cbIsoUtf_substr( $paymentBasket->address_street, 0, 30 ),
													'BILLTOCITY'		=>	cbIsoUtf_substr( $paymentBasket->address_city, 0, 20 ),
													'BILLTOSTATE'		=>	cbIsoUtf_substr( $paymentBasket->getInvoiceState(), 0, 2 ),
													'BILLTOCOUNTRY'		=>	cbIsoUtf_substr( $paymentBasket->getInvoiceCountry( 3 ), 0, 3 ),
													'BILLTOZIP'			=>	cbIsoUtf_substr( $paymentBasket->address_zip, 0, 9 ),
													'ORDERID'			=>	$paymentBasket->item_number,
													'INVNUM'			=>	cbIsoUtf_substr( $paymentBasket->invoice, 0, 127 ),
													'COMMENT1'			=>	cbIsoUtf_substr( $paymentBasket->item_name, 0, 128 ),
													'USER1'				=>	$paymentBasket->id,
													'BUTTONSOURCE'		=>	'Joomlapolis_Cart_WPS' // our bn code per email of Greg Campagnolo of 21 avril 2012 02:39:24 and 1 mai 2012 21:33:22 and reminder of Rickard of 16.3.2015
												);
				}

				$signedParams			=	$this->_signRequestParams( $requestParams );
				$formUrl				=	array();

				foreach ( $signedParams as $k => $v ) {
					$formUrl[$k]		=	$k . '[' . strlen( $v ) . ']=' . $v;
				}

				$formUrl				=	implode( '&', $formUrl );
			} else {
				// https://developer.paypal.com/docs/classic/paypal-payments-pro/integration-guide/direct-payment/
				$requestParams		=	array(	'METHOD'			=>	'DoDirectPayment',
												'PAYMENTACTION'		=>	'Sale',
												'IPADDRESS'			=>	substr( array_pop( $ipAddresses ), 0, 15 ),
												'CREDITCARDTYPE'	=>	cbIsoUtf_substr( ( $card['type'] == 'amexco' ? 'amex' : $card['type'] ), 0, 10 ),
												'ACCT'				=>	substr( preg_replace ( '/[^0-9]+/', '', strval( $card['number'] ) ), 0, 22 ),
												'EXPDATE'			=>	substr( sprintf( '%02d', intval( $card['expmonth'] ) ), 0, 2 ) . substr( strval( intval( $card['expyear'] ) ), 0, 4 ),
												'CVV2'				=>	substr( preg_replace ( '/[^0-9]+/', '', strval( $card['cvv'] ) ), 0, 4 ),
												'EMAIL'				=>	cbIsoUtf_substr( $paymentBasket->payer_email, 0, 127 ),
												'FIRSTNAME'			=>	cbIsoUtf_substr( $card['firstname'], 0, 32 ),
												'LASTNAME'			=>	cbIsoUtf_substr( $card['lastname'], 0, 32 ),
												'STREET'			=>	cbIsoUtf_substr( $paymentBasket->address_street, 0, 100 ),
												'CITY'				=>	cbIsoUtf_substr( $paymentBasket->address_city, 0, 40 ),
												'STATE'				=>	cbIsoUtf_substr( $paymentBasket->getInvoiceState(), 0, 40 ),
												'COUNTRYCODE'		=>	cbIsoUtf_substr( $paymentBasket->getInvoiceCountry( 2 ), 0, 2 ),
												'ZIP'				=>	cbIsoUtf_substr( $paymentBasket->address_zip, 0, 20 ),
												'AMT'				=>	$amount,
												'CURRENCYCODE'		=>	$paymentBasket->mc_currency,
												'DESC'				=>	$paymentBasket->item_name,
												'BUTTONSOURCE'		=>	'Joomlapolis_Cart_WPS', // our bn code per email of Greg Campagnolo of 21 avril 2012 02:39:24 and 1 mai 2012 21:33:22 and reminder of Rickard of 16.3.2015
												'CUSTOM'			=>	$paymentBasket->id,
												'INVNUM'			=>	$paymentBasket->invoice,
												'NOTIFYURL'			=>	$this->getNotifyUrl( $paymentBasket )
											);

				$signedParams		=	$this->_signRequestParams( $requestParams );
				$formUrl			=	$signedParams;
				$endpoint			=	str_replace( 'www', 'api-3t', $endpoint . '/nvp' );
			}

			$results				=	array();
			$response				=	null;
			$status					=	null;
			$error					=	$this->_httpsRequest( $endpoint, $formUrl, 105, $response, $status, 'post', 'normal' );

			if ( $response ) {
				parse_str( $response, $results );
			}

			if ( $error || ( $status != 200 ) || ( ! $response ) ) {
				$return				=	array(	'level'		=>	'spurious',
												'errorText'	=>	CBTxt::T( "Submitted subscription payment didn't return an error but didn't complete." ) . ' ' . CBTxt::T( 'Please contact site administrator to check error log.' ),
												'errorCode'	=>	'8888'
											);

				$logType			=	'B';
			} else {
				if ( $this->getAccountParam( 'paypal_api' ) == 2 ) {
					if ( in_array( cbGetParam( $results, 'RESULT' ), array( '0', '126' ) ) ) {
						$return		=	cbGetParam( $results, 'PNREF' );

						$logType	=	'P';
					} else {
						$return		=	array(	'level'		=>	'fatal',
												'errorText'	=>	cbGetParam( $results, 'RESPMSG' ) . '. ' . CBTxt::T( 'Please contact site administrator to check error log.' ),
												'errorCode'	=>	cbGetParam( $results, 'RESULT' )
											);

						$logType	=	'V';
					}
				} else {
					if ( cbGetParam( $results, 'ACK' ) == 'Success' ) {
						$return		=	cbGetParam( $results, 'TRANSACTIONID' );

						$logType	=	'P';
					} else {
						$return		=	array(	'level'		=>	'fatal',
												'errorText'	=>	cbGetParam( $results, 'L_SHORTMESSAGE0' ) . '. ' . CBTxt::T( 'Please contact site administrator to check error log.' ),
												'errorCode'	=>	cbGetParam( $results, 'L_ERRORCODE0' )
											);

						$logType	=	'V';
					}
				}
			}

			$ipn					=	$this->_logNotification( $logType, $now, $paymentBasket, $card, $requestParams, $response, $results, $return );
		} else {
			if ( $this->getAccountParam( 'paypal_api' ) == 2 ) {
				$return				=	array(	'level'		=>	'fatal',
												'errorText'	=>	CBTxt::T( 'Needed Paypal API vendor and password not set.' ) . ' ' . CBTxt::T( 'Please contact site administrator to check error log.' ),
												'errorCode'	=>	'8888'
											);
			} else {
				$return				=	array(	'level'		=>	'fatal',
												'errorText'	=>	CBTxt::T( 'Needed Paypal API username, password and signature not set.' ) . ' ' . CBTxt::T( 'Please contact site administrator to check error log.' ),
												'errorCode'	=>	'8888'
											);
			}
		}

		return $return;
	}

	/**
	 * Attempts to subscribe a credit card for recurring subscription of a payment basket.
	 *
	 * @param array $card                           contains type, number, firstname, lastname, expmonth, expyear, and optionally: address, zip, country
	 * @param cbpaidPaymentBasket $paymentBasket
	 * @param int $now                              unix timestamp of now
	 * @param cbpaidsubscriptionsNotification $ipn  returns the stored notification
	 * @param int $occurrences                      returns the number of occurences pay-subscribed firmly
	 * @param int $autorecurring_type               returns:  0: not auto-recurring, 1: auto-recurring without payment processor notifications, 2: auto-renewing with processor notifications updating $expiry_date
	 * @param int $autorenew_type                   returns:  0: not auto-renewing (manual renewals), 1: asked for by user, 2: mandatory by configuration
	 * @return mixed                                subscriptionId if subscription request succeeded, otherwise ARRAY( 'level' => 'inform', 'spurious' or 'fatal', 'errorText', 'errorCode' => string ) of error to display
	 */
	protected function processSubscriptionPayment( $card, $paymentBasket, $now, &$ipn, &$occurrences, &$autorecurring_type, &$autorenew_type )
	{
		if ( $this->hasPaypalApi() ) {
			$countries										=	new cbpaidCountries();

			list( $p3, $t3, $start )						=	$this->_paypalPeriodsLimits( explode( ' ', $paymentBasket->period3 ), $now );

			if ( $paymentBasket->period1 ) {
				list( /* $p1 */, /* $t1 */, $start )		=	$this->_paypalPeriodsLimits( explode( ' ', $paymentBasket->period1 ), $now );

				$initialAmount								=	$paymentBasket->mc_amount1;
			} else {
				$initialAmount								=	$paymentBasket->mc_amount3;
			}

			$ipAddresses									=	cbpaidRequest::getIParray();
			$endpoint										=	$this->gatewayUrl( 'psp' );

			if ( $this->getAccountParam( 'paypal_api' ) == 2 ) {
				// https://developer.paypal.com/docs/classic/payflow/recurring-billing/
				$requestParams								=	array(	'TENDER'			=>	'C',
																		'TRXTYPE'			=>	'R',
																		'ACTION'			=>	'A',
																		'CUSTIP'			=>	substr( array_pop( $ipAddresses ), 0, 15 ),
																		'PROFILENAME'		=>	cbIsoUtf_substr( $paymentBasket->item_name, 0, 128 ),
																		'CURRENCY'			=>	$paymentBasket->mc_currency,
																		'AMT'				=>	sprintf( '%.2f', $paymentBasket->mc_amount3 ),
																		'START'				=>	substr( gmdate( 'mdY', $start ), 0, 8 ),
																		'TERM'				=>	( $paymentBasket->recur_times ? $paymentBasket->recur_times : 0 ),
																		'PAYPERIOD'			=>	$t3,
																		'EMAIL'				=>	cbIsoUtf_substr( $paymentBasket->payer_email, 0, 127 ),
																		'BILLTOSTREET'		=>	cbIsoUtf_substr( $paymentBasket->address_street, 0, 30 ),
																		'BILLTOCITY'		=>	cbIsoUtf_substr( $paymentBasket->address_city, 0, 20 ),
																		'BILLTOSTATE'		=>	cbIsoUtf_substr( $paymentBasket->getInvoiceState(), 0, 2 ),
																		'BILLTOCOUNTRY'		=>	cbIsoUtf_substr( $paymentBasket->getInvoiceCountry( 3 ), 0, 3 ),
																		'BILLTOZIP'			=>	cbIsoUtf_substr( $paymentBasket->address_zip, 0, 9 ),
																		'ORDERID'			=>	$paymentBasket->item_number,
																		'INVNUM'			=>	cbIsoUtf_substr( $paymentBasket->invoice, 0, 127 ),
																		'COMMENT1'			=>	cbIsoUtf_substr( $paymentBasket->item_name, 0, 128 ),
																		'USER1'				=>	$paymentBasket->id,
																		'BUTTONSOURCE'		=>	'Joomlapolis_Cart_WPS' // our bn code per email of Greg Campagnolo of 21 avril 2012 02:39:24 and 1 mai 2012 21:33:22 and reminder of Rickard of 16.3.2015
																	);

				if ( $t3 == 'DAYS' ) {
					$requestParams['FREQUENCY']				=	$p3;
				}

				if ( $this->getAccountParam( 'paypal_payflow_redirect' ) ) {
					$requestParams['BILLTOFIRSTNAME']		=	cbIsoUtf_substr( Application::Input()->get( 'BILLTOFIRSTNAME', $paymentBasket->get( 'first_name', null, GetterInterface::STRING ), GetterInterface::STRING ), 0, 25 );
					$requestParams['BILLTOLASTNAME']		=	cbIsoUtf_substr( Application::Input()->get( 'BILLTOLASTNAME', $paymentBasket->get( 'last_name', null, GetterInterface::STRING ), GetterInterface::STRING ), 0, 25 );
					$requestParams['ORIGID']				=	Application::Input()->get( 'PNREF', null, GetterInterface::STRING );
				} else {
					$requestParams['BILLTOFIRSTNAME']		=	cbIsoUtf_substr( $card['firstname'], 0, 25 );
					$requestParams['BILLTOLASTNAME']		=	cbIsoUtf_substr( $card['lastname'], 0, 25 );
					$requestParams['ACCT']					=	substr( preg_replace ( '/[^0-9]+/', '', strval( $card['number'] ) ), 0, 22 );
					$requestParams['EXPDATE']				=	substr( sprintf( '%02d', intval( $card['expmonth'] ) ), 0, 2 ) . substr( strval( intval( $card['expyear'] ) ), 2, 2 );
					$requestParams['CVV2']					=	substr( preg_replace ( '/[^0-9]+/', '', strval( $card['cvv'] ) ), 0, 4 );
					$requestParams['OPTIONALTRX']			=	'S';
					$requestParams['OPTIONALTRXAMT']		=	sprintf( '%.2f', $initialAmount );
				}

				$signedParams								=	$this->_signRequestParams( $requestParams );
				$formUrl									=	array();

				foreach ( $signedParams as $k => $v ) {
					$formUrl[$k]							=	$k . '[' . strlen( $v ) . ']=' . $v;
				}

				$formUrl									=	implode( '&', $formUrl );
			} else {
				// https://developer.paypal.com/docs/classic/paypal-payments-pro/integration-guide/recurring-payments/
				$requestParams								=	array(	'METHOD'			=>	'CreateRecurringPaymentsProfile',
																		'SUBSCRIBERNAME'	=>	cbIsoUtf_substr( $card['firstname'] . ' ' . $card['lastname'], 0, 32 ),
																		'PROFILESTARTDATE'	=>	substr( gmdate( 'c', $start ), 0, 19 ),
																		'PROFILEREFERENCE'	=> $paymentBasket->invoice,
																		'DESC'				=>	cbIsoUtf_substr( $paymentBasket->item_name, 0, 127 ),
																		'BILLINGPERIOD'		=>	$t3,
																		'BILLINGFREQUENCY'	=>	$p3,
																		'INITAMT'			=>	sprintf( '%.2f', $initialAmount ),
																		'AMT'				=>	sprintf( '%.2f', $paymentBasket->mc_amount3 ),
																		'CURRENCYCODE'		=>	$paymentBasket->mc_currency,
																		'CREDITCARDTYPE'	=>	cbIsoUtf_substr( ( $card['type'] == 'amexco' ? 'amex' : $card['type'] ), 0, 10 ),
																		'ACCT'				=>	substr( preg_replace ( '/[^0-9]+/', '', strval( $card['number'] ) ), 0, 22 ),
																		'EXPDATE'			=>	substr( sprintf( '%02d', intval( $card['expmonth'] ) ), 0, 2 ) . substr( strval( intval( $card['expyear'] ) ), 0, 4 ),
																		'CVV2'				=>	substr( preg_replace ( '/[^0-9]+/', '', strval( $card['cvv'] ) ), 0, 4 ),
																		'EMAIL'				=>	cbIsoUtf_substr( $paymentBasket->payer_email, 0, 127 ),
																		'PAYERID'			=>	$paymentBasket->user_id,
																		'FIRSTNAME'			=>	cbIsoUtf_substr( $card['firstname'], 0, 32 ),
																		'LASTNAME'			=>	cbIsoUtf_substr( $card['lastname'], 0, 32 ),
																		'STREET'			=>	cbIsoUtf_substr( $paymentBasket->address_street, 0, 100 ),
																		'CITY'				=>	cbIsoUtf_substr( $paymentBasket->address_city, 0, 40 ),
																		'STATE'				=>	cbIsoUtf_substr( $paymentBasket->getInvoiceState(), 0, 40 ),
																		'COUNTRYCODE'		=>	cbIsoUtf_substr( $paymentBasket->getInvoiceCountry( 2 ), 0, 2 ),
																		'ZIP'				=>	cbIsoUtf_substr( $paymentBasket->address_zip, 0, 20 ),
																		'BUTTONSOURCE'		=>	'Joomlapolis_Cart_WPS'		// our bn code per email of Greg Campagnolo of 21 avril 2012 02:39:24 and 1 mai 2012 21:33:22 and reminder of Rickard of 16.3.2015
																	);

				if ( $paymentBasket->recur_times ) {
					$requestParams['TOTALBILLINGCYCLES']	=	$paymentBasket->recur_times;
				}

				$signedParams								=	$this->_signRequestParams( $requestParams );
				$formUrl									=	$signedParams;
				$endpoint									=	str_replace( 'www', 'api-3t', $this->gatewayUrl( 'psp' ) . '/nvp' );
			}

			$results										=	array();
			$response										=	null;
			$status											=	null;
			$error											=	$this->_httpsRequest( $endpoint, $formUrl, 105, $response, $status, 'post', 'normal' );

			if ( $response ) {
				parse_str( $response, $results );
			}

			if ( $error || ( $status != 200 ) || ( ! $response ) ) {
				$return										=	array(	'level'		=>	'spurious',
																		'errorText'	=>	CBTxt::T( "Submitted subscription payment didn't return an error but didn't complete." ) . ' ' . CBTxt::T( 'Please contact site administrator to check error log.' ),
																		'errorCode'	=>	'8888'
																	);

				$logType									=	'C';
			} else {
				if ( $this->getAccountParam( 'paypal_api' ) == 2 ) {
					if ( ( $this->getAccountParam( 'paypal_payflow_redirect' ) && in_array( cbGetParam( $results, 'RESULT' ), array( '0', '126' ) ) )
						 || ( in_array( cbGetParam( $results, 'RESULT' ), array( '0', '126' ) ) && in_array( cbGetParam( $results, 'TRXRESULT' ), array( '0', '126' ) ) )
					) {
						$autorecurring_type					=	2;
						$autorenew_type						=	( ( $this->getAccountParam( 'enabled', 0 ) == 3 ) && ( $paymentBasket->isAnyAutoRecurring() == 2 ) ? 1 : 2 );

						$return								=	cbGetParam( $results, 'PROFILEID' );

						$logType							=	'A';
					} else {
						$return								=	array(	'level'		=>	'fatal',
																		'errorText'	=>	cbGetParam( $results, 'RESPMSG' ) . ' / ' . cbGetParam( $results, 'TRXRESPMSG' ) . '. ' . CBTxt::T( 'Please contact site administrator to check error log.' ),
																		'errorCode'	=>	cbGetParam( $results, 'RESULT' ) . ' / ' . cbGetParam( $results, 'TRXRESULT' )
																	);

						$logType							=	'W';
					}
				} else {
					if ( cbGetParam( $results, 'ACK' ) == 'Success' ) {
						$autorecurring_type					=	2;
						$autorenew_type						=	( ( $this->getAccountParam( 'enabled', 0 ) == 3 ) && ( $paymentBasket->isAnyAutoRecurring() == 2 ) ? 1 : 2 );

						$return								=	cbGetParam( $results, 'TRANSACTIONID' );

						$logType							=	'A';
					} else {
						$return								=	array(	'level'		=>	'fatal',
																		'errorText'	=>	cbGetParam( $results, 'L_SHORTMESSAGE0' ) . '. ' . CBTxt::T( 'Please contact site administrator to check error log.' ),
																		'errorCode'	=>	cbGetParam( $results, 'L_ERRORCODE0' )
																	);

						$logType							=	'W';
					}
				}
			}

			$ipn											=	$this->_logNotification( $logType, $now, $paymentBasket, $card, $requestParams, $response, $results, $return );
		} else {
			if ( $this->getAccountParam( 'paypal_api' ) == 2 ) {
				$return										=	array(	'level'		=>	'fatal',
																		'errorText'	=>	CBTxt::T( 'Needed Paypal API vendor and password not set.' ) . ' ' . CBTxt::T( 'Please contact site administrator to check error log.' ),
																		'errorCode'	=>	'8888'
																	);
			} else {
				$return										=	array(	'level'		=>	'fatal',
																		'errorText'	=>	CBTxt::T( 'Needed Paypal API username, password and signature not set.' ) . ' ' . CBTxt::T( 'Please contact site administrator to check error log.' ),
																		'errorCode'	=>	'8888'
																	);
			}
		}

		return $return;
	}

	/**
	 * Attempts to unsubscribe a subscription of a payment basket.
	 *
	 * @param  cbpaidPaymentBasket              $paymentBasket
	 * @param  cbpaidPaymentItem[]              $paymentItems
	 * @param  cbpaidsubscriptionsNotification  $ipn                        returns the stored notification
	 * @param  string                           $authorize_subscription_id
	 * @return mixed                                                        subscriptionId if subscription request succeeded, otherwise ARRAY( 'level' => 'inform', 'spurious' or 'fatal', 'errorText', 'errorCode' => string ) of error to display
	 */
	protected function processSubscriptionCancellation( $paymentBasket, $paymentItems, &$ipn, $authorize_subscription_id )
	{
		global $_CB_framework;

		if ( $this->hasPaypalApi() ) {
			$endpoint					=	$this->gatewayUrl( 'psp' );

			if ( $this->getAccountParam( 'paypal_api' ) == 2 ) {
				// https://developer.paypal.com/docs/classic/payflow/recurring-billing/
				$requestParams			=	array(	'TENDER'			=>	'C',
													'TRXTYPE'			=>	'R',
													'ACTION'			=>	'C',
													'ORIGPROFILEID'		=>	$authorize_subscription_id
												);

				$signedParams			=	$this->_signRequestParams( $requestParams );
				$formUrl				=	array();

				foreach ( $signedParams as $k => $v ) {
					$formUrl[$k]		=	$k . '[' . strlen( $v ) . ']=' . $v;
				}

				$formUrl				=	implode( '&', $formUrl );
			} else {
				// https://developer.paypal.com/docs/classic/paypal-payments-pro/integration-guide/recurring-payments/
				$requestParams			=	array(	'METHOD'		=>	'ManageRecurringPaymentsProfileStatus',
													'PROFILEID'		=>	$authorize_subscription_id,
													'ACTION'		=>	'Cancel'
												);

				$signedParams			=	$this->_signRequestParams( $requestParams );
				$formUrl				=	$signedParams;
				$endpoint				=	str_replace( 'www', 'api-3t', $this->gatewayUrl( 'psp' ) . '/nvp' );
			}

			$results					=	array();
			$response					=	null;
			$status						=	null;
			$error						=	$this->_httpsRequest( $endpoint, $formUrl, 105, $response, $status, 'post', 'normal' );

			if ( $response ) {
				parse_str( $response, $results );
			}

			if ( $error || ( $status != 200 ) || ( ! $response ) ) {
				$return					=	array(	'level'		=>	'spurious',
													'errorText'	=>	CBTxt::T( "Submitted subscription payment didn't return an error but didn't complete." ) . ' ' . CBTxt::T( 'Please contact site administrator to check error log.' ),
													'errorCode'	=>	'8888'
												);

				$logType				=	'C';
			} else {
				if ( $this->getAccountParam( 'paypal_api' ) == 2 ) {
					if ( cbGetParam( $results, 'RESULT' ) == '0' ) {
						$return			=	cbGetParam( $results, 'PROFILEID' );

						$logType		=	'5';
					} else {
						$return			=	array(	'level'		=>	'fatal',
													'errorText'	=>	cbGetParam( $results, 'RESPMSG' ) . '. ' . CBTxt::T( 'Please contact site administrator to check error log.' ),
													'errorCode'	=>	cbGetParam( $results, 'RESULT' )
												);

						$logType		=	'W';
					}
				} else {
					if ( cbGetParam( $results, 'ACK' ) == 'Success' ) {
						$return			=	cbGetParam( $results, 'PROFILEID' );

						$logType		=	'5';
					} else {
						if ( cbGetParam( $results, 'L_ERRORCODE0' ) == '11556' ) {
							$return		=	$authorize_subscription_id;

							$logType	=	'5';
						} else {
							$return		=	array(	'level'		=>	'fatal',
													'errorText'	=>	cbGetParam( $results, 'L_SHORTMESSAGE0' ) . '. ' . CBTxt::T( 'Please contact site administrator to check error log.' ),
													'errorCode'	=>	cbGetParam( $results, 'L_ERRORCODE0' )
												);

							$logType	=	'W';
						}
					}
				}
			}

			$ipn						=	$this->_logNotification( $logType, $_CB_framework->now(), $paymentBasket, null, $requestParams, $response, $results, $return );
		} else {
			if ( $this->getAccountParam( 'paypal_api' ) == 2 ) {
				$return					=	array(	'level'		=>	'fatal',
													'errorText'	=>	CBTxt::T( 'Needed Paypal API vendor and password not set.' ) . ' ' . CBTxt::T( 'Please contact site administrator to check error log.' ),
													'errorCode'	=>	'8888'
												);
			} else {
				$return					=	array(	'level'		=>	'fatal',
													'errorText'	=>	CBTxt::T( 'Needed Paypal API username, password and signature not set.' ) . ' ' . CBTxt::T( 'Please contact site administrator to check error log.' ),
													'errorCode'	=>	'8888'
												);
			}
		}

		return $return;
	}

	/**
	 * Handles a gateway notification
	 *
	 * @param cbpaidPaymentBasket $paymentBasket
	 * @param array $postdata
	 * @return bool
	 */
	protected function handleNotify( $paymentBasket, $postdata )
	{
		global $_CB_framework;

		if ( $this->getAccountParam( 'paypal_api' ) == 2 ) {
			$this->_setLogErrorMSG( 3, null, $this->getPayName() . ' This gateway does not support notifications.', CBTxt::T( 'This gateway does not support notifications.' ) . ' ' . CBTxt::T( 'Please contact site administrator to check error log.' ) );

			return false;
		}

		$transactionId												=	cbGetParam( $postdata, 'txn_id', null );
		$subscriptionId												=	cbGetParam( $postdata, 'recurring_payment_id', null );
		$invoiceId													=	cbGetParam( $postdata, 'invoice', null );

		if ( ! $invoiceId ) {
			$invoiceId												=	cbGetParam( $postdata, 'rp_invoice_id', null );
		}

		$return														=	false;

		if ( $transactionId || $subscriptionId || $invoiceId ) {
			if ( $this->hasPaypalApi() ) {
				// We need to perform a series of checks to locate the basket since we actually have 2 different APIs implemented in this gateway:
				$custom												=	(int) cbGetParam( $postdata, 'custom', null );
				$exists												=	false;

				if ( $custom ) {
					// First check if our custom variable was set back containing basket id (some paypal APIs do not allow custom):
					$exists											=	$paymentBasket->load( $custom );
				}

				if ( $subscriptionId && ( ! $exists ) ) {
					// We couldn't find a basket with custom, but we've a subscription id so lets try that:
					$exists											=	$paymentBasket->loadThisMatching( array( 'subscr_id' => $subscriptionId ) );
				}

				if ( $transactionId && ( ! $exists ) ) {
					// We couldn't find a basket with custom and we're not a subscription, but we've a transaction id so lets try that:
					$exists											=	$paymentBasket->loadThisMatching( array( 'txn_id' => $transactionId ) );

					// Set subscription id if it isn't set as we could be a transaction for a subscription:
					if ( ( ! $subscriptionId ) && $exists ) {
						$subscriptionId								=	$paymentBasket->subscr_id;
					}
				}

				if ( $invoiceId && ( ! $exists ) ) {
					// We don't have custom, subscription id, or transaction id so lets failsafe and check for the invoice:
					$exists											=	$paymentBasket->loadThisMatching( array( 'invoice' => $invoiceId ) );

					if ( $exists ) {
						// Set transaction id if it isn't set as we could be a invoice for a single purchase:
						if ( ! $transactionId )  {
							$transactionId							=	$paymentBasket->txn_id;
						}

						// Set subscription id if it isn't set as we could be a invoice for a subscription:
						if ( ! $subscriptionId )  {
							$subscriptionId							=	$paymentBasket->subscr_id;
						}
					}
				}

				if ( $exists ) {
					$paymentStatus									=	cbGetParam( $postdata, 'payment_status', null );

					if ( ! $paymentStatus )  {
						$paymentStatus								=	cbGetParam( $postdata, 'initial_payment_status', null );
					}

					$paymentType									=	cbGetParam( $postdata, 'payment_type', null );

					$ipn											=	$this->_prepareIpn( 'I', $paymentStatus, $paymentType, null, $_CB_framework->now(), 'utf-8' );

					$ipn->bindBasket( $paymentBasket );

					$ipn->user_id									=	(int) $paymentBasket->user_id;

					$ipn->bind( $postdata );

					$ipn->item_number								=	$paymentBasket->item_number;

					if ( $subscriptionId ) {
						if ( ! $ipn->payment_status ) {
							$profileStatus							=	cbGetParam( $postdata, 'profile_status', null );

							if ( $profileStatus == 'Cancelled' ) {
								$ipn->payment_status				=	'Unsubscribed';
								$ipn->payment_date					=	cbGetParam( $postdata, 'time_created', null );
							} elseif ( $profileStatus == 'Active' ) {
								$ipn->payment_status				=	'Completed';
								$ipn->payment_date					=	cbGetParam( $postdata, 'time_created', null );
							} elseif ( $profileStatus == 'Expired' ) {
								$ipn->payment_status				=	'Unsubscribed';
								$ipn->payment_date					=	cbGetParam( $postdata, 'time_created', null );
							} elseif ( $profileStatus == 'Suspended' ) {
								$ipn->payment_status				=	'Denied';
								$ipn->payment_date					=	cbGetParam( $postdata, 'time_created', null );
							} elseif ( $profileStatus == 'Pending' ) {
								$ipn->payment_status				=	'Pending';
								$ipn->payment_date					=	cbGetParam( $postdata, 'time_created', null );
							}
						}

						$requestParams								=	array(	'METHOD'		=>	'GetRecurringPaymentsProfileDetails',
																				'PROFILEID'		=>	$subscriptionId
																			);
					} else {
						$requestParams								=	array(	'METHOD'		=>	'GetTransactionDetails',
																				'TRANSACTIONID'	=>	$transactionId
																			);
					}

					$signedParams									=	$this->_signRequestParams( $requestParams );
					$results										=	array();
					$response										=	null;
					$status											=	null;
					$error											=	$this->_httpsRequest( str_replace( 'www', 'api-3t', $this->gatewayUrl( 'psp' ) . '/nvp' ), $signedParams, 105, $response, $status, 'post', 'normal' );

					if ( $response ) {
						parse_str( $response, $results );
					}

					$rawData										=	'$response="' . preg_replace( '/([^\s]{100})/', '$1 ', $response ) . "\"\n"
																	.	'$results=' . var_export( $results, true ) . ";\n"
																	.	'$requestParams=' . var_export( $requestParams, true ) . ";\n"
																	.	'$postdata=' . var_export( $postdata, true ) . ";\n";

					$ipn->setRawData( $rawData );

					if ( $error || ( $status != 200 ) || ( ! $response ) ) {
						$ipn->log_type								=	'D';

						$ipn->setRawResult( 'COMMUNICATION ERROR' );

						$this->_setLogErrorMSG( 3, $ipn, $this->getPayName() . ' HTTPS POST request to payment gateway server failed.', CBTxt::T( "Submitted transaction details request didn't return an error but didn't complete." ) . ' ' . CBTxt::T( 'Please contact site administrator to check error log.' ) );
					} else {
						$insToIpn									=	array(	'address_street'		=>	'STREET',
																				'address_city'			=>	'CITY',
																				'address_state'			=>	'STATE',
																				'address_zip'			=>	'ZIP',
																				'address_country'		=>	'COUNTRY',
																				'address_country_code'	=>	'COUNTRYCODE',
																				'address_status'		=>	'ADDRESSSTATUS',
																				'first_name'			=>	'FIRSTNAME',
																				'last_name'				=>	'LASTNAME',
																				'payer_email'			=>	'EMAIL',
																				'payer_id'				=>	'PAYERID',
																				'payer_status'			=>	'PAYERSTATUS',
																				'auth_id'				=>	'CORRELATIONID',
																				'tax'					=>	'TAXAMT',
																				'mc_currency'			=>	'CURRENCYCODE',
																				'mc_fee'				=>	'FEEAMT',
																				'mc_gross'				=>	'AMT'
																			);

						if ( $ipn->payment_status == 'Refunded' ) {
							unset( $insToIpn['mc_fee'] );
							unset( $insToIpn['mc_gross'] );
						}

						foreach ( $insToIpn as $k => $v ) {
							$apiValue								=	cbGetParam( $results, $v );

							if ( $apiValue && ( ! in_array( $apiValue, array( '0.00', 'None' ) ) ) ) {
								$ipn->$k							=	$apiValue;
							}
						}

						switch ( $ipn->txn_type ) {
							case 'recurring_payment':
								$ipn->txn_type						=	'subscr_payment';
								break;
							case 'recurring_payment_profile_created':
								$ipn->txn_type						=	'subscr_signup';
								break;
							case 'recurring_payment_profile_cancel':
								$ipn->txn_type						=	'subscr_cancel';
								break;
							case 'recurring_payment_expired':
								$ipn->txn_type						=	'subscr_eot';
								break;
							case 'recurring_payment_skipped':
								$ipn->txn_type						=	'subscr_failed';
								break;
						}

						$valid										=	$this->_validateIPN( $ipn, $paymentBasket, cbGetParam( $_REQUEST, 'cbpid' ) );

						if ( $valid === true ) {
							if ( cbGetParam( $results, 'ACK' ) == 'Success' ) {
								$ipn->setRawResult( 'SUCCESS' );

								if ( $ipn->txn_type != 'subscr_signup' ) {
									$autorecurring_type				=	( in_array( $ipn->txn_type, array( 'subscr_payment', 'subscr_signup', 'subscr_modify', 'subscr_eot', 'subscr_cancel', 'subscr_failed' ) ) ? 2 : 0 );
									$autorenew_type					=	( $autorecurring_type ? ( ( ( $this->getAccountParam( 'enabled', 0 ) == 3 ) && ( $paymentBasket->isAnyAutoRecurring() == 2 ) ) ? 1 : 2 ) : 0 );

									if ( $autorecurring_type && ( $ipn->txn_type == 'subscr_signup' ) && ( ( $paymentBasket->period1 ) && ( $paymentBasket->mc_amount1 == 0 ) ) && ( $ipn->payment_status == '' ) ) {
										$ipn->payment_status		=	'Completed';
									}

									if ( ( $ipn->payment_status == 'Refunded' ) && ( $paymentBasket->mc_gross != ( - $ipn->mc_gross ) ) ) {
										$ipn->payment_status		=	'Partially-Refunded';
									}

									if ( in_array( $ipn->txn_type, array( 'subscr_eot', 'subscr_cancel', 'subscr_failed' ) ) ) {
										$autorecurring_type			=	0;
									}

									if ( $autorecurring_type ) {
										$paymentBasket->reattempt	=	1;
									}

									$this->updatePaymentStatus( $paymentBasket, $ipn->txn_type, $ipn->payment_status, $ipn, 1, $autorecurring_type, $autorenew_type, false );

									$return							=	true;
								} else {
									// subscr_signup has already been handled during api call to create the subscription so just skip it:
									return null;
								}
							} else {
								$ipn->log_type						=	'M';

								$ipn->setRawResult( 'FAILED' );

								$this->_setLogErrorMSG( 3, $ipn, $this->getPayName() . ' Paypal API error returned. ERROR: ' . cbGetParam( $results, 'L_LONGMESSAGE0' ) . ' CODE: ' . cbGetParam( $results, 'L_ERRORCODE0' ), cbGetParam( $results, 'L_SHORTMESSAGE0' ) . '. ' . CBTxt::T( 'Please contact site administrator to check error log.' ) );
							}
						} else {
							$ipn->log_type							=	'O';

							$ipn->setRawResult( 'MISMATCH' );

							$this->_setLogErrorMSG( 3, $ipn, $this->getPayName() . ' Paypal IPN fraud attempt. ERROR: ' . $valid, CBTxt::T( 'Invalid transaction.' ) . ' ' . CBTxt::T( 'Please contact site administrator to check error log.' ) );
						}
					}

					$ipn->store();
				} else {
					$return											=	null;

					if ( $_CB_framework->getCfg( 'debug' ) ) {
						// We don't normally bother logging missing baskets, but lets do so if debug is enabled:
						$this->_setLogErrorMSG( 7, null, $this->getPayName() . ' Payment Basket missing' . "\n" . '$subscriptionId=' . $subscriptionId . "\n" . '$transactionId=' . $transactionId . "\n" . '$invoiceId=' . $invoiceId . "\n" . '$_GET=' . var_export( $_GET, true ) . "\n" . '$_POST=' . var_export( $postdata, true ) . "\n", null );
					}
				}
			} else {
				$this->_setLogErrorMSG( 3, null, $this->getPayName() . ' Needed Paypal API username, password and signature not set.' . "\n" . '$_GET=' . var_export( $_GET, true ) . "\n" . '$_POST=' . var_export( $_POST, true ) . "\n", CBTxt::T( 'Needed Paypal API username, password and signature not set.' ) . ' ' . CBTxt::T( 'Please contact site administrator to check error log.' ) );
			}
		} else {
			$this->_setLogErrorMSG( 3, null, $this->getPayName() . ' Needed Paypal Transaction ID (txn_id) or Subscription ID (recurring_payment_id) missing.' . "\n" . '$_GET=' . var_export( $_GET, true ) . "\n" . '$_POST=' . var_export( $_POST, true ) . "\n", CBTxt::T( 'Transaction or Subscription not found.' ) . ' ' . CBTxt::T( 'Please contact site administrator to check error log.' ) );
		}

		return $return;
	}

	/**
	 * PRIVATE METHODS OF THIS CLASS
	 */

	/**
	 * Checks if a Paypal API is set
	 *
	 * @return bool
	 */
	private function hasPaypalApi( )
	{
		if ( $this->getAccountParam( 'paypal_api' ) == 2 ) {
			return ( $this->getAccountParam( 'paypal_payflow_vendor' ) && $this->getAccountParam( 'paypal_payflow_password' ) );
		}

		return ( $this->getAccountParam( 'paypal_api_username' ) && $this->getAccountParam( 'paypal_api_password' ) && $this->getAccountParam( 'paypal_api_signature' ) );
	}

	/**
	 * Returns the single or recurring payment token inputs for custom CC form
	 *
	 * @param cbpaidPaymentBasket $paymentBasket
	 * @param string              $cardType
	 * @param bool                $recurring
	 * @return null|string
	 */
	private function getPayPalTokenInputs( $paymentBasket, $cardType, $recurring = false )
	{
		if ( $paymentBasket->get( 'mc_amount3', 0, GetterInterface::FLOAT ) ) {
			if ( $paymentBasket->get( 'period1', null, GetterInterface::STRING ) ) {
				$amount		=	sprintf( '%.2f', $paymentBasket->get( 'mc_amount1', 0, GetterInterface::FLOAT ) );
			} else {
				$amount		=	sprintf( '%.2f', $paymentBasket->get( 'mc_amount3', 0, GetterInterface::FLOAT ) );
			}
		} else {
			$amount			=	sprintf( '%.2f', $paymentBasket->get( 'mc_gross', 0, GetterInterface::FLOAT ) );
		}

		// https://developer.paypal.com/docs/classic/payflow/integration-guide/#integrating-the-secure-token-without-the-hosted-checkout-pages---transparent-redirect
		$requestParams						=	array(	'TENDER'			=>	'C',
														'TRXTYPE'			=>	'A',
														'CREATESECURETOKEN'	=>	'Y',
														'SECURETOKENID'		=>	uniqid(),
														'SILENTTRAN'		=>	'TRUE',
														'AMT'				=>	$amount,
														'CURRENCY'			=>	$paymentBasket->get( 'mc_currency', null, GetterInterface::STRING ),
														'ORDERID'			=>	$paymentBasket->get( 'item_number', null, GetterInterface::STRING ),
														'INVNUM'			=>	$paymentBasket->get( 'invoice', null, GetterInterface::STRING ),
														'COMMENT1'			=>	$paymentBasket->get( 'item_name', null, GetterInterface::STRING ),
														'USER1'				=>	$paymentBasket->get( 'id', 0, GetterInterface::INT ),
														'RETURNURL'			=>	$this->_getPayNowUrl( $paymentBasket, array( 'paymenttype' => ( $recurring ? 2 : 1 ), 'shopuser' => $this->shopuserParam( $paymentBasket ), 'cardtype' => $cardType ) ),
														'CANCELURL'			=>	$this->getCancelUrl( $paymentBasket ),
														'ERRORURL'			=>	$this->cbsubsGatewayUrl( 'payform', null, $paymentBasket, array( 'shopuser' => $this->shopuserParam( $paymentBasket ), 'cardtype' => $cardType ), false ),
														'BUTTONSOURCE'		=>	'Joomlapolis_Cart_WPS' // our bn code per email of Greg Campagnolo of 21 avril 2012 02:39:24 and 1 mai 2012 21:33:22 and reminder of Rickard of 16.3.2015
													);

		if ( $recurring ) {
			$requestParams['RECURRING']		=	'Y';
		}

		$signedParams						=	$this->_signRequestParams( $requestParams );
		$formUrl							=	array();

		foreach ( $signedParams as $k => $v ) {
			$formUrl[$k]					=	$k . '[' . strlen( $v ) . ']=' . $v;
		}

		$formUrl							=	implode( '&', $formUrl );

		$results							=	array();
		$response							=	null;
		$status								=	null;
		$error								=	$this->httpsRequestGuzzle( $this->gatewayUrl( 'payflow' ), $formUrl, 105, $response, $status, 'post', 'normal' );

		if ( $response ) {
			parse_str( $response, $results );
		}

		$return								=	null;

		if ( $error || ( $status != 200 ) || ( ! $response ) ) {
			$this->_setLogErrorMSG( 3, null, $this->getPayName() . ' HTTPS POST request to payment gateway server failed.', CBTxt::T( "Submitted subscription payment didn't return an error but didn't complete." ) . ' ' . CBTxt::T( 'Please contact site administrator to check error log.' ) );
			return null;
		} else {
			if ( cbGetParam( $results, 'RESULT' ) == '0' ) {
				$return						=	'<input type="hidden" name="SECURETOKEN" value="' . htmlspecialchars( cbGetParam( $results, 'SECURETOKEN' ) ) . '" class="' . ( $recurring ? 'paypalTokenRecurring' : 'paypalTokenSingle' ) . '" />'
											.	'<input type="hidden" name="SECURETOKENID" value="' . htmlspecialchars( cbGetParam( $results, 'SECURETOKENID' ) ) . '" class="' . ( $recurring ? 'paypalTokenRecurring' : 'paypalTokenSingle' ) . '" />';
			} else{
				$this->_setLogErrorMSG( 3, null, $this->getPayName() . ' Paypal Payflow error returned. ERROR: ' . cbGetParam( $results, 'RESPMSG' ), CBTxt::T( 'Please contact site administrator to check error log.' ) );
				return null;
			}
		}

		return $return;
	}

	/**
	 * perform anti fraud checks on ipn values
	 *
	 * @param cbpaidPaymentNotification $ipn
	 * @param cbpaidPaymentBasket $paymentBasket
	 * @param string $cbpid
	 * @return bool|string
	 */
	private function _validateIPN( $ipn, $paymentBasket, $cbpid )
	{
		global $_CB_database;

		$matching						=	true;

		if ( in_array( $ipn->payment_status, array( 'Completed', 'Processed', 'Canceled_Reversal' ) ) ) {
			if ( in_array( $ipn->txn_type, array( 'subscr_payment', 'subscr_signup' ) ) ) {
				$payments				=	$paymentBasket->getPaymentsTotals( $ipn->txn_id );

				if ( ( $paymentBasket->mc_amount1 != 0 ) && ( $payments->count == 0 ) ) {
					$amount				=	$paymentBasket->mc_amount1;
				} else {
					$amount				=	$paymentBasket->mc_amount3;
				}

				if ( sprintf( '%.2f', $ipn->mc_gross ) != sprintf( '%.2f', $amount ) ) {
					if ( ( sprintf( '%.2f', $ipn->mc_gross ) < sprintf( '%.2f', $amount ) ) || ( sprintf( '%.2f', ( $ipn->mc_gross - $ipn->tax ) ) != sprintf( '%.2f', $amount ) ) ) {
						if ( ( ! ( ( $paymentBasket->mc_amount1 != 0 ) && ( $payments->count == 0 ) ) ) && ( ( (float) sprintf( '%.2f', ( $ipn->mc_gross - abs( $ipn->tax ) ) ) ) < ( (float) sprintf( '%.2f', $amount ) ) ) ) {
							$matching	=	CBTxt::T( 'amount mismatch on recurring_payment: amount: [amount] != IPN mc_gross: [gross] or IPN mc_gross - IPN tax: [net] where IPN tax = [tax]', null, array( '[amount]' => $amount, '[net]' => ( $ipn->mc_gross - $ipn->tax ), '[gross]' => $ipn->mc_gross, '[tax]' => $ipn->tax ) );
						}
					}
				}
			} else {
				if ( sprintf( '%.2f', $ipn->mc_gross ) != sprintf( '%.2f', $paymentBasket->mc_gross ) ) {
					if ( ( sprintf( '%.2f', $ipn->mc_gross ) < sprintf( '%.2f', $paymentBasket->mc_gross ) ) || ( sprintf( '%.2f', $ipn->mc_gross - $ipn->tax ) != sprintf( '%.2f', $paymentBasket->mc_gross ) ) ) {
						$matching		=	CBTxt::T( 'amount mismatch on webaccept: BASKET mc_gross: [basket_gross] != IPN mc_gross: [gross] or IPN mc_gross - IPN tax: [net] where IPN tax = [tax]', null, array( '[basket_gross]' => $paymentBasket->mc_gross, '[net]' => ( $ipn->mc_gross - $ipn->tax ), '[gross]' => $ipn->mc_gross, '[tax]' => $ipn->tax ) );
					}
				}
			}
		}

		if ( in_array( $ipn->txn_type, array( 'subscr_payment', 'subscr_signup', 'subscr_cancel', 'subscr_eot', 'subscr_failed' ) ) ) {
			if ( ! $paymentBasket->isAnyAutoRecurring() ) {
				$matching				=	CBTxt::T( 'paypal subscription IPN type [txn_type] for a basket without auto-recurring items', null, array( '[txn_type]' => $ipn->translatedTransactionType() ) );
			}
		}

		if ( ! in_array( $ipn->txn_type, array( 'subscr_signup', 'subscr_cancel', 'subscr_eot', 'subscr_failed' ) ) ) {
			if ( ( $ipn->txn_id === '' ) || ( $ipn->txn_id === 0 ) || ( $ipn->txn_id === null ) ) {
				$matching				=	CBTxt::T( 'illegal transaction id' );
			} else {
				$countBaskets			=	$paymentBasket->countRows( "txn_id = '" . $_CB_database->getEscaped( $ipn->txn_id ) . "' AND payment_status = 'Completed'" );

				if ( ( $countBaskets == 1 ) && ( $paymentBasket->txn_id != $ipn->txn_id ) || ( $countBaskets > 1 ) ) {
					$matching			=	CBTxt::T( 'transaction already used for [count] other already completed payment(s)', null, array( '[count]' => $countBaskets ) );
				}
			}
		}

		if ( $cbpid && ( $cbpid != $paymentBasket->shared_secret ) ) {
			$matching					=	CBTxt::T( 'shared secret [cbpid] returned by Paypal does not match the value we expected', null, array( '[cbpid]' => htmlspecialchars( $cbpid ) ) );
		}

		return $matching;
	}

	/**
	 * Logs payment notification
	 *
	 * @param string $logType
	 * @param int $now
	 * @param cbpaidPaymentBasket $paymentBasket
	 * @param array|null $card
	 * @param array $request
	 * @param null|string $response
	 * @param array $results
	 * @param null|string|array $return
	 * @return cbpaidPaymentNotification
	 */
	private function _logNotification( $logType, $now, $paymentBasket, $card, $request, $response, $results, $return )
	{
		$paymentType											=	( $card && isset( $card['type'] ) ? $card['type'] . ' Credit Card' : 'Credit Card' );

		if ( is_string( $return ) ) {
			$transactionId										=	$return;
			$paymentStatus										=	'Completed';
			$reason												=	null;
			$rawResult											=	'SUCCESS';

			if ( $logType == '5' ) {
				$paymentStatus									=	'Unsubscribed';
			}
		} else {
			$transactionId										=	null;
			$paymentStatus										=	'Denied';
			$reason												=	( isset( $return['errorCode'] ) ? $return['errorCode'] : null );
			$rawResult											=	'FAILED';
		}

		$ipn													=	$this->_prepareIpn( $logType, $paymentStatus, $paymentType, $reason, $now, 'utf-8' );

		$ipn->bindBasket( $paymentBasket );

		$ipn->user_id											=	(int) $paymentBasket->user_id;

		if ( $this->getAccountParam( 'paypal_api' ) == 2 ) {
			$ipn->auth_id										=	cbGetParam( $results, 'AUTHCODE' );
		} else {
			$ipn->auth_id										=	cbGetParam( $results, 'CORRELATIONID' );
		}

		if ( isset( $request['ACCT'] ) ) {
			$request['ACCT']									=	'XXXX XXXX XXXX ' . substr( $request['ACCT'], -4, 4 );
		}

		if ( isset( $request['CVV2'] ) ) {
			$request['CVV2']									=	'XXX';
		}

		if ( isset( $request['USER'] ) ) {
			unset( $request['USER'] );
		}

		if ( isset( $request['PWD'] ) ) {
			unset( $request['PWD'] );
		}

		if ( isset( $request['SIGNATURE'] ) ) {
			unset( $request['SIGNATURE'] );
		}

		$legalCCStore											=	/* cbGetParam not needed, we want raw log here! */ $_POST;

		if ( isset( $legalCCStore[$this->_getPagingParamName('number')] ) ) {
			$legalCCStore[$this->_getPagingParamName('number')]	=	'XXXX XXXX XXXX ' . substr( trim( $legalCCStore[$this->_getPagingParamName('number')] ), -4, 4 );
		}

		if ( isset( $legalCCStore[$this->_getPagingParamName('cvv')] ) ) {
			$legalCCStore[$this->_getPagingParamName('cvv')]	= 'XXX';
		}

		if ( $card && isset( $card['firstname'] ) && isset( $card['lastname'] ) && isset( $legalCCStore[$this->_getPagingParamName('number')]  ) ) {
			$ipn->setPayerNameId( $card['firstname'], $card['lastname'], $legalCCStore[$this->_getPagingParamName('number')] );
		} else {
			$ipn->setPayerNameId( $paymentBasket->first_name, $paymentBasket->last_name );
		}

		$ipn->setRawResult( $rawResult );

		$rawData												=	'$response="' . preg_replace( '/([^\s]{100})/', '$1 ', $response ) . "\"\n"
																.	'$results=' . var_export( $results, true ) . ";\n"
																.	'$return=' . var_export( $return, true ) . ";\n"
																.	'$request=' . var_export( $request, true ) . ";\n"
																.	'$_POST=' . var_export( $legalCCStore, true ) . ";\n";

		$ipn->setRawData( $rawData );

		if ( in_array( $logType, array( 'P', 'B', 'Q', 'V', 'X' ) ) ) {
			$ipn->setTxnSingle( $transactionId );
		} elseif ( in_array( $logType, array( 'A', 'C', 'Z', 'U', 'Y', 'W', '5' ) ) ) {
			if ( ! $ipn->txn_id ) {
				if ( $this->getAccountParam( 'paypal_api' ) == 2 ) {
					$ipn->txn_id								=	cbGetParam( $results, 'TRXPNREF' );

					if ( ! $ipn->txn_id ) {
						$ipn->txn_id							=	cbGetParam( $results, 'PNREF' );
					}
				} else {
					$ipn->txn_id								=	cbGetParam( $results, 'TRANSACTIONID' );
				}
			}

			if ( ! $paymentBasket->subscr_id ) {
				$firstPayment									=	true;
			} else {
				$firstPayment									=	false;
			}

			$ipn->setTxnSubscription( $paymentBasket, $transactionId, $now, 1 );

			if ( $logType == '5' ) {
				$ipn->txn_type									=	'subscr_cancel';
			} else {
				if ( $firstPayment ) {
					$ipn->txn_type								=	'subscr_signup';
				}
			}

			if ( $transactionId ) {
				$this->_bindNotificationToBasket( $ipn, $paymentBasket );

				if ( ( $this->getAccountParam( 'paypal_api' ) == 2 ) && ( $ipn->txn_type == 'subscr_signup' ) ) {
					$paymentBasket->scheduleAutoRecurringPayments();
				}
			}
		}

		$ipn->store();

		return $ipn;
	}

	/**
	 * sign payment request $requestParams with api access added to $requestParams array
	 *
	 * @param array $requestParams
	 * @return array
	 */
	private function _signRequestParams( $requestParams )
	{
		if ( $this->hasPaypalApi() ) {
			if ( $this->getAccountParam( 'paypal_api' ) == 2 ) {
				$requestParams['PARTNER']	=	( $this->getAccountParam( 'paypal_payflow_partner' ) ? $this->getAccountParam( 'paypal_payflow_partner' ) : 'PayPal' );
				$requestParams['VENDOR']	=	$this->getAccountParam( 'paypal_payflow_vendor' );
				$requestParams['USER']		=	( $this->getAccountParam( 'paypal_payflow_user' ) ? $this->getAccountParam( 'paypal_payflow_user' ) : $this->getAccountParam( 'paypal_payflow_vendor' ) );
				$requestParams['PWD']		=	$this->getAccountParam( 'paypal_payflow_password' );
			} else {
				$requestParams['VERSION']	=	'97.0'; // October 2012 - https://cms.paypal.com/cms_content/US/en_US/files/developer/PP_NVPAPI_DeveloperGuide.pdf
				$requestParams['USER']		=	$this->getAccountParam( 'paypal_api_username' );
				$requestParams['PWD']		=	$this->getAccountParam( 'paypal_api_password' );
				$requestParams['SIGNATURE']	=	$this->getAccountParam( 'paypal_api_signature' );
			}
		}

		return $requestParams;
	}

	/**
	 * Limits and converts periods to Paypal limits
	 *
	 * @param array $periodTypeArray  ( int $value, string $periodCOde ) : $periodCode: 'Day','Week','Month','Year'
	 * @param int $now                unix timestamp of now
	 * @return array                  same encoding, but limited
	 */
	private function _paypalPeriodsLimits( $periodTypeArray, $now )
	{
		$p				=	$periodTypeArray[0];
		$t				=	$periodTypeArray[1];
		$s				=	$now;

		if ( $t == 'D' ) {
			$s			=	cbpaidTimes::getInstance()->gmStrToTime( '+' . intval( $p ) . ' DAY', $now );

			if ( $this->getAccountParam( 'paypal_api' ) == 2 ) {
				$t		=	'DAYS';

				if ( $p == 7 ) {
					$t	=	'WEEK';
					$p	=	1;
				} elseif ( $p == 14 ) {
					$t	=	'BIWK';
					$p	=	1;
				} elseif ( $p == 15 ) {
					$t	=	'SMMO';
					$p	=	1;
				} elseif ( $p == 28 ) {
					$t	=	'FRWK';
					$p	=	1;
				} elseif ( $p == 30 ) {
					$t	=	'MONT';
					$p	=	1;
				} elseif ( $p == 90 ) {
					$t	=	'QTER';
					$p	=	1;
				} elseif ( $p == 180 ) {
					$t	=	'SMYR';
					$p	=	1;
				} elseif ( $p == 365 ) {
					$t	=	'YEAR';
					$p	=	1;
				}
			} else {
				$t		=	'Day';

				if ( $p > 90 ) {
					$t	=	'W';
					$p	=	floor( $p / 7 );
				}
			}
		}

		if ( $t == 'W' ) {
			$s			=	cbpaidTimes::getInstance()->gmStrToTime( '+' . intval( $p ) . ' WEEK', $now );

			if ( $this->getAccountParam( 'paypal_api' ) == 2 ) {
				$t		=	'WEEK';

				if ( $p == 2 ) {
					$t	=	'BIWK';
					$p	=	1;
				} elseif ( $p == 4 ) {
					$t	=	'FRWK';
					$p	=	1;
				}

				if ( $p > 1 ) {
					$t	=	'DAYS';
					$p	=	( $p * 7 );
				}
			} else {
				$t		=	'Week';

				if ( $p > 52 ) {
					$t	=	'M';
					$p	=	floor( $p * 12 / 52 );
				}
			}
		}

		if ( $t == 'M' ) {
			$s			=	cbpaidTimes::getInstance()->gmStrToTime( '+' . intval( $p ) . ' MONTH', $now );

			if ( $this->getAccountParam( 'paypal_api' ) == 2 ) {
				$t		=	'MONT';

				if ( $p == 3 ) {
					$t	=	'QTER';
					$p	=	1;
				} elseif ( $p == 6 ) {
					$t	=	'SMYR';
					$p	=	1;
				} elseif ( $p == 12 ) {
					$t	=	'YEAR';
					$p	=	1;
				}

				if ( $p > 1 ) {
					$t	=	'DAYS';
					$p	=	( $p * 30 );
				}
			} else {
				$t		=	'Month';

				if ( $p > 24 ) {
					$t	=	'Y';
					$p	=	floor( $p / 12 );
				}
			}
		}

		if ( $t == 'Y' ) {
			$s			=	cbpaidTimes::getInstance()->gmStrToTime( '+' . intval( $p ) . ' YEAR', $now );

			if ( $this->getAccountParam( 'paypal_api' ) == 2 ) {
				$t		=	'YEAR';

				if ( $p > 1 ) {
					$t	=	'DAYS';
					$p	=	( $p * 365 );
				}
			} else {
				$t		=	'Year';
			}
		}

		return array( $p, $t, $s );
	}

	/**
	 * Utility for gateways to get the payment gateway URL without https:// out of $this->serverUrls array
	 * - depends on $case
	 * - depends on 'normal_gateway' account-param: 0 = test, 1 = normal, 2 = special url in 'gateway_$case_url' account-param
	 *
	 * @param  string  $case   Must be safe ! 'single', 'recurring' or any other case, from constant, not request
	 * @return string          URL with HTTPS://
	 */
	protected function gatewayUrl( $case = 'single' ) {
		if ( ( $case == 'psp' ) && ( $this->getAccountParam( 'paypal_api' ) == 2 ) ) {
			$case	=	'payflow';
		}

		return parent::gatewayUrl( $case );
	}

	/**
	 * FUNCTIONS FOR BACKEND INTERFACE:
	 */

	/**
	 * Renders URL to set in the gateway interface for notifications
	 *
	 * @param string $urlType
	 * @return string
	 */
	public function adminUrlRender( $urlType )
	{
		switch ( $urlType ) {
			case 'successurl':
				return $this->getSuccessUrl( null );
				break;
			case 'cancelurl':
				return $this->getCancelUrl( null );
				break;
			case 'notifyurl':
				return $this->getNotifyUrl( null );
				break;
			default:
		}

		return 'Error: Unknown url type: ' . htmlspecialchars( $urlType );
	}
}

/**
 * Payment account class for this gateway: Stores the settings for that gateway instance, and is used when editing and storing gateway parameters in the backend.
 *
 * OEM base
 * No methods need to be implemented or overriden in this class, except to implement the private-type params used specifically for this gateway:
 */
class cbpaidGatewayAccountpaypalprooem extends cbpaidGatewayAccountCreditCards
{
	/**
	 * USED by XML interface ONLY !!! Renders URL for successful returns
	 *
	 * @param  string              $value   Variable value ( 'successurl', 'cancelurl', 'notifyurl' )
	 * @param  ParamsInterface     $params
	 * @param  string              $name    The name of the form element
	 * @param  CBSimpleXMLElement  $node    The xml element for the parameter
	 * @return string                       HTML to display
	 */
	public function renderUrl( /** @noinspection PhpUnusedParameterInspection */ $value, $params, $name, $node )
	{
		/** @noinspection PhpUndefinedMethodInspection */
		return str_replace( 'http://', 'https://', $this->getPayMean()->adminUrlRender( $node->attributes( 'value' ) ) );
	}
}

/**
 * Payment handler class for this gateway: Handles all payment events and notifications, called by the parent class:
 *
 * Gateway-specific
 * Please note that except the constructor and the API version this class does not implement any public methods.
 */
class cbpaidpaypalpro extends cbpaidpaypalprooem
{
}

/**
 * Payment account class for this gateway: Stores the settings for that gateway instance, and is used when editing and storing gateway parameters in the backend.
 *
 *
 * Gateway-specific
 * No methods need to be implemented or overriden in this class, except to implement the private-type params used specifically for this gateway:
 */
class cbpaidGatewayAccountpaypalpro extends cbpaidGatewayAccountpaypalprooem
{
}
