<?php
/**
* @version $Id: cbpaidsubscriptions.moneriseselectplus.php 1581 2012-12-24 02:36:44Z beat $
* @package CBSubs (TM) Community Builder Plugin for Paid Subscriptions (TM)
* @subpackage Plugin for Paid Subscriptions
* @copyright (C) 2007-2022 and Trademark of Lightning MultiCom SA, Switzerland - www.joomlapolis.com - and its licensors, all rights reserved
* @license http://www.gnu.org/licenses/old-licenses/gpl-2.0.html GNU/GPL version 2
*/

use CBLib\Xml\SimpleXMLElement;
use CBLib\Language\CBTxt;

/** ensure this file is being included by a parent file */
if ( ! ( defined( '_VALID_CB' ) || defined( '_JEXEC' ) || defined( '_VALID_MOS' ) ) ) { die( 'Direct Access to this location is not allowed.' ); }

global $_CB_framework;

// avoids errors in plugin edit:
/** @noinspection PhpIncludeInspection */
include_once( $_CB_framework->getCfg( 'absolute_path' ) . '/components/com_comprofiler/plugin/user/plug_cbpaidsubscriptions/cbpaidsubscriptions.class.php' );

// This gateway implements a payment handler using a hosted page at the PSP:
// Import class cbpaidHostedPagePayHandler that extends cbpaidPayHandler
// and implements all gateway-generic CBSubs methods.

/**
 * Payment handler class for this gateway: Handles all payment events and notifications, called by the parent class:
 *
 * OEM base
 * Please note that except the constructor and the API version this class does not implement any public methods.
 */
class cbpaidmoneriseselectplusoem extends cbpaidHostedPagePayHandler
{
	/**
	 * Gateway API version used
	 * @var int
	 */
	public $gatewayApiVersion	=	"1.3.0";

	/**
	 * Constructor
	 *
	 * @param cbpaidGatewayAccount $account
	 */
	public function __construct( $account )
	{
		parent::__construct( $account );

		$this->_gatewayUrls	=	array(	'psp+normal' => $this->getAccountParam( 'psp_normal_url' ),
										'psp+test' => $this->getAccountParam( 'psp_test_url' ) );
	}

	/**
	 * CBSUBS HOSTED PAGE PAYMENT API METHODS:
	 */

	/**
	 * Returns single payment request parameters for gateway depending on basket (without specifying payment type)
	 *
	 * @param  cbpaidPaymentBasket  $paymentBasket   paymentBasket object
	 * @return array                                 Returns array $requestParams
	 */
	protected function getSinglePaymentRequstParams( $paymentBasket )
	{
		// build hidden form fields or redirect to gateway url parameters array:
		$requestParams						=	$this->_getBasicRequstParams( $paymentBasket );
		// sign single payment params:
		$this->_signRequestParams( $requestParams, 'single' );

		return $requestParams;
	}

	/**
	 * Optional function: only needed for recurring payments:
	 * Returns subscription request parameters for gateway depending on basket (without specifying payment type)
	 *
	 * @param  cbpaidPaymentBasket  $paymentBasket   paymentBasket object
	 * @return array                                 Returns array $requestParams
	 */
	protected function getSubscriptionRequstParams( $paymentBasket )
	{
		// mandatory parameters:
		$requestParams											=	$this->_getBasicRequstParams( $paymentBasket );

		// check for subscription or if single payment:
		if ( $paymentBasket->period3 ) {
			// calculate duration and type:
			list( $duration1, $type1, $start, $recurr1 )		=	$this->_periodsLimits( explode( ' ', $paymentBasket->period3 ), $paymentBasket->recur_times );

			// determine if there is a trial (initial) period:
			if ( $paymentBasket->period1 ) {
				list( /* $duration2 */, /* $type2 */, $start, /* $recurr2 */ )	=	$this->_periodsLimits( explode( ' ', $paymentBasket->period1 ), $paymentBasket->recur_times );

				$requestParams['charge_total']					=	sprintf( '%.2f', $paymentBasket->mc_amount1 );
			}

			// add subscription specific parameters:
			$requestParams['doRecur']							=	1;
			$requestParams['recurUnit']							=	$type1;
			$requestParams['recurPeriod']						=	$duration1;
			$requestParams['recurStartDate']					=	date( 'Y/m/d', ( strtotime( $paymentBasket->subscr_date ) + $start ) );
			$requestParams['recurStartNow']						=	'true';
			$requestParams['recurAmount']						=	sprintf( '%.2f', $paymentBasket->mc_amount3 );
			$requestParams['recurNum']							=	$recurr1;
		}

		return $requestParams;
	}

	/**
	 * The user got redirected back from the payment service provider with a success message: Let's see how successfull it was
	 *
	 * @param  cbpaidPaymentBasket  $paymentBasket  New empty object. returning: includes the id of the payment basket of this callback (strictly verified, otherwise untouched)
	 * @param  array                $postdata       _POST data for saving edited tab content as generated with getEditTab
	 * @return string                               HTML to display if frontend, FALSE if XML error (and not yet ErrorMSG generated), or NULL if nothing to display
	 */
	protected function handleReturn( $paymentBasket, $postdata )
	{
		return $this->_returnParamsHandler( $paymentBasket, $this->_fetchResponse( $postdata ), 'R' );
	}

	/**
	 * The user cancelled his payment
	 *
	 * @param  cbpaidPaymentBasket  $paymentBasket  New empty object. returning: includes the id of the payment basket of this callback (strictly verified, otherwise untouched)
	 * @param  array                $postdata       _POST data for saving edited tab content as generated with getEditTab
	 * @return string                               HTML to display, FALSE if registration cancelled and ErrorMSG generated, or NULL if nothing to display
	 */
	protected function handleCancel( $paymentBasket, $postdata )
	{
		$requestdata							=	$this->_fetchResponse( $postdata );

		if ( $this->hashPdtBackCheck( $this->_getReqParam( 'pdtback' ) ) || $this->_pspVerifySignature( $requestdata ) ) {
			// check if cancel was from site:
			$paymentBasketId					=	(int) $this->_getReqParam( 'basket' );

			// check if cancel was from gateway:
			if ( ! $paymentBasketId ) {
				$paymentBasketId				=	(int) str_replace( $this->getAccountParam( 'pspprefix' ), '', cbGetParam( $requestdata, 'response_order_id', null ) );
			}

			$exists								=	$paymentBasket->load( (int) $paymentBasketId );

			if ( $exists && ( $this->_getReqParam( 'id' ) == $paymentBasket->shared_secret ) && ( $paymentBasket->payment_status != 'Completed' ) ) {
				$paymentBasket->payment_status	=	'RegistrationCancelled';
				$message						=	stripslashes( cbGetParam( $requestdata, 'message' ) );

				if ( ! $message ) {
					$message					=	'Payment cancelled.';
				}

				$this->_setErrorMSG( CBTxt::T( $message ) );
			}
		}
		return false;
	}

	/**
	 * The payment service provider server did a server-to-server notification: Verify and handle it here:
	 *
	 * @param  cbpaidPaymentBasket  $paymentBasket  New empty object. returning: includes the id of the payment basket of this callback (strictly verified, otherwise untouched)
	 * @param  array                $postdata       _POST data for saving edited tab content as generated with getEditTab
	 * @return string                              Text to return to gateway if notification, or NULL if nothing to display
	 */
	protected function handleNotification( $paymentBasket, $postdata )
	{
		return $this->_returnParamsHandler( $paymentBasket, $this->_fetchResponse( $postdata ), 'I' );
	}
	/**
	 * Attempts to validate a successful recurring payment
	 *
	 * @param  cbpaidPaymentBasket  $paymentBasket
	 * @param  string               $returnText                  RETURN param
	 * @param  boolean              $transientErrorDoReschedule  RETURN param
	 * @return boolean              TRUE: succes, FALSE: failed or unknown result
	 */
	protected function processScheduledAutoRecurringPayment( $paymentBasket, &$returnText, &$transientErrorDoReschedule )
	{
		// TODO implement processing of auto recurring payments (payment validation):
		$user											=	CBuser::getUserDataInstance( $paymentBasket->user_id );
		$username										=	( $user ? $user->username : '?' );
		$returnText										=	sprintf( CBTxt::T( 'FAILED (Error) Auto-recurring payment of %s for basket %s Order_id %s of %s (username %s - user id %s) using %s due to error %s.' ), $paymentBasket->renderPrice( null, null, null, false ), $paymentBasket->id, $paymentBasket->sale_id, $paymentBasket->first_name . ' ' . $paymentBasket->last_name, $username, $paymentBasket->user_id, $this->getPayName(), CBTxt::T( 'Recurring subscriptions not yet supported.' ) );
		$transientErrorDoReschedule						=	false; // always cancel recurring as not yet implemented
		return false; // always fail subscription as not yet implemented
/* NOT IMPLEMENTED: THIS CODE IS NOT CORRECT:
		// send API request to validate subscription payment:
		$request										=	'<?xml version="1.0" encoding="UTF-8"?>'
														.	'<request>'
														.		'<store_id>' . $this->getAccountParam( 'pspstore' ) . '</store_id>'
														.		'<api_token>' . $this->getAccountParam( 'psptoken' ) . '</api_token>'
														.		'<recur_update>'
														.			'<order_id>' . $this->getAccountParam( 'pspprefix' ) . $paymentBasket->id . '</order_id>'
														.			'<cust_id>' . $paymentBasket->user_id . '</cust_id>'
														.			'<terminate>true</terminate>'
														.		'</recur_update>'
														.	'</request>';

		$response										=	null;
		$status											=	null;
		$error											=	$this->_httpsRequest( str_replace( '/HPPDP/index.php', '/gateway2/servlet/MpgRequest', $this->pspUrl( $paymentBasket, true ) ), $request, 45, $response, $status, 'post', 'xml' );

		$user											=	CBuser::getUserDataInstance( $paymentBasket->user_id );
		$username										=	( $user ? $user->username : '?' );

		$return											=	false;
		$transientErrorDoReschedule						=	true;

		if ( $error || ( $status != 200 ) || ( ! $response ) ) {
			$returnText									=	sprintf( CBTxt::T( 'FAILED (Error) Auto-recurring payment of %s for basket %s Order_id %s of %s (username %s - user id %s) using %s due to error %s.' ), $paymentBasket->renderPrice( null, null, null, false ), $paymentBasket->id, $paymentBasket->sale_id, $paymentBasket->first_name . ' ' . $paymentBasket->last_name, $username, $paymentBasket->user_id, $this->getPayName(), 'HTTPS error: ' . $error . ' Status: ' . $status );
		} else {
			$xml_response								=	$this->_responseXmlToArray( new SimpleXMLElement( $response, LIBXML_NONET | ( defined('LIBXML_COMPACT') ? LIBXML_COMPACT : 0 ) ) );

			if ( $xml_response ) {
				$response_code							=	(int) cbGetParam( $xml_response, 'responsecode' );

				if ( $response_code < 50 ) {
					$paymentResult						=	$this->_returnParamsHandler( $paymentBasket, $this->fillinSubscriptionRequstParams( $paymentBasket ), 'A' );

					if ( $paymentResult ) {
						if ( in_array( $paymentBasket->payment_status, array( 'Completed', 'Pending' ) ) ) {
							if ( $paymentBasket->payment_status == 'Completed' ) {
								$returnText				=	sprintf( CBTxt::T( 'Completed Auto-recurring payment of %s for basket %s Order_id %s of %s (username %s - user id %s) using %s with txn_id %s and auth_id %s.' ), $paymentBasket->renderPrice( null, null, null, false ), $paymentBasket->id, $paymentBasket->sale_id, $paymentBasket->first_name . ' ' . $paymentBasket->last_name, $username, $paymentBasket->user_id, $this->getPayName(), $paymentBasket->txn_id, $paymentBasket->auth_id );
							} else {
								$returnText				=	sprintf( CBTxt::T( 'Pending Auto-recurring payment of %s for basket %s Order_id %s of %s (username %s - user id %s) using %s with txn_id %s and auth_id %s for reason: %s.' ), $paymentBasket->renderPrice( null, null, null, false ), $paymentBasket->id, $paymentBasket->sale_id, $paymentBasket->first_name . ' ' . $paymentBasket->last_name, $username, $paymentBasket->user_id, $this->getPayName(), $paymentBasket->txn_id, $paymentBasket->auth_id, $paymentBasket->reason_code );
							}

							$transientErrorDoReschedule	=	false;
							$return						=	true;
						} else {
							$returnText					=	sprintf( CBTxt::T( 'FAILED (%s) Auto-recurring payment of %s for basket %s Order_id %s of %s (username %s - user id %s) using %s due to error %s.' ), $paymentBasket->translatedPaymentStatus(), $paymentBasket->renderPrice( null, null, null, false ), $paymentBasket->id, $paymentBasket->sale_id, $paymentBasket->first_name . ' ' . $paymentBasket->last_name, $username, $paymentBasket->user_id, $this->getPayName(), $paymentBasket->reason_code );
						}
					} else {
						$returnText						=	sprintf( CBTxt::T( 'FAILED (Error) Auto-recurring payment of %s for basket %s Order_id %s of %s (username %s - user id %s) using %s due to error %s.' ), $paymentBasket->renderPrice( null, null, null, false ), $paymentBasket->id, $paymentBasket->sale_id, $paymentBasket->first_name . ' ' . $paymentBasket->last_name, $username, $paymentBasket->user_id, $this->getPayName(), $paymentBasket->reason_code );
					}
				} else {
					$returnText							=	sprintf( CBTxt::T( 'FAILED (Error) Auto-recurring payment of %s for basket %s Order_id %s of %s (username %s - user id %s) using %s due to error %s.' ), $paymentBasket->renderPrice( null, null, null, false ), $paymentBasket->id, $paymentBasket->sale_id, $paymentBasket->first_name . ' ' . $paymentBasket->last_name, $username, $paymentBasket->user_id, $this->getPayName(), cbGetParam( $xml_response, 'message', null ) );
				}
			} else {
				$returnText								=	sprintf( CBTxt::T( 'FAILED (Error) Auto-recurring payment of %s for basket %s Order_id %s of %s (username %s - user id %s) using %s due to error %s.' ), $paymentBasket->renderPrice( null, null, null, false ), $paymentBasket->id, $paymentBasket->sale_id, $paymentBasket->first_name . ' ' . $paymentBasket->last_name, $username, $paymentBasket->user_id, $this->getPayName(), CBTxt::T( 'XML response failed to parse' ) );
			}
		}

		return $return;
*/
	}

	/**
	* Cancels an existing recurring subscription
	*
	 * @param  cbpaidPaymentBasket  $paymentBasket  paymentBasket object
	 * @param  cbpaidPaymentItem[]  $paymentItems   redirect immediately instead of returning HTML for output
	 * @return boolean|string                       TRUE if unsubscription done successfully, STRING if error
	*/
	protected function handleStopPaymentSubscription( $paymentBasket, $paymentItems )
	{
		global $_CB_framework;

		$return									=	false;

		// only for recurring subscriptions:
		if ( $paymentBasket->mc_amount3 ) {
			$subscription_id					=	$paymentBasket->subscr_id;

			// only if an actual subscription id exists:
			if ( $subscription_id ) {
				// send API request to cancel subscription:
				$request						=	'<?xml version="1.0" encoding="UTF-8"?>'
												.	'<request>'
												.		'<store_id>' . $this->getAccountParam( 'pspstore' ) . '</store_id>'
												.		'<api_token>' . $this->getAccountParam( 'psptoken' ) . '</api_token>'
												.		'<recur_update>'
												.			'<order_id>' . $this->getAccountParam( 'pspprefix' ) . $paymentBasket->id . '</order_id>'
												.			'<cust_id>' . $paymentBasket->user_id . '</cust_id>'
												.			'<terminate>true</terminate>'
												.		'</recur_update>'
												.	'</request>';

				$response						=	null;
				$status							=	null;
				$error							=	$this->_httpsRequest( str_replace( '/HPPDP/index.php', '/gateway2/servlet/MpgRequest', $this->pspUrl( $paymentBasket, true ) ), $request, 45, $response, $status, 'post', 'xml' );

				if ( $error || ( $status != 200 ) || ( ! $response ) ) {
					$this->_setLogErrorMSG( 3, null, $this->getPayName() . ': unsubscribe API response failed', CBTxt::T( 'Submitted unsubscription failed on-site.' ) . ' ' . CBTxt::T( 'Please contact site administrator to check error log.' ) );
				} else {
					$xml_response				=	$this->_responseXmlToArray( new SimpleXMLElement( $response, LIBXML_NONET | ( defined('LIBXML_COMPACT') ? LIBXML_COMPACT : 0 ) ) );

					if ( $xml_response ) {
						$response_code			=	(int) cbGetParam( $xml_response, 'responsecode' );

						if ( $response_code < 50 ) {
							$ipn				=	$this->_prepareIpn( 'R', $paymentBasket->payment_status, $paymentBasket->payment_type, 'Unsubscribe', $_CB_framework->now(), 'utf-8' );
							$ipn->test_ipn		=	( $this->getAccountParam( 'normal_gateway' ) == '0' ? 1 : 0 );
							$ipn->raw_result	=	'SUCCESS';
							$ipn->raw_data		=	'$message_type="STOP_PAYMENT_SUBSCRIPTION"' . ";\n"
												.	/* cbGetParam() not needed: we want raw info */ '$xml_response=' . var_export( $xml_response, true ) . ";\n"
												.	/* cbGetParam() not needed: we want raw info */ '$_GET=' . var_export( $_GET, true ) . ";\n"
												.	/* cbGetParam() not needed: we want raw info */ '$_POST=' . var_export( $_POST, true ) . ";\n";

							$ipn->bindBasket( $paymentBasket );

							$bskToIpn			=	array(	'sale_id' => 'sale_id',
															'txn_id' => 'txn_id',
															'subscr_id' => 'subscr_id'
														);

							foreach ( $bskToIpn as $k => $v ) {
								$ipn->$k		=	$paymentBasket->$v;
							}

							$ipn->txn_type		=	'subscr_cancel';

							if( ! $ipn->store() ) {
								$this->_setLogErrorMSG( 3, null, $this->getPayName() . ': unsubscribe IPN failed to store', CBTxt::T( 'Submitted unsubscription failed on-site.' ) . ' ' . CBTxt::T( 'Please contact site administrator to check error log.' ) );
							} else {
								$return			=	true;
							}
						} else {
							$this->_setLogErrorMSG( 3, null, $this->getPayName() . ': unsubscribe API reponse not completed: ' . cbGetParam( $xml_response, 'message', null ), CBTxt::T( 'Submitted unsubscription failed on-site.' ) . ' ' . CBTxt::T( 'Please contact site administrator to check error log.' ) );
						}
					} else {
						$this->_setLogErrorMSG( 3, null, $this->getPayName() . ': unsubscribe API reponse empty', CBTxt::T( 'Submitted unsubscription failed on-site.' ) . ' ' . CBTxt::T( 'Please contact site administrator to check error log.' ) );
					}
				}
			} else {
				$this->_setLogErrorMSG( 3, null, $this->getPayName() . ': unsubscribe failed from missing subscr_id in payment basket', CBTxt::T( 'Submitted unsubscription failed on-site.' ) . ' ' . CBTxt::T( 'Please contact site administrator to check error log.' ) );
			}
		}

		return $return;
	}


	/**
	 * GATEWAY-INTERNAL SPECIFIC METHODS:
	 */

	/**
	 * generate validation to check on return of PDT or IPN
	 *
	 * @param  array   $requestParams
	 * @return string
	 */
	private function _pspSignMD5( $requestParams )
	{
		$signParams				=	array();

		// parse parameters into all uppercase associate array:
		foreach ( $requestParams as $k => $v ) {
			if ( ( $v !== '' ) && ( $v !== null ) ) {
				$k				=	strtoupper( $k );
				$signParams[$k]	=	$v;
			}
		}

		$string					=	null;

		// add item code (basket id):
		if ( isset( $signParams['ORDER_ID'] ) ) {
			$string				.=	$signParams['ORDER_ID'];
		} elseif ( isset( $signParams['RESPONSE_ORDER_ID'] ) ) {
			$string				.=	$signParams['RESPONSE_ORDER_ID'];
		}

		// add purchase price (10.01, etc..):
		if ( isset( $signParams['CHARGE_TOTAL'] ) ) {
			$string				.=	$signParams['CHARGE_TOTAL'];
		}

		// add subscription type (purchase or subscription):
		if ( isset( $signParams['RECUR_RESULT'] ) && ( (bool) $signParams['RECUR_RESULT'] === true ) ) {
			$string				.=	'subscription';
		} else {
			$string				.=	'purchase';
		}

		// add security code generated at gateway:
		$string					.=	$this->getAccountParam( 'pspkey' );

		// convert to MD5 and uppercase entire string:
		return strtoupper( md5( $string ) );
	}

	/**
	 * sign parameters with validation code
	 *
	 * @param array $requestParams
	 */
	private function _signRequestParams( &$requestParams )
	{
		// add validation code:
		$requestParams['rvar_sign']	=	$this->_pspSignMD5( $requestParams );
	}

	/**
	 * validate PDT or IPN using signed validation code sent to gateway
	 *
	 * @param array $requestParams
	 * @param boolean $validateTransaction
	 * @return boolean
	 */
	private function _pspVerifySignature( $requestParams, $validateTransaction = false )
	{
		// new instance of returned data:
		$signParams						=	$requestParams;

		// remove the valiation code:
		unset( $signParams['rvar_sign'] );

		$signParams						=	array();

		// load existing returned data into seperate array cleanly:
		foreach ( $requestParams as $k => $v ) {
			if ( ( $k != 'rvar_sign' ) && ( $v !== '' ) && ( $v !== null ) ) {
				$signParams[$k]			=	stripslashes( cbGetParam( $requestParams, $k, null, _CB_NOTRIM ) );
			}
		}

		// generate validation code for data returned from gateway:
		$string							=	$this->_pspSignMD5( $signParams );

		// confirm validation:
		$valid							=	( $string === cbGetParam( $requestParams, 'rvar_sign' ) ? true : false );

		// validate the the transaction only if signature is valid or contains a key and transaction validation specified:
		if ( $valid && $validateTransaction ) {
			$formvars					=	array();
			$formvars['ps_store_id']	=	$this->getAccountParam( 'pspid' );
			$formvars['hpp_key']		=	$this->getAccountParam( 'pspkey' );
			$formvars['transactionKey']	=	cbGetParam( $requestParams, 'transactionKey' );

			$response					=	null;
			$status						=	null;
			$error						=	$this->_httpsRequest( str_replace( 'index.php', 'verifyTxn.php', $this->pspUrl( null, true ) ), $formvars, 30, $response, $status, 'post', 'normal', '*/*', true, 443, '', '', true, null );

			if ( $error || ( $status != 200 ) || ( ! $response ) ) {
				$valid					=	false;
			} else {
				$xml_response			=	$this->_responseXmlToArray( new SimpleXMLElement( $response, LIBXML_NONET | ( defined('LIBXML_COMPACT') ? LIBXML_COMPACT : 0 ) ) );

				if ( $xml_response ) {
					$response_code		=	(int) cbGetParam( $xml_response, 'response_code' );

					if ( $response_code < 50 ) {
						$valid			=	true;
					} else {
						$valid			=	false;
					}
				} else {
					$valid				=	false;
				}
			}
		}

		return $valid;
	}

	/**
	 * Compute the CBSubs payment_status based on gateway's params:
	 *
	 * STATUS:	Status of the payment:
		Statuses in 1 digit are 'normal' statuses:
		0 means the payment is invalid (e.g. data validation error) or the processing is not complete either because it is still underway, or because the transaction was interrupted. If the cause is a validation error, an additional error code (*) (NCERROR) identifies the error.
		1 means the customer cancelled the transaction.
		2 means the acquirer did not authorise the payment.
		5 means the acquirer autorised the payment.
		9 means the payment was captured.
		Statuses in 2 digits correspond either to 'intermediary' situations or to abnormal events. When the second digit is:
		1, this means the payment processing is on hold.
		2, this means an unrecoverable error occurred during the communication with the acquirer. The result is therefore not determined. You must therefore call the acquirer's helpdesk to find out the actual result of this transaction.
		3, this means the payment processing (capture or cancellation) was refused by the acquirer whilst the payment had been authorised beforehand. It can be due to a technical error or to the expiration of the authorisation. You must therefore call the acquirer's helpdesk to find out the actual result of this transaction.
		4, this means our system has been notified the transaction was rejected well after the transaction was sent to your acquirer.
		5, this means our system hasn't sent the requested transaction to the acquirer since the merchant will send the transaction to the acquirer himself, like he specified in his configuration.
	 *
	 * ACCEPTANCE:
	 * Acquirer's acceptance (authorization) code.
	 * The acquirer sends back this code to confirm the amount of the transaction has been blocked on the card of the customer. The acceptance code is not unique.
	 *
	 * @param  array                $postdata        raw POST data received from the payment gateway
	 * @param  string               $reason          OUT: reason_code
	 * @param  string               $previousStatus  previous CBSubs status
	 * @param  cbpaidPaymentBasket  $paymentBasket   (only for error logging purposes)
	 * @return string
	 */
	private function _paymentStatus( $postdata, &$reason, /** @noinspection PhpUnusedParameterInspection */ $previousStatus, /** @noinspection PhpUnusedParameterInspection */ &$paymentBasket )
	{
		$status					=	cbGetParam( $postdata, 'status' );

		if ( $status ) {
			switch ( $status ) {
				case 'Valid-Approved':
				case 'Invalid-ReConfirmed':
					$reason		=	null;
					$status		=	'Completed';
					break;
				case 'Valid-Declined':
					$reason		=	'The transaction was declined and successfully validated';
					$status		=	'Denied';
					break;
				case 'Invalid':
					$reason		=	'No reference to this transactionKey, validation failed';
					$status		=	'Error';
					break;
				case 'Invalid-Bad_Source':
					$reason		=	'The Referring URL is not correct, validation failed';
					$status		=	'Error';
					break;
			}
		} else {
			$response			=	cbGetParam( $postdata, 'response_code' );

			if ( intval( $response ) < 50 ) {
				$reason			=	null;
				$status			=	'Completed';
			} elseif ( ( intval( $response ) >= 50 ) || ( $response == null ) ) {
				$reason			=	stripslashes( cbGetParam( $postdata, 'message' ) );
				$status			=	'Denied';
			} else {
				$reason			=	'No Response Returned';
				$status			=	'Error';
			}
		}

		return $status;
	}

	/**
	 * Compute the CBSubs payment_type based on gateway's params:
	 *
	 * @param array $postdata raw POST data received from the payment gateway
	 * @return string
	 */
	private function _getPaymentType( $postdata )
	{
		$brand			=	cbGetParam( $postdata, 'card' );
		$type			=	null;

		switch ( $brand ) {
			case 'M':
				$type	=	'Mastercard';
				break;
			case 'V':
				$type	=	'Visa';
				break;
			case 'AX':
				$type	=	'American Express';
				break;
			case 'DC':
				$type	=	'Diners Card';
				break;
			case 'NO':
				$type	=	'Novus / Discover';
				break;
			case 'SE':
				$type	=	'Sears';
				break;
		}

		return ( ( $type ? $type . ' ' : null ) . 'Credit Card' );
	}

	/**
	 * parse period into period and type then calculate new period limitations:
	 *
	 * @param array $periodTypeArray
	 * @param int $recurrTimes
	 * @return array
	 */
	private function _periodsLimits( $periodTypeArray, $recurrTimes = 0 )
	{
		global $_CB_framework;

		// parse period into period and type:
		$p					=	$periodTypeArray[0];
		$t					=	$periodTypeArray[1];
		$now				=	$_CB_framework->now();

		if ( ( $recurrTimes <= 0 ) || ( $recurrTimes > 5 ) ) {
			$recurrTimes	=	5;
		}

		// change single letter type to full length:
		if ( $t == 'Y' ) {
			$t				=	'Month';
			$p				=	( $p * 12 );
			$s				=	cbpaidTimes::getInstance()->gmStrToTime( '+' . intval( $p ) . ' MONTH', $now );
			$r				=	( ( 12 / $p ) * $recurrTimes );
		} elseif ( $t == 'M' ) {
			$t				=	'Month';
			$s				=	cbpaidTimes::getInstance()->gmStrToTime( '+' . intval( $p ) . ' MONTH', $now );
			$r				=	( ( 12 / $p ) * $recurrTimes );
		} elseif  ( $t == 'W' ) {
			$t				=	'Week';
			$s				=	cbpaidTimes::getInstance()->gmStrToTime( '+' . intval( $p ) . ' WEEK', $now );
			$r				=	( ( 52 / $p ) * $recurrTimes );
		} elseif ( $t == 'D' ) {
			$t				=	'Day';
			$s				=	cbpaidTimes::getInstance()->gmStrToTime( '+' . intval( $p ) . ' DAY', $now );
			$r				=	( ( 365 / $p ) * $recurrTimes );
		} else {
			trigger_error( __CLASS__ . '::' . __FUNCTION__ . ' wrong parameters.', E_USER_WARNING );
			$s				=	0;
			$r				=	0;
		}

		return array( $p, $t, $s, $r );
	}

	/**
	* popoulates basic request parameters for gateway depending on basket (without specifying payment type)
	*
	* @param cbpaidPaymentBasket $paymentBasket paymentBasket object
	* @return array $requestParams
	*/
	private function _getBasicRequstParams( $paymentBasket )
	{
		// mandatory parameters:
		$requestParams										=	array();
		$requestParams['ps_store_id']						=	$this->getAccountParam( 'pspid' );
		$requestParams['hpp_key']							=	$this->getAccountParam( 'pspkey' );
		$requestParams['order_id']							=	$this->getAccountParam( 'pspprefix' ) . $paymentBasket->id;
		$requestParams['charge_total']						=	sprintf( '%.2f', $paymentBasket->mc_gross );
		$requestParams['lang']								=	$this->getAccountParam( 'language', 'en-ca' );

		$items												=	$paymentBasket->loadPaymentItems();

		if ( count( $items ) > 0 ) {
			$k												=	1;

			foreach ( $items as $item ) {
				$requestParams['id' . $k]					=	$item->plan_id;
				$requestParams['description' . $k]			=	$item->description;
				$requestParams['quantity' . $k]				=	$item->quantity;
				$requestParams['price' . $k]				=	$item->rate;
				$requestParams['subtotal' . $k]				=	$item->getPrice();

				++$k;
			}
		}

		// courtesy fields (pre-filled but editable on credit card mask):
		$requestParams['cust_id']							=	$paymentBasket->user_id;
		$requestParams['bill_first_name']					=	$paymentBasket->first_name;
		$requestParams['bill_last_name']					=	$paymentBasket->last_name;
		$requestParams['ship_first_name']					=	$paymentBasket->first_name;
		$requestParams['ship_last_name']					=	$paymentBasket->last_name;

		// recommended anti-fraud fields:
		if ( $this->getAccountParam( 'givehiddenemail' ) && ( strlen( $paymentBasket->payer_email ) <= 50 ) ) {
			$requestParams['email']							=	$paymentBasket->payer_email;
		}

		if ( $this->getAccountParam( 'givehiddenbilladdress' ) ) {
			cbimport( 'cb.tabs' ); // needed for cbIsoUtf_substr()

			$addressFields									=	array(	'bill_address_one' => array( $paymentBasket->address_street, 30 ),
																		'bill_postal_code' => array( $paymentBasket->address_zip, 10 ),
																		'bill_city' => array( $paymentBasket->address_city, 30 ),
																		'bill_country' => array( $paymentBasket->getInvoiceCountry( 3 ), 3 ),
																		'bill_state_or_province' => array( $paymentBasket->getInvoiceState(), 30 )
																	);

			foreach ( $addressFields as $k => $value_maxlength ) {
				$adrField									=	cbIsoUtf_substr( $value_maxlength[0], 0, $value_maxlength[1] );

				if ( $adrField ) {
					$requestParams[$k]						=	$adrField;
				}
			}
		}

		if ( $this->getAccountParam( 'givehiddenshipaddress' ) ) {
			cbimport( 'cb.tabs' ); // needed for cbIsoUtf_substr()

			$addressFields									=	array(	'ship_address_one' => array( $paymentBasket->address_street, 30 ),
																		'ship_postal_code' => array( $paymentBasket->address_zip, 10 ),
																		'ship_city' => array( $paymentBasket->address_city, 30 ),
																		'ship_country' => array( $paymentBasket->getInvoiceCountry( 3 ), 3 ),
																		'ship_state_or_province' => array( $paymentBasket->getInvoiceState(), 30 )
																	);

			foreach ( $addressFields as $k => $value_maxlength ) {
				$adrField									=	cbIsoUtf_substr( $value_maxlength[0], 0, $value_maxlength[1] );

				if ( $adrField ) {
					$requestParams[$k]						=	$adrField;
				}
			}
		}

		if ( $this->getAccountParam( 'givehiddenbilltelno' ) && ( strlen( $paymentBasket->contact_phone ) <= 50 ) ) {
			$requestParams['bill_phone']					=	$paymentBasket->contact_phone;
		}

		if ( $this->getAccountParam( 'givehiddenshiptelno' ) && ( strlen( $paymentBasket->contact_phone ) <= 50 ) ) {
			$requestParams['ship_phone']					=	$paymentBasket->contact_phone;
		}

		return $requestParams;
	}

	/**
	 * The user got redirected back from the payment service provider with a success message: let's see how successfull it was
	 *
	 * @param  cbpaidPaymentBasket  $paymentBasket       New empty object. returning: includes the id of the payment basket of this callback (strictly verified, otherwise untouched)
	 * @param  array                $requestdata         Data returned by gateway
	 * @param  string               $type                Type of return ('R' for PDT, 'I' for INS, 'A' for Autorecurring payment (Vault) )
	 * @param  array|null           $additionalLogData   Additional data to log with the notification log
	 * @return string                                    HTML to display if frontend, text to return to gateway if notification, FALSE if registration cancelled and ErrorMSG generated, or NULL if nothing to display
	 */
	private function _returnParamsHandler( $paymentBasket, $requestdata, $type, $additionalLogData = null )
	{
		global $_CB_framework, $_GET, $_POST;

		$ret													=	null;
		$paymentBasketId										=	(int) str_replace( $this->getAccountParam( 'pspprefix' ), '', cbGetParam( $requestdata, 'response_order_id', null ) );

		if ( $paymentBasketId ) {
			$exists												=	$paymentBasket->load( (int) $paymentBasketId );

			if ( $exists && ( ( $type == 'A' ) || ( ( cbGetParam( $requestdata, $this->_getPagingParamName( 'id' ), 0 ) == $paymentBasket->shared_secret ) && ( ! ( ( ( $type == 'R' ) || ( $type == 'I' ) ) && ( $paymentBasket->payment_status == 'Completed' ) ) ) ) ) ) {
				// Log the return record:
				$log_type										=	$type;
				$reason											=	null;
				$paymentStatus									=	$this->_paymentStatus( $requestdata, $reason, $paymentBasket->payment_status, $paymentBasket );
				$paymentType									=	$this->_getPaymentType( $requestdata );
				$paymentTime									=	$_CB_framework->now();

				if ( $paymentStatus == 'Error' ) {
					$errorTypes									=	array( 'I' => 'D', 'R' => 'E', 'A' => 'W' );

					if ( isset( $errorTypes[$type] ) ) {
						$log_type								=	$errorTypes[$type];
					}
				}

				$ipn											=	$this->_prepareIpn( $log_type, $paymentStatus, $paymentType, $reason, $paymentTime, 'utf-8' );

				if ( $paymentStatus == 'Refunded' ) {
					// in case of refund we need to log the payment as it has same TnxId as first payment: so we need payment_date for discrimination:
					$ipn->payment_date							=	gmdate( 'H:i:s M d, Y T', $paymentTime ); // paypal-style
				}

				$ipn->test_ipn									=	( $this->getAccountParam( 'normal_gateway' ) == '0' ? 1 : 0 );
				$ipn->raw_data									=	'$message_type="' . ( $type == 'R' ? 'RETURN_TO_SITE' : ( $type == 'I' ? 'NOTIFICATION' : ( $type == 'A' ? 'AUTORECURRING VAULT PAYMENT' : 'UNKNOWN' ) ) ) . '";' . "\n";

				if ( $additionalLogData ) {
					foreach ( $additionalLogData as $k => $v ) {
						$ipn->raw_data							.=	'$' . $k . '="' . var_export( $v, true ) . '";' . "\n";
					}
				}

				$ipn->raw_data									.=	/* cbGetParam() not needed: we want raw info */ '$requestdata=' . var_export( $requestdata, true ) . ";\n"
																.	/* cbGetParam() not needed: we want raw info */ '$_GET=' . var_export( $_GET, true ) . ";\n"
																.	/* cbGetParam() not needed: we want raw info */ '$_POST=' . var_export( $_POST, true ) . ";\n";

				if ( $paymentStatus == 'Error' ) {
					$paymentBasket->reason_code					=	$reason;

					$this->_storeIpnResult( $ipn, 'ERROR:' . $reason );
					$this->_setLogErrorMSG( 4, $ipn, $this->getPayName() . ': ' . $reason, CBTxt::T( 'Sorry, the payment server replied with an error.' ) . ' ' . CBTxt::T( 'Please contact site administrator to check payment status and error log.' ) );

					$ret										=	false;
				} else {
					$ipn->bindBasket( $paymentBasket );

					$ipn->sale_id								=	$paymentBasketId;

					$insToIpn									=	array(	'txn_id' => 'txn_num',
																			'auth_id' => 'bank_transaction_id',
																			'first_name' => 'bill_first_name',
																			'last_name' => 'bill_last_name',
																			'address_street' => 'bill_address_one',
																			'address_zip' => 'bill_postal_code',
																			'address_city' => 'bill_city',
																			'address_country' => 'bill_country',
																			'address_state' => 'bill_state_or_province',
																			'contact_phone' => 'bill_phone',
																			'payer_email' => 'email',
																			'memo' => 'note'
																		);

					foreach ( $insToIpn as $k => $v ) {
						$ipn->$k								=	cbGetParam( $requestdata, $v );
					}

					// try to build name from credit card if not found:
					if ( ( ! $ipn->first_name ) && ( ! $ipn->last_name ) ) {
						$name									=	cbGetParam( $requestdata, 'cardholder' );

						if ( $name ) {
							$card_names							=	explode( ' ', $name, 2 );

							if ( count( $card_names ) < 2 ) {
								$card_names						=	array( $name, '' );
							}

							$ipn->first_name					=	$card_names[0];
							$ipn->last_name						=	$card_names[1];
						}
					}

					$ipn->mc_gross								=	sprintf( '%.2f', cbGetParam( $requestdata, 'charge_total' ) );
					$ipn->user_id								=	(int) $paymentBasket->user_id;

					// check what type of purchase this is:
					$recurring									=	( ( (bool) cbGetParam( $requestdata, 'recur_result' ) === true ) || ( $type == 'A' ) ? true : false );

					// handle recurring subscriptions properly or default to single payment:
					if ( $recurring ) {
						if ( ( $paymentStatus == 'Completed' ) && ( ! $paymentBasket->subscr_id ) && ( $type != 'A' ) ) {
							$ipn->txn_type						=	'subscr_signup';
							$ipn->subscr_id						=	cbGetParam( $requestdata, 'txn_num' );
							$ipn->subscr_date					=	$ipn->payment_date;
						} elseif ( $paymentStatus == 'Denied' ) {
							if ( ( $paymentBasket->reattempts_tried + 1 ) <= cbpaidScheduler::getInstance( $this )->retries ) {
								$ipn->txn_type					=	'subscr_failed';
							} else {
								$ipn->txn_type					=	'subscr_cancel';
							}
						} elseif ( in_array( $paymentStatus, array( 'Completed', 'Processed', 'Pending' ) ) ) {
							$ipn->txn_type						=	'subscr_payment';
						}
					} else {
						$ipn->txn_type							=	'web_accept';
					}

					// validate payment from PDT or IPN
					if ( ( $type == 'A' ) || $this->_pspVerifySignature( $requestdata, true ) ) {
						if ( ( ( sprintf( '%.2f', $paymentBasket->mc_gross ) == $ipn->mc_gross ) || ( $ipn->payment_status == 'Refunded' ) ) && ( $paymentBasket->mc_currency == $ipn->mc_currency ) ) {
							if ( in_array( $ipn->payment_status, array( 'Completed', 'Processed', 'Pending', 'Refunded', 'Denied' ) ) ) {
								$this->_storeIpnResult( $ipn, 'SUCCESS' );
								$this->_bindIpnToBasket( $ipn, $paymentBasket );

								// add the gateway to the basket:
								$paymentBasket->payment_method	=	$this->getPayName();
								$paymentBasket->gateway_account	=	$this->getAccountParam( 'id' );

								// 0: not auto-recurring, 1: auto-recurring without payment processor notifications, 2: auto-renewing with processor notifications updating $expiry_date:
								$autorecurring_type				=	( in_array( $ipn->txn_type, array( 'subscr_payment', 'subscr_signup', 'subscr_modify', 'subscr_eot', 'subscr_cancel', 'subscr_failed' ) ) ? 2 : 0 );

								// 0: not auto-renewing (manual renewals), 1: asked for by user, 2: mandatory by configuration:
								$autorenew_type					=	( $autorecurring_type ? ( ( $this->getAccountParam( 'enabled', 0 ) == 3 ) && ( $paymentBasket->isAnyAutoRecurring() == 2 ) ? 1 : 2 ) : 0 );

								if ( $recurring ) {
									$paymentBasket->reattempt	=	1; // we want to reattempt auto-recurring payment in case of failure
								}

								$this->updatePaymentStatus( $paymentBasket, $ipn->txn_type, $ipn->payment_status, $ipn, 1, $autorecurring_type, $autorenew_type, false );

								if ( in_array( $ipn->payment_status, array( 'Completed', 'Processed', 'Pending' ) ) ) {
									$ret						=	true;

									if ( $recurring && ( $type != 'A' ) ) {
										// Now we need to schedule the automatic payments to be done by CBSubs using Vault API, and which will trigger the auto-renewals (if it's not already the autorecurring payment (A)):
										$paymentBasket->scheduleAutoRecurringPayments();
									}
								}
							} else {
								$this->_storeIpnResult( $ipn, 'FAILED' );

								$paymentBasket->payment_status	=	$ipn->payment_status;

								$this->_setErrorMSG( '<div class="alert alert-info">' . $this->getTxtNextStep( $paymentBasket ) . '</div>' );

								$paymentBasket->payment_status	=	'RedisplayOriginalBasket';
								$ret							=	false;
							}
						} else {
							$this->_storeIpnResult( $ipn, 'MISMATCH' );
							$this->_setLogErrorMSG( 3, $ipn, $this->getPayName() . ': amount or currency missmatch', CBTxt::T( 'Sorry, the payment does not match the basket.' ) . ' ' . CBTxt::T( 'Please contact site administrator to check error log.' ) );

							$ret								=	false;
						}
					} else {
						$this->_storeIpnResult( $ipn, 'SIGNERROR' );
						$this->_setLogErrorMSG( 3, $ipn, $this->getPayName() . ': rvar_sign or transaction does not match with gateway. Please check HPP Key setting', CBTxt::T( 'The HPP Key signature is incorrect.' ) . ' ' . CBTxt::T( 'Please contact site administrator to check error log.' ) );

						$ret									=	false;
					}
				}
			}
		} else {
			$this->_setLogErrorMSG( 3, null, $this->getPayName() . ': The response_order_id is missing in the return URL', CBTxt::T( 'Please contact site administrator to check error log.' ) );
		}

		return  $ret;
	}

	/**
	 * trys to find a suitable response data
	 *
	 * @param array $postdata
	 * @return array
	 */
	private function _fetchResponse( $postdata )
	{
		if ( ( count( $postdata ) > 0 ) && isset( $postdata['xml_response'] ) ) {
			$requestdata	=	$this->_responseXmlToArray( new SimpleXMLElement( stripslashes( $postdata['xml_response'] ), LIBXML_NONET | ( defined('LIBXML_COMPACT') ? LIBXML_COMPACT : 0 ) ) );
		} elseif ( ( count( $postdata ) > 0 ) && ( isset( $postdata['response_order_id'] ) || isset( $postdata['order_id'] ) || isset( $postdata['rvar_basket'] ) ) ) {
			$requestdata	=	$_POST;
		} else {
			$requestdata	=	$this->_getGetParams();
		}

		return $requestdata;
	}

	/**
	 * parses xml response into associative array
	 *
	 * @param SimpleXMLElement $xml
	 * @return array
	 */
	private function _responseXmlToArray( $xml )
	{
		$requestdata				=	array();

		foreach ( $xml->children() as $child ) {
			/** @var $child SimpleXMLElement */
			$name					=	strtolower( $child->getName() );
			$value					=	trim( $child->data() );

			if ( strlen( $value ) == 0 ) {
				$requestdata		=	array_merge( $requestdata, $this->_responseXmlToArray( $child ) );
			} else {
				$requestdata[$name]	=	$value;
			}
		}

		return $requestdata;
	}
}

/**
 * Payment account class for this gateway: Stores the settings for that gateway instance, and is used when editing and storing gateway parameters in the backend.
 *
 * OEM-basis
 * No methods need to be implemented or overriden in this class, except to implement the private-type params used specifically for this gateway:
 */
class cbpaidGatewayAccountmoneriseselectplusoem extends cbpaidGatewayAccounthostedpage
{
}

/**
 * Payment handler class for this gateway: Handles all payment events and notifications, called by the parent class:
 *
 * Gateway-specific
 * Please note that except the constructor and the API version this class does not implement any public methods.
 */
class cbpaidmoneriseselectplus extends cbpaidmoneriseselectplusoem
{
}

/**
 * Payment account class for this gateway: Stores the settings for that gateway instance, and is used when editing and storing gateway parameters in the backend.
 *
 * Gateway-specific
 * No methods need to be implemented or overriden in this class, except to implement the private-type params used specifically for this gateway:
 */
class cbpaidGatewayAccountmoneriseselectplus extends cbpaidGatewayAccountmoneriseselectplusoem
{
}
