<?php
/**
* @version $Id: cbpaidsubscriptions.ciccreditmutuel.php 1581 2012-12-24 02:36:44Z beat $
* @package CBSubs (TM) Community Builder Plugin for Paid Subscriptions (TM)
* @subpackage Plugin for Paid Subscriptions
* @copyright (C) 2007-2022 and Trademark of Lightning MultiCom SA, Switzerland - www.joomlapolis.com - and its licensors, all rights reserved
* @license http://www.gnu.org/licenses/old-licenses/gpl-2.0.html GNU/GPL version 2
*/

use CBLib\Language\CBTxt;

/** Ensure this file is being included by a parent file */
if ( ! ( defined( '_VALID_CB' ) || defined( '_JEXEC' ) || defined( '_VALID_MOS' ) ) ) { die( 'Direct Access to this location is not allowed.' ); }

global $_CB_framework;

// Avoids errors in CB plugin edit:
/** @noinspection PhpIncludeInspection */
include_once( $_CB_framework->getCfg( 'absolute_path' ) . '/components/com_comprofiler/plugin/user/plug_cbpaidsubscriptions/cbpaidsubscriptions.class.php' );

// This gateway implements a payment handler using a hosted page at the PSP:
// Import class cbpaidHostedPagePayHandler that extends cbpaidPayHandler
// and implements all gateway-generic CBSubs methods.

/**
 * Payment handler class for this gateway: Handles all payment events and notifications, called by the parent class:
 *
 * OEM base
 * Please note that except the constructor and the API version this class does not implement any public methods.
 */
class cbpaidciccreditmutueloem extends cbpaidHostedPagePayHandler
{
	/**
	 * Gateway API version used
	 * @var int
	 */
	public $gatewayApiVersion	=	"1.3.0";

	/**
	 * Constructor
	 *
	 * @param cbpaidGatewayAccount $account
	 */
	public function __construct( $account )
	{
		parent::__construct( $account );

		// Set gateway URLS for $this->pspUrl() results: first 2 are the main hosted payment page posting URL, next ones are gateway-specific:
		$this->_gatewayUrls	=	array(	'psp+normal'	 => $this->getAccountParam( 'psp_normal_url' ),
										'psp+test'		 => $this->getAccountParam( 'psp_test_url' ) );
	}

	/**
	 * CBSUBS HOSTED PAGE PAYMENT API METHODS:
	 */

	/**
	 * Returns single payment request parameters for gateway depending on basket (without specifying payment type)
	 *
	 * @param  cbpaidPaymentBasket  $paymentBasket   paymentBasket object
	 * @return array                                 Returns array $requestParams
	 */
	protected function getSinglePaymentRequstParams( $paymentBasket )
	{
		// build hidden form fields or redirect to gateway url parameters array:
		$requestParams	=	$this->_getBasicRequstParams( $paymentBasket );

		// sign single payment params:
		$this->_signRequestParams( $requestParams );

		return $requestParams;
	}

	/**
	 * Optional function: only needed for recurring payments:
	 * Returns subscription request parameters for gateway depending on basket (without specifying payment type)
	 *
	 * @param  cbpaidPaymentBasket  $paymentBasket   paymentBasket object
	 * @return array                                 Returns array $requestParams
	 */
	protected function getSubscriptionRequstParams( $paymentBasket )
	{
		return $this->getSinglePaymentRequstParams( $paymentBasket );
	}

	/**
	 * The user got redirected back from the payment service provider with a success message: Let's see how successfull it was
	 *
	 * @param  cbpaidPaymentBasket  $paymentBasket  New empty object. returning: includes the id of the payment basket of this callback (strictly verified, otherwise untouched)
	 * @param  array                $postdata       _POST data for saving edited tab content as generated with getEditTab
	 * @return string                               HTML to display if frontend, FALSE if XML error (and not yet ErrorMSG generated), or NULL if nothing to display
	 */
	protected function handleReturn( $paymentBasket, $postdata )
	{
		$reference		=	( (int) cbGetParam( $_GET, 'cbpgacctno', null ) . '-' . (int) cbGetParam( $_GET, 'cbpbasket', null ) );
		$requestdata	=	array( 'reference' => $reference );

		return $this->_returnParamsHandler( $paymentBasket, $requestdata, 'R' );
	}

	/**
	 * The user cancelled his payment
	 *
	 * @param  cbpaidPaymentBasket  $paymentBasket  New empty object. returning: includes the id of the payment basket of this callback (strictly verified, otherwise untouched)
	 * @param  array                $postdata       _POST data for saving edited tab content as generated with getEditTab
	 * @return string                               HTML to display, FALSE if registration cancelled and ErrorMSG generated, or NULL if nothing to display
	 */
	protected function handleCancel( $paymentBasket, $postdata )
	{
		// The user cancelled his payment (and registration):
		if ( $this->hashPdtBackCheck( $this->_getReqParam( 'pdtback' ) ) ) {
			$paymentBasketId					=	(int) $this->_getReqParam( 'basket' );

			// check if cancel was from gateway:
			if ( ! $paymentBasketId ) {
				$paymentBasketId				=	(int) $this->_getBasketId( $postdata );
			}

			$exists								=	$paymentBasket->load( (int) $paymentBasketId );

			if ( $exists && ( $this->_getReqParam( 'id' ) == $paymentBasket->shared_secret ) && ( $paymentBasket->payment_status != 'Completed' ) ) {
				$paymentBasket->payment_status	=	'RedisplayOriginalBasket';

				$this->_setErrorMSG( CBTxt::T( 'Payment cancelled.' ) );
			}
		}

		return false;
	}

	/**
	 * The payment service provider server did a server-to-server notification: Verify and handle it here:
	 *
	 * @param  cbpaidPaymentBasket  $paymentBasket  New empty object. returning: includes the id of the payment basket of this callback (strictly verified, otherwise untouched)
	 * @param  array                $postdata       _POST data for saving edited tab content as generated with getEditTab
	 * @return string                              Text to return to gateway if notification, or NULL if nothing to display
	 */
	protected function handleNotification( $paymentBasket, $postdata )
	{
		if ( ( count( $postdata ) > 0 ) && isset( $postdata['reference'] ) ) {
			// we prefer POST for sensitive data:
			$requestdata	=	$postdata;
		} else {
			// but if gateway needs GET, we will work with it too:
			$requestdata	=	$this->_getGetParams();
		}

		if ( $this->_returnParamsHandler( $paymentBasket, $requestdata, 'I' ) ) {
			print( "version=2\ncdr=0\n" );
			// exit();
		} else {
			print( "version=2\ncdr=1\n" );
			// exit();
		}
	}

	/**
	 * Returns the notification URL for the callbacks/IPNs (not htmlspecialchared)
	 * Uses getAccountParam( 'notifications_host' ) if it is defined
	 *
	 * @param  cbpaidPaymentBasket  $paymentBasket
	 * @param  boolean              $noAccount      TRUE: do not include account number, but account method, FALSE (default): include account number
	 * @return string               URL
	 */
	protected function getNotifyUrl( $paymentBasket, $noAccount = false )
	{
		global $_CB_framework;

		return $_CB_framework->getCfg( 'live_site' ) . '/components/com_comprofiler/plugin/user/plug_cbpaidsubscriptions/processors/ciccreditmutuel/notif.php';
	}

	/**
	 * sign payment request $requestParams with validation code added to $requestParams array
	 *
	 * @param  array     $requestParams        The keyed parameters array to add generate validation to $requestParams['signature']
	 */
	private function _signRequestParams( &$requestParams )
	{
		$requestParams['MAC']	=	hash_hmac( 'sha1', $this->_concatVars( $requestParams, [], null, '*', '=', true, true, false, true ), hex2bin( $this->_getUsableKey( $this->getAccountParam( 'pspsecret' ) ) ) );
	}

	/**
	 * Validate a reply in $requestParams using md5check value, and then also through the API if it is available
	 *
	 * @return boolean                 TRUE: valid, FALSE: invalid
	 */
	private function _pspVerifySignature()
	{
		// We need to work with the raw payload as it's vital EVERY parameter in it matches EXACTLY what CIC Credit Mutuel sent or the MAC check WILL fail
		$payload		=	( $_REQUEST['payload'] ?? [] );
		$MAC			=	( $payload['MAC'] ?? '' );

		unset( $payload['MAC'] );

		if ( isset( $payload['texte-libre'] ) ) {
			// The MAC is created before slashes are added to GET/POST so we need to remove them for this to validate
			$payload['texte-libre']	=	stripslashes( $payload['texte-libre'] );
		}

		// MAC is case sensitive so just uppercase both to be safe
		return ( strtoupper( $MAC ) === strtoupper( hash_hmac( 'sha1', $this->_concatVars( $payload, [], null, '*', '=', true, true, false, true ), hex2bin( $this->_getUsableKey( $this->getAccountParam( 'pspsecret' ) ) ) ) ) );
	}

	/**
	 * Compute the CBSubs payment_status based on gateway's reply in $postdata:
	 *
	 * @param  array   $postdata  raw POST data received from the payment gateway
	 * @param  string  $reason    OUT: reason_code
     * @return string             CBSubs status
	 */
	private function _paymentStatus( $postdata, &$reason )
	{
		$status			=	cbGetParam( $postdata, 'code-retour', '' );

		switch ( strtolower( $status ) ) {
			case 'payetest':
			case 'paiement':
			case 'paiement_pf2':
			case 'paiement_pf3':
			case 'paiement_pf4':
				$reason	=	'';
				$status	=	'Completed';
				break;
			case 'annulation':
			case 'annulation_pf2':
			case 'annulation_pf3':
			case 'annulation_pf4':
				$reason	=	'Payment declined';
				$status	=	'Denied';
				break;
		}

		return $status;
	}

	/**
	 * Compute the CBSubs payment_type based on gateway's reply $postdata:
	 *
	 * @param  array   $postdata raw POST data received from the payment gateway
	 * @return string  Human-readable string
	 */
	private function _getPaymentType( $postdata )
	{
		$type			=	cbGetParam( $postdata, 'brand', '' );

		switch ( strtolower( $type ) ) {
			case 'am':
				$type	=	'American Express';
				break;
			case 'cb':
				$type	=	'GIE CB';
				break;
			case 'mc':
				$type	=	'Mastercard';
				break;
			case 'vi':
				$type	=	'Visa';
				break;
			case 'na':
				$type	=	'Not Available';
				break;
		}

		return $type;
	}

	/**
	 * Popoulates basic request parameters for gateway depending on basket (without specifying payment type)
	 *
	 * @param  cbpaidPaymentBasket  $paymentBasket   paymentBasket object
	 * @return array                                 Returns array $requestParams
	 */
	private function _getBasicRequstParams( $paymentBasket )
	{
		global $_CB_framework;

		// mandatory parameters:
		$requestParams						=	array();
		$requestParams['version']			=	$this->getAccountParam( 'pspversion' );
		$requestParams['TPE']				=	$this->getAccountParam( 'pspid' );
		$requestParams['date']				=	date( 'd/m/Y:H:m:s', $_CB_framework->now() );
		$requestParams['montant']			=	sprintf( '%.2f', $paymentBasket->mc_gross ) . $paymentBasket->mc_currency;
		$requestParams['reference']			=	( $this->getAccountParam( 'id' ) . '-' . $paymentBasket->id );
		$requestParams['lgue']				=	$this->getAccountParam( 'language' );

		$contexte							=	[	'billing'	=>	[	'addressLine1'		=>	cbIsoUtf_substr( $paymentBasket->address_street, 0, 50 ),
																		'city'				=>	cbIsoUtf_substr( $paymentBasket->address_city, 0, 50 ),
																		'postalCode'		=>	$paymentBasket->address_zip,
																		'country'			=>	cbIsoUtf_substr( $paymentBasket->address_country_code, 0, 10 ),
																	 ]
												];

		if ( $paymentBasket->address_country_code === 'US' ) {
			// State only appears required for US purchases and passing it for other countries will error the form
			$contexte['billing']['stateOrProvince']	=	$paymentBasket->address_state;
		}

		$requestParams['contexte_commande']	=	base64_encode( utf8_encode( json_encode( $contexte ) ) );

		$societe							=	$this->getAccountParam( 'pspsociete' );

		if ( $societe ) {
			$requestParams['societe']		=	$societe;
		}

		$threeDS							=	$this->getAccountParam( '3ds' );

		if ( $threeDS ) {
			$requestParams['ThreeDSecureChallenge']		=	$threeDS;
		}

		// optional parameters:
		$requestParams['texte-libre']		=	$paymentBasket->item_name;

		// recommended anti-fraud fields:
		$requestParams['mail']				=	$paymentBasket->payer_email;

		// urls for return and cancel:
		$requestParams['url_retour_ok']		=	$this->getSuccessUrl( $paymentBasket );
		$requestParams['url_retour_err']	=	$this->getCancelUrl( $paymentBasket );

		return $requestParams;
	}

	/**
	 * The user got redirected back from the payment service provider with a success message: let's see how successfull it was
	 *
	 * @param  cbpaidPaymentBasket  $paymentBasket       New empty object. returning: includes the id of the payment basket of this callback (strictly verified, otherwise untouched)
	 * @param  array                $requestdata         Data returned by gateway
	 * @param  string               $type                Type of return ('R' for PDT, 'I' for INS, 'A' for Autorecurring payment (Vault) )
     * @param  array                $additionalLogData   Additional strings to log with IPN
	 * @return string                                    HTML to display if frontend, text to return to gateway if notification, FALSE if registration cancelled and ErrorMSG generated, or NULL if nothing to display
	 */
	private function _returnParamsHandler( $paymentBasket, $requestdata, $type, $additionalLogData = null )
	{
		global $_CB_framework, $_GET, $_POST;

		$ret													=	null;
		$paymentBasketId										=	(int) $this->_getBasketId( $requestdata );

		if ( $paymentBasketId ) {
			$exists												=	$paymentBasket->load( (int) $paymentBasketId );

			if ( $exists && ( $type === 'R' ) && ( $paymentBasket->getString( 'payment_status', '' ) === 'NotInitiated' ) ) {
				// We have no information to go off of for a return but lets try to put the basket in a pending state so it doesn't expire while waiting for the IPN otherwise the user just sees a blank page
				$ipn											=	$this->_prepareIpn( 'R', 'Pending', 'Credit Card', '', $_CB_framework->now(), 'utf-8' );

				$ipn->test_ipn									=	( (int) $this->getAccountParam( 'normal_gateway' ) === 0 ? 1 : 0 );
				$ipn->raw_data									=	'$message_type="RETURN_TO_SITE";' . "\n";

				$ipn->bindBasket( $paymentBasket );

				$ipn->sale_id									=	$paymentBasketId;
				$ipn->user_id									=	(int) $paymentBasket->user_id;
				$ipn->txn_type									=	'web_accept';
				$ipn->txn_id									=	( $this->getAccountParam( 'id' ) . '-' . $paymentBasket->id );

				$this->_storeIpnResult( $ipn, 'SUCCESS' );
				$this->_bindIpnToBasket( $ipn, $paymentBasket );

				$paymentBasket->payment_method					=	$this->getPayName();
				$paymentBasket->gateway_account					=	$this->getAccountParam( 'id' );

				$this->updatePaymentStatus( $paymentBasket, $ipn->txn_type, $ipn->payment_status, $ipn, 1, 0, 0, false );

				$ret											=	true;
			} elseif ( $exists && ( $type === 'I' ) && ( $paymentBasket->getString( 'payment_status', '' ) !== 'Completed' ) ) {
				// Log the return record:
				$log_type										=	$type;
				$reason											=	null;
				$paymentStatus									=	$this->_paymentStatus( $requestdata, $reason );
				$paymentType									=	$this->_getPaymentType( $requestdata );
				$paymentTime									=	$_CB_framework->now();

				if ( $paymentStatus == 'Error' ) {
					$errorTypes									=	array( 'I' => 'D', 'R' => 'E' );

					if ( isset( $errorTypes[$type] ) ) {
						$log_type								=	$errorTypes[$type];
					}
				}

				$ipn											=	$this->_prepareIpn( $log_type, $paymentStatus, $paymentType, $reason, $paymentTime, 'utf-8' );

				if ( $paymentStatus == 'Refunded' ) {
					// in case of refund we need to log the payment as it has same TnxId as first payment: so we need payment_date for discrimination:
					$ipn->payment_date							=	gmdate( 'H:i:s M d, Y T', $paymentTime ); // paypal-style
				}

				$ipn->test_ipn									=	( cbGetParam( $requestdata, 'code-retour' ) == 'payetest' ? 1 : 0 );
				$ipn->raw_data									=	'$message_type="NOTIFICATION";' . "\n";

				if ( $additionalLogData ) {
					foreach ( $additionalLogData as $k => $v ) {
						$ipn->raw_data							.=	'$' . $k . '="' . var_export( $v, true ) . '";' . "\n";
					}
				}

				$ipn->raw_data									.=	/* cbGetParam() not needed: we want raw info */ '$requestdata=' . var_export( $requestdata, true ) . ";\n"
																.	/* cbGetParam() not needed: we want raw info */ '$_GET=' . var_export( $_GET, true ) . ";\n"
																.	/* cbGetParam() not needed: we want raw info */ '$_POST=' . var_export( $_POST, true ) . ";\n";

				if ( $paymentStatus == 'Error' ) {
					$paymentBasket->reason_code					=	$reason;

					$this->_storeIpnResult( $ipn, 'ERROR:' . $reason );
					$this->_setLogErrorMSG( 4, $ipn, $this->getPayName() . ': ' . $reason, CBTxt::T( 'Sorry, the payment server replied with an error.' ) . ' ' . CBTxt::T( 'Please contact site administrator to check payment status and error log.' ) );

					$ret										=	false;
				} else {
					$ipn->bindBasket( $paymentBasket );

					$ipn->sale_id								=	$paymentBasketId;

					$insToIpn									=	array(	'auth_id' => 'numauto',
																			'address_country_code' => 'originetr',
																			'residence_country' => 'originecb'
																		);

					foreach ( $insToIpn as $k => $v ) {
						$ipn->$k								=	cbGetParam( $requestdata, $v );
					}

					$ipn->mc_gross								=	sprintf( '%.2f', cbGetParam( $requestdata, 'montant' ) );
					$ipn->mc_currency							=	substr( cbGetParam( $requestdata, 'montant' ), -3, 3 );
					$ipn->user_id								=	(int) $paymentBasket->user_id;
					$ipn->txn_type								=	'web_accept';
					$ipn->txn_id								=	cbGetParam( $requestdata, 'reference' );

					// validate payment from PDT or IPN
					if ( $this->_pspVerifySignature() ) {
						if ( ( $paymentBasketId == $this->_getBasketId( $requestdata ) ) && ( ( sprintf( '%.2f', $paymentBasket->mc_gross ) == $ipn->mc_gross ) || ( $ipn->payment_status == 'Refunded' ) ) && ( $paymentBasket->mc_currency == $ipn->mc_currency ) ) {
							if ( in_array( $ipn->payment_status, array( 'Completed', 'Processed', 'Pending', 'Refunded', 'Denied' ) ) ) {
								$this->_storeIpnResult( $ipn, 'SUCCESS' );
								$this->_bindIpnToBasket( $ipn, $paymentBasket );

								// add the gateway to the basket:
								$paymentBasket->payment_method	=	$this->getPayName();
								$paymentBasket->gateway_account	=	$this->getAccountParam( 'id' );

								$this->updatePaymentStatus( $paymentBasket, $ipn->txn_type, $ipn->payment_status, $ipn, 1, 0, 0, false );

								if ( in_array( $ipn->payment_status, array( 'Completed', 'Processed', 'Pending' ) ) ) {
									$ret						=	true;
								}
							} else {
								$this->_storeIpnResult( $ipn, 'FAILED' );

								$paymentBasket->payment_status	=	$ipn->payment_status;

								$this->_setErrorMSG( '<div class="alert alert-info">' . $this->getTxtNextStep( $paymentBasket ) . '</div>' );

								$paymentBasket->payment_status	=	'RedisplayOriginalBasket';
								$ret							=	false;
							}
						} else {
							$this->_storeIpnResult( $ipn, 'MISMATCH' );
							$this->_setLogErrorMSG( 3, $ipn, $this->getPayName() . ': amount or currency missmatch', CBTxt::T( 'Sorry, the payment does not match the basket.' ) . ' ' . CBTxt::T( 'Please contact site administrator to check error log.' ) );

							$ret								=	false;
						}
					} else {
						$this->_storeIpnResult( $ipn, 'SIGNERROR' );
						$this->_setLogErrorMSG( 3, $ipn, $this->getPayName() . ': md5check or transaction does not match with gateway. Please check Secret Key setting', CBTxt::T( 'The Secret Key signature is incorrect.' ) . ' ' . CBTxt::T( 'Please contact site administrator to check error log.' ) );

						$ret									=	false;
					}
				}
			}
		} else {
			$this->_setLogErrorMSG( 3, null, $this->getPayName() . ': reference is missing in the return URL: ' . var_export( $_GET, true ), CBTxt::T( 'Please contact site administrator to check error log.' ) );
		}

		return  $ret;
	}

	/**
	 * Return the key to be used in the hmac function
	 * function is as provided by PSP
	 *
	 * @param  string  $secret  pspsecret
	 * @return string
	 */
	private function _getUsableKey( $secret )
	{
		$hexStrKey			=	substr( $secret, 0, 38 );
		$hexFinal			=	substr( $secret, 38, 2 ) . '00';
		$cca0				=	ord( $hexFinal );

		if ( ( $cca0 > 70 ) && ( $cca0 < 97 ) ) {
			$hexStrKey		.=	chr( ( $cca0 - 23 ) ) . substr( $hexFinal, 1, 1 );
		} else {
			if ( substr( $hexFinal, 1, 1 ) == 'M' ) {
				$hexStrKey	.=	substr( $hexFinal, 0, 1 ) . '0';
			} else {
				$hexStrKey	.=	substr( $hexFinal, 0, 2 );
			}
		}

		return $hexStrKey;
	}

	/**
	 * Return the basket id from reference
	 * removes the gateway id only needed for notifications
	 *
	 * @param array $requestdata
	 * @return string
	 */
	private function _getBasketId( $requestdata )
	{
		$reference	=	cbGetParam( $requestdata, 'reference', null );
		$ids		=	explode( '-', $reference, 2 );

		if ( isset( $ids[1] ) ) {
			$basket	=	(int) $ids[1];
		} else {
			$basket	=	0;
		}

		return $basket;
	}
}

/**
 * Payment account class for this gateway: Stores the settings for that gateway instance, and is used when editing and storing gateway parameters in the backend.
 *
 * OEM base
 * No methods need to be implemented or overriden in this class, except to implement the private-type params used specifically for this gateway:
 */
class cbpaidGatewayAccountciccreditmutueloem extends cbpaidGatewayAccounthostedpage
{
}

/**
 * Payment handler class for this gateway: Handles all payment events and notifications, called by the parent class:
 *
 * Gateway-specific
 * Please note that except the constructor and the API version this class does not implement any public methods.
 */
class cbpaidciccreditmutuel extends cbpaidciccreditmutueloem
{
}

/**
 * Payment account class for this gateway: Stores the settings for that gateway instance, and is used when editing and storing gateway parameters in the backend.
 *
 * Gateway-specific
 * No methods need to be implemented or overriden in this class, except to implement the private-type params used specifically for this gateway:
 */
class cbpaidGatewayAccountciccreditmutuel extends cbpaidGatewayAccountciccreditmutueloem
{
}
