<?php
/**
* CBSubs (TM): Community Builder Paid Subscriptions Plugin: cbsubspiwik
* @version $Id: cbsubs.piwik.php 1525 2012-11-23 00:08:43Z beat $
* @package CBSubs (TM) Community Builder Plugin for Paid Subscriptions (TM)
* @subpackage cbsubs.piwik.php
* @author Krileon
* @copyright (C) 2007-2022 and Trademark of Lightning MultiCom SA, Switzerland - www.joomlapolis.com - and its licensors, all rights reserved
* @license http://www.gnu.org/licenses/old-licenses/gpl-2.0.html GNU/GPL version 2
*/

use CB\Database\Table\UserTable;
use CBLib\Registry\ParamsInterface;
use CBLib\Language\CBTxt;

if ( ! ( defined( '_VALID_CB' ) || defined( '_JEXEC' ) || defined( '_VALID_MOS' ) ) ) { die( 'Direct Access to this location is not allowed.' ); }

global $_PLUGINS;
$_PLUGINS->registerFunction( 'onCPayBeforeDrawPlan', 'onCPayBeforeDrawPlan', 'getcbsubspiwikTab' );
$_PLUGINS->registerFunction( 'onCPayAfterPaymentBasketUpdated', 'onCPayAfterPaymentBasketUpdated', 'getcbsubspiwikTab' );
$_PLUGINS->registerFunction( 'onCPayAfterPaymentStatusUpdateEvent', 'onCPayAfterPaymentStatusUpdateEvent', 'getcbsubspiwikTab' );
$_PLUGINS->registerFunction( 'onCPayBeforeStorePaymentBasketUpdated', 'onCPayBeforeStorePaymentBasketUpdated', 'getcbsubspiwikTab' );

/**
 * CBSubs Piwik integration plugin class
 */
class getcbsubspiwikTab extends cbTabHandler {
	/**
	 * Event handler for onCPayBeforeDrawPlan, called just before drawing product/plan.
	 *
	 * @param  cbpaidProduct  $plan                 Product/Plan being rendered
	 * @param  string         $childrenRendering    Rendered children plans
	 * @param  string         $reason               Payment reason: 'N'=new subscription (default), 'R'=renewal, 'U'=update
	 * @param  boolean        $drawOnlyAsContainer  TRUE: only rendered as a container (not selected nor selectable, just to surround children plans)
	 * @param  int            $userId               User id for whom this product is displayed
	 * @return string                               HTML
	 */
	public function onCPayBeforeDrawPlan( &$plan, /** @noinspection PhpUnusedParameterInspection */ &$childrenRendering, /** @noinspection PhpUnusedParameterInspection */ $reason, /** @noinspection PhpUnusedParameterInspection */ $drawOnlyAsContainer, /** @noinspection PhpUnusedParameterInspection */ $userId ) {
		global $_CB_framework;

		if ( $_CB_framework->getUi() != 1 ) {
			return;
		}

		$user	=	CBuser::getUserDataInstance( (int) $_CB_framework->myId() );

		if ( ! is_object( $user ) ) {
			return;
		}

		$params	=	cbpaidApp::settingsParams();
		$pages	=	explode( '|*|', $params->get( 'piwik_pages', '1|*|2' ) );

		if ( ! $pages ) {
			return;
		} elseif ( cbGetParam( $_GET, 'do', null ) == 'displayplans' ) {
			if ( ! in_array( 1, $pages ) ) {
				return;
			}
		} elseif ( cbGetParam( $_GET, 'do', null ) == 'accessdenied' ) {
			if ( ! in_array( 2, $pages ) ) {
				return;
			}
		} elseif ( cbGetParam( $_GET, 'task', null ) == 'registers' ) {
			if ( ! in_array( 3, $pages ) ) {
				return;
			}
		} elseif ( ( cbGetParam( $_GET, 'task', null ) == 'userprofile' ) || ( ! cbGetParam( $_GET, 'task', null ) ) ) {
			if ( ! in_array( 4, $pages ) ) {
				return;
			}
		} elseif ( ! in_array( 5, $pages ) ) {
			return;
		}

		$this->piwikTracker( $plan, $user, 'plan' );
	}
	/**
	 * Event handler for onCPayAfterPaymentBasketUpdated, called each time a payment basket has been (re)computed and stored to database
	 *
	 * @param  cbpaidPaymentBasket       $paymentBasket          Payment basket
	 * @param  cbpaidPaymentItem[]       $paymentItems           Payment items in the basket
	 * @param  cbpaidPaymentTotalizer[]  $paymentTotalizers      Payment totalizers in the basket
	 * @return void
	 */
	public function onCPayAfterPaymentBasketUpdated( $paymentBasket, /** @noinspection PhpUnusedParameterInspection */ $paymentItems, /** @noinspection PhpUnusedParameterInspection */ $paymentTotalizers ) {
		global $_CB_framework;

		if ( $_CB_framework->getUi() != 1 ) {
			return;
		}

		$user	=	CBuser::getUserDataInstance( (int) $paymentBasket->user_id );

		if ( ! is_object( $user ) ) {
			return;
		}

		$this->piwikTracker( $paymentBasket, $user, 'basket' );
	}
	/**
	 * Event handler for onCPayAfterPaymentStatusUpdateEvent, called each time an update occurs to the payment status of the payment basket
	 *
	 * Paypal status mappings:
	 * CB Unified status				Paypal status
	 * Completed				<--		Completed
	 * Processed				<--		Processed, Canceled_Reversal
	 * Denied					<--		Denied, Expired, Failed, Voided
	 * Refunded					<--		Reversed, Refunded, Partially-Refunded
	 * Pending					<--		Pending, In-Progress
	 * RegistrationCancelled	<--		A new cb registration got cancelled by user (e.g. paypal cancel payment button)
	 *
	 * @param  UserTable           $user                   User paying the payment basket
	 * @param  cbpaidPaymentBasket $paymentBasket          Payment basket
	 * @param  cbpaidSomething[]   $subscriptions          Subscriptions in the basket
	 * @param  string              $unifiedStatus          New Unified status (see above)
	 * @param  string              $previousUnifiedStatus  Previous Unified status (see above)
	 * @param  string              $eventType              type of event (paypal type): 'web_accept', 'subscr_payment', 'subscr_signup', 'subscr_modify', 'subscr_eot', 'subscr_cancel', 'subscr_failed'
	 * @return void
	 */
	public function onCPayAfterPaymentStatusUpdateEvent( &$user, &$paymentBasket, /** @noinspection PhpUnusedParameterInspection */ &$subscriptions, $unifiedStatus, $previousUnifiedStatus, /** @noinspection PhpUnusedParameterInspection */ $eventType ) {
		global $_CB_framework;

		$params	=	cbpaidApp::settingsParams();

		if ( ( ! is_object( $user ) ) || ( ! in_array( $unifiedStatus, array( 'Completed', 'Processed' ) ) ) || ( in_array( $previousUnifiedStatus, array( 'Completed', 'Processed' ) ) ) || ( ( $_CB_framework->getUi() != 1 ) && ( $params->get( 'piwik_method', 1 ) == 0 ) ) ) {
			return;
		}

		$this->piwikTracker( $paymentBasket, $user, 'purchase' );
	}
	/**
	 * Event handler for onCPayBeforeStorePaymentBasketUpdated, called each time the basket has been stored after an update at ordering time
	 *
	 * @param  cbpaidPaymentBasket       $paymentBasket          Payment basket
	 * @param  cbpaidPaymentItem[]       $paymentItems           Payment items in the basket
	 * @param  cbpaidPaymentTotalizer[]  $paymentTotalizers      Payment totalizers in the basket
	 * @return void
	 */
	public function onCPayBeforeStorePaymentBasketUpdated( $paymentBasket, /** @noinspection PhpUnusedParameterInspection */ $paymentItems, /** @noinspection PhpUnusedParameterInspection */ $paymentTotalizers ) {
		$cookie					=	$this->getPiwikCookie();

		if ( $cookie ) {
			$integrationParams	=	$paymentBasket->getParams( 'integrations' );
			$visitorId			=	substr( $cookie, 0, strpos( $cookie, '.' ) );

			if ( $visitorId && ( strlen( $visitorId ) == 16 ) ) {
				$integrationParams->set( 'piwik_visitorid', $visitorId );
				$paymentBasket->storeParams( 'integrations' );
			}
		}
	}
	/**
	 * Computes the factored currency-converted price
	 *
	 * @param  float|int  $price      Price in $currency
	 * @param  string     $currency   Currency (ISO)
	 * @return float|int              Price in plugin's currency with plugin's price factor
	 */
	private function price( $price, $currency ) {
		$params					=	cbpaidApp::settingsParams();
		$priceFactor			=	(float) $params->get( 'piwik_price_factor', '1.0' );
		$piwikCurrency			=	$params->get( 'piwik_currency', '' );

		if ( $piwikCurrency == '' ) {
			$piwikCurrency		=	$params->get( 'currency_code', 'USD' );
		}

		if ( $currency == '' ) {
			$currency			=	$params->get( 'currency_code', 'USD' );
		}

		$itemPriceFactored		=	round( $price * $priceFactor, 2 );

		if ( $currency != $piwikCurrency ) {
			$_CBPAY_CURRENCIES	=	cbpaidApp::getCurrenciesConverter();
			$itemPriceFactored	=	round( $_CBPAY_CURRENCIES->convertCurrency( $currency, $piwikCurrency, $itemPriceFactored ), 2 );
		}

		return $itemPriceFactored;
	}
	/**
	 * Does the Piwik tracker integration by RPC or by outputing Javascript depending on plugin parameters
	 *
	 * @param  cbpaidProduct|cbpaidPaymentBasket  $row          The item to log
	 * @param  UserTable                          $user         The user doing the action
	 * @param  string                             $trackerType  'plan', 'purchase', 'basket'
	 * @return void
	 */
	private function piwikTracker( $row, $user, $trackerType = 'purchase' ) {
		global $_CB_framework;

		$params							=	cbpaidApp::settingsParams();
		$piwikUrl						=	trim( trim( $params->get( 'piwik_url', null ) ), '\/' );
		$siteId							=	(int) $params->get( 'piwik_siteid', null );

		if ( ( ! $piwikUrl ) || ( ! $siteId ) ) {
			return;
		}

		$piwikUrl						=	$piwikUrl . '/';
		$method							=	(int) $params->get( 'piwik_method', 1 );

		if ( $method == 3 ) {
			if ( ( $_CB_framework->getUi() != 1 ) || ( $trackerType == 'purchase' ) ) {
				$method					=	1;
			} else {
				$method					=	0;
			}
		}

		if ( $method ) {
			$token						=	trim( $params->get( 'piwik_token', null ) );

			if ( ! $token ) {
				return;
			}

			$trackerPath				=	trim( trim( $params->get( 'piwik_path', null ) ), '\/' );
			$trackerUrl					=	$_CB_framework->getCfg( 'absolute_path' ) . ( $trackerPath ? '/' . $trackerPath : null ) . '/PiwikTracker.php';

			if ( ! file_exists( $trackerUrl ) ) {
				return;
			}

			/** @noinspection PhpIncludeInspection */
			require_once( $trackerUrl );
		} else {
			$token						=	null;
		}

		$cbUser							=	CBuser::getInstance( (int) $user->id, false );
		$pageTitle						=	null;
		$pageUrl						=	null;
		$extraStrings					=	array();

		if ( $trackerType == 'purchase' ) {
			$pageTitle					=	CBTxt::T( 'CB Paid Subscriptions Purchase' );
			$extraStrings				=	$row->substitutionStrings( false );
		} elseif ( $trackerType == 'basket' ) {
			$pageTitle					=	CBTxt::T( 'CB Paid Subscriptions Basket' );
			$extraStrings				=	$row->substitutionStrings( false );
		} elseif ( $trackerType == 'plan' ) {
			$pageTitle					=	$row->name;
			$pageUrl					=	$_CB_framework->pluginClassUrl( array( 'plugin' => 'cbpaidsubscriptions', 'do' => 'displayplans', 'plans' => (int) $row->id ), true );
			$extraStrings				=	$row->substitutionStrings( false, (int) $user->id );
		}

		if ( in_array( $trackerType, array( 'purchase', 'basket' ) ) ) {
			$saleIpAddresses			=	explode( ',', $row->ip_addresses );
			/** @var ParamsInterface $integrationParams */
			$integrationParams			=	$row->getParams( 'integrations' );
			$visitorId					=	$integrationParams->get( 'piwik_visitorid', null );
		} else {
			$saleIpAddresses			=	cbGetIParray();
			$visitorId					=	null;
		}

		$trackPageView					=	true;

		if ( $method ) {
			$tracker					=	new PiwikTracker( $siteId, $piwikUrl );

			$tracker->setTokenAuth( $token );
			$tracker->setIp( $saleIpAddresses[0] );

			if ( $visitorId ) {
				$tracker->setVisitorId( $visitorId );
			}

			if ( $pageUrl ) {
				$tracker->setUrl( $pageUrl );
			}
			$js							=	null;
		} else {
			$tracker					=	null;
			$cookieDomain				=	trim( $params->get( 'piwik_cookiedomain', null ) );
			$domainName					=	trim( $params->get( 'piwik_domainname', null ) );

			$js							= 	"if ( window.Piwik ) {"
										.		"var pkBaseURL = '" . addslashes( $piwikUrl ) . "';"
										.		"var piwikTracker = Piwik.getTracker( pkBaseURL + 'js/', " . $siteId . " );"
										.		( $cookieDomain ? "piwikTracker.setCookieDomain( '" . addslashes( $cookieDomain ) . "' );" : null )
										.		( $domainName ? "piwikTracker.setDomains( '" . addslashes( $domainName ) . "' );" : null )
										.		( $pageTitle ? "piwikTracker.setDocumentTitle( '" . addslashes( $pageTitle ) . "' );" : null )
										.		( $pageUrl ? "piwikTracker.setCustomUrl( '" . addslashes( $pageUrl ) . "' );" : null );
		}

		if ( in_array( $trackerType, array( 'purchase', 'basket' ) ) ) {
			/** @var $paymentItems cbpaidPaymentItem[] */
			$paymentItems				=	$row->loadPaymentItems();

			if ( $paymentItems ) foreach ( $paymentItems as $item ) {

				/** @var $subscription cbpaidSomething */
				$subscription			=	$item->loadSubscription();
				$plan					=	$subscription ? $subscription->getPlan() : cbpaidPlansMgr::getInstance()->loadPlan( $item->plan_id );
				if ( ! $plan ) {
					continue;
				}

				$itemPriceFactored		=	$this->price( $item->getPrice(), $item->currency );
				$planName				=	$plan->getPersonalized( 'name', $subscription ? $subscription->user_id : $row->user_id, false );

				if ( $method ) {
					$tracker->addEcommerceItem( $plan->id, $planName, $plan->item_type, $itemPriceFactored, $item->quantity );

					if ( $trackerType == 'purchase' ) {
						for ( $i = 1; $i <= 5; $i++ ) {
							$goal		=	(int) $plan->getParam( 'piwik_goal' . $i, null, 'integrations' );

							if ( $goal ) {
								$tracker->doTrackGoal( $goal, $itemPriceFactored );
							}
						}
					}
				} else {
					$js					.=		"piwikTracker.addEcommerceItem("
										.			"'" . addslashes( $plan->id ) . "',"					// SKU
										.			"'" . addslashes( $planName ) . "',"					// Product Name
										.			"'" . addslashes( $plan->item_type ) . "',"				// Category
										.			"'" . addslashes( $itemPriceFactored ) . "',"			// Price
										.			"'" . addslashes( $item->quantity ) . "'"				// Quantity
										.		");";

					if ( $trackerType == 'purchase' ) {
						for ( $i = 1; $i <= 5; $i++ ) {
							$goal		=	(int) $plan->getParam( 'piwik_goal' . $i, null, 'integrations' );

							if ( $goal ) {
								$js		.=		"piwikTracker.trackGoal( '" . addslashes( $goal ) . "', '" . addslashes( $itemPriceFactored ) . "' );";
							}
						}
					}
				}
			}
		}

		$subTotal						=	null;
		if ( $trackerType == 'purchase' ) {
			$subTotal					=	$this->price( $row->mc_gross - ( $row->tax + $row->mc_shipping ), $row->mc_currency );
		} elseif ( $trackerType == 'basket' ) {
			if ( cbGetParam( $_GET, 'cbpact', null ) ) {
				$trackPageView			=	false;
			}
		}

		if ( $method ) {
			if ( $trackerType == 'purchase' ) {
				$tracker->doTrackEcommerceOrder( $row->item_number, $this->price( $row->mc_gross, $row->mc_currency ), $subTotal, $this->price( $row->tax, $row->mc_currency ), $this->price( $row->mc_shipping, $row->mc_currency ), false );
			} elseif ( $trackerType == 'basket' ) {
				$tracker->doTrackEcommerceCartUpdate( $this->price( $row->mc_gross, $row->mc_currency ) );
			} elseif ( $trackerType == 'plan' ) {
				$tracker->setEcommerceView( $row->id, $row->name, $row->item_type, $this->price( $row->rate, $row->currency ) );
			}

			if ( $trackPageView ) {
				for ( $i = 1; $i <= 5; $i++ ) {
					$customName			=	trim( $cbUser->replaceUserVars( $params->get( 'piwik_custom_name' . $i, null ), false, true, $extraStrings ) );

					if ( $customName ) {
						$customValue	=	trim( $cbUser->replaceUserVars( $params->get( 'piwik_custom_value' . $i, null ), false, true, $extraStrings ) );

						$tracker->setCustomVariable( $i, $customName, $customValue );
					}
				}

				$tracker->doTrackPageView( $pageTitle );
			}
		} else {
			if ( $trackerType == 'purchase' ) {
				$js						.=		"piwikTracker.trackEcommerceOrder("
										.			"'" . addslashes( $row->item_number ) . "',"										// Order ID
										.			"'" . addslashes( $this->price( $row->mc_gross, $row->mc_currency ) ) . "',"		// Grand Total
										.			"'" . addslashes( $subTotal ) . "',"												// Sub Total
										.			"'" . addslashes( $this->price( $row->tax, $row->mc_currency ) ) . "',"				// Tax
										.			"'" . addslashes( $this->price( $row->mc_shipping, $row->mc_currency ) ) . "',"		// Shipping
										.			"false"																				// Discount
										.		");";
			} elseif ( $trackerType == 'basket' ) {
				$js						.=		"piwikTracker.trackEcommerceCartUpdate( '" . addslashes( $this->price( $row->mc_gross, $row->mc_currency ) ) . "' );";
			} elseif ( $trackerType == 'plan' ) {
				$js						.=		"piwikTracker.setEcommerceView( '" . addslashes( $row->id ) . "', '" . addslashes( $row->name ) . "', '" . addslashes( $row->item_type ) . "', '" . addslashes( $this->price( $row->rate, $row->currency ) ) . "' );";
			}

			if ( $trackPageView ) {
				for ( $i = 1; $i <= 5; $i++ ) {
					$customName			=	trim( $cbUser->replaceUserVars( $params->get( 'piwik_custom_name' . $i, null ), false, true, $extraStrings ) );

					if ( $customName ) {
						$customValue	=	trim( $cbUser->replaceUserVars( $params->get( 'piwik_custom_value' . $i, null ), false, true, $extraStrings ) );

						$js				.=		"piwikTracker.setCustomVariable( $i, '" . addslashes( $customName ) . "', '" . addslashes( $customValue ) . "', 'page' );";
					}
				}

				$js						.=		"piwikTracker.trackPageView();";
			}

			$js							.=	"}";

			$_CB_framework->document->addHeadScriptUrl( $piwikUrl . 'js/' );
			$_CB_framework->document->addHeadScriptDeclaration( $js );
		}
	}
	/**
	 * Gets the Piwik cookie name if it can find it
	 *
	 * @return string|boolean
	 */
	private function getPiwikCookie() {
		$params		=	cbpaidApp::settingsParams();
		$piwikUrl	=	trim( trim( $params->get( 'piwik_url', null ) ), '\/' );
		$siteId		=	(int) $params->get( 'piwik_siteid', null );

		if ( ( ! $piwikUrl ) || ( ! $siteId ) ) {
			return false;
		}

		$cookieName	=	'_pk_id_' . $siteId . '_';

		if ( $_COOKIE ) foreach ( $_COOKIE as $k => $v ) {
			if ( strpos( $k, $cookieName ) !== false ) {
				return $v;
			}
		}

		return false;
	}
}
