<?php
/**
* Community Builder (TM)
* @version $Id: $
* @package CommunityBuilder
* @copyright (C) 2004-2022 www.joomlapolis.com / Lightning MultiCom SA - and its licensors, all rights reserved
* @license http://www.gnu.org/licenses/old-licenses/gpl-2.0.html GNU/GPL version 2
*/

namespace CB\Plugin\PlanOptions;

use CB\Plugin\PlanOptions\Entity\OptionEntity;
use CB\Plugin\PlanOptions\Entity\PaidEntity;
use CB\Plugin\PlanOptions\Entity\PriceEntity;
use CB\Plugin\PlanOptions\Entity\ValueEntity;
use CBLib\Application\Application;
use CBLib\Registry\ParamsInterface;
use CBLib\Registry\Registry;
use cbpaidPaymentBasket;
use cbpaidProduct;
use cbpaidProductDonation;
use cbpaidProductMerchandise;
use cbpaidProductUsersubscription;
use cbpaidSomething;
use moscomprofilerHTML;

\defined('CBLIB') or die();

class Helper
{
	/**
	 * @param cbpaidProduct $plan
	 * @return bool
	 */
	public static function hasPlanPrices( cbpaidProduct $plan ): bool
	{
		if ( ! $plan->getParam( 'options_prices_enabled', 0, 'integrations' ) ) {
			return false;
		}

		if ( ! ( $plan instanceof cbpaidProductUsersubscription ) ) {
			return false;
		}

		return true;
	}

	/**
	 * @param cbpaidProduct $plan
	 * @return bool
	 */
	public static function hasPlanOptions( cbpaidProduct $plan ): bool
	{
		if ( ! $plan->getParam( 'options_enabled', 0, 'integrations' ) ) {
			return false;
		}

		if ( $plan instanceof cbpaidProductDonation ) {
			return false;
		}

		return true;
	}

	/**
	 * @param cbpaidProduct        $orgPlan
	 * @param null|cbpaidSomething $orgSubscription
	 * @param null|PriceEntity     $selectedPlanPrice
	 * @param bool                 $triggers
	 * @return string
	 */
	public static function renderPlanPeriodPrice( cbpaidProduct $orgPlan, ?cbpaidSomething $orgSubscription = null, ?PriceEntity $selectedPlanPrice = null, bool $triggers = false ): string
	{
		global $_CB_framework;

		// We need a new instance that's not connected to the plans manager as we do not want to alter the original plan at this time
		if ( $orgPlan instanceof cbpaidProductMerchandise ) {
			$plan			=	new cbpaidProductMerchandise();
		} else {
			$plan			=	new cbpaidProductUsersubscription();
		}

		$plan->load( $orgPlan->getInt( 'id', 0 ) );

		self::setBasePlanPrice( $plan, $orgSubscription );

		if ( $selectedPlanPrice ) {
			self::setPlanOverrides( $plan, $selectedPlanPrice );
		}

		self::setPlanOverrides( $plan, self::getSelectedPrice( $plan, $orgSubscription, $selectedPlanPrice ) );

		if ( ! $triggers ) {
			$plan->id		=	0; // this will cause displayPeriodPrice triggers to be skipped
		}

		if ( $orgSubscription ) {
			if ( $orgSubscription->realStatus( $_CB_framework->now() ) === 'C' ) {
				$reason		=	'N';
			} else {
				$reason		=	'R';
			}

			$now			=	$_CB_framework->now();
			$expiryTime		=	$orgSubscription->computeStartTimeIfActivatedNow( $now, $reason );
			$price			=	$plan->displayPeriodPrice( $orgSubscription->getInt( 'user_id', 0 ), $reason, $orgSubscription->getString( 'status', '' ), ( $orgSubscription->getOccurrence() + 1 ), null, $expiryTime, true );
		} else {
			$myId			=	Application::MyUser()->getUserId();
			$price			=	$plan->displayPeriodPrice( $myId, ( $myId ? 'U' : 'N' ) );
		}

		$plan->set( '_displayPeriodPriceRecursionsLimiter', 1 );

		if ( $orgSubscription ) {
			$orgSubscription->set( '_displayPeriodPriceRecursionsLimiter', 1 );
		}

		return $price;
	}

	/**
	 * @param cbpaidProduct        $plan
	 * @param null|cbpaidSomething $subscription
	 * @return PriceEntity[]
	 */
	public static function getPlanPrices( cbpaidProduct $plan, ?cbpaidSomething $subscription = null ): array
	{
		if ( ! self::hasPlanPrices( $plan ) ) {
			return [];
		}

		$options			=	[];

		foreach ( new Registry( $plan->getParam( 'options_prices', '', 'integrations' ) ) as $option ) {
			/** @var ParamsInterface $option */
			$name			=	$option->getString( 'name', '' );

			if ( $name === '' ) {
				continue;
			}

			$options[$name]	=	new PriceEntity( $plan, $option, $subscription );
		}

		return $options;
	}

	/**
	 * @param cbpaidProduct        $plan
	 * @param null|cbpaidSomething $subscription
	 * @param null|string          $selected
	 * @return string
	 */
	public static function renderPlanPrices( cbpaidProduct $plan, ?cbpaidSomething $subscription = null, ?string $selected = null ): string
	{
		if ( ! self::hasPlanPrices( $plan ) ) {
			return '';
		}

		if ( ( ! $selected ) && $subscription ) {
			$selected			=	$subscription->getParams( 'integrations' )->getRaw( 'planprices', '' );
		}

		$type					=	$plan->getParam( 'options_prices_type', 'select', 'integrations' );
		$prices					=	[];
		$orgPrice				=	self::renderPlanPeriodPrice( $plan, $subscription, null, true );

		if ( $type !== 'radio' ) {
			$orgPrice			=	trim( \strip_tags( $orgPrice ) );
		}

		$prices[]				=	moscomprofilerHTML::makeOption( '', $orgPrice );
		$renderedPrices			=	[];

		foreach ( self::getPlanPrices( $plan, $subscription ) as $price ) {
			$renderedPrice		=	$price->renderPrice( ( $type === 'radio' ) );

			if ( ( $orgPrice === $renderedPrice ) || \in_array( $renderedPrice, $renderedPrices, true ) ) {
				continue;
			}

			$renderedPrices[]	=	$renderedPrice;

			$prices[]			=	moscomprofilerHTML::makeOption( $price->getName(), $renderedPrice );
		}

		if ( ! $prices ) {
			return '';
		}

		$name					=	'planprices[' . $plan->getInt( 'id', 0 ) . ']';
		$hidden					=	'<input type="hidden" name="changeplanprices[' . $plan->getInt( 'id', 0 ) . ']" value="1" class="cbsubsPlanPricesInput cbsubsPlanPricesAvailable" />';

		if ( $type === 'radio' ) {
			return moscomprofilerHTML::radioListTable( $prices, $name, '', 'value', 'text', ( $selected ?? 0 ), 1, 1, 0, false, [ 'cbsubsPlanPricesSelect' ], '', false ) . $hidden;
		}

		return moscomprofilerHTML::selectList( $prices, $name, 'class="w-auto form-control cbsubsPlanPricesSelect"', 'value', 'text', $selected, false, false, false ) . $hidden;
	}

	/**
	 * @param cbpaidProduct        $plan
	 * @param null|cbpaidSomething $subscription
	 * @return null|PriceEntity
	 */
	public static function getSelectedPlanPrice( cbpaidProduct $plan, ?cbpaidSomething $subscription = null ): ?PriceEntity
	{
		if ( ! self::hasPlanPrices( $plan ) ) {
			return null;
		}

		// This is necessary for handle cases like renewing the plan without changing the price as in that situation we do want to fallback to existing
		if ( Application::Input()->getBool( 'changeplanprices.' . $plan->getInt( 'id', 0 ), false ) ) {
			$value	=	Application::Input()->getString( 'planprices.' . $plan->getInt( 'id', 0 ), '' );

			if ( $subscription && ( ! $plan->getParam( 'options_prices_changeable', 1, 'integrations' ) ) ) {
				return ( self::getPlanPrices( $plan, $subscription )[$subscription->getParams( 'integrations' )->getString( 'planprices', '' )] ?? null );
			}

			return ( self::getPlanPrices( $plan, $subscription )[$value] ?? null );
		}

		if ( $subscription ) {
			return ( self::getPlanPrices( $plan, $subscription )[$subscription->getParams( 'integrations' )->getString( 'planprices', '' )] ?? null );
		}

		return null;
	}

	/**
	 * Returns an array of available plan options
	 *
	 * @param cbpaidProduct        $plan
	 * @param null|cbpaidSomething $subscription
	 * @param null|PriceEntity     $selectedPlanPrice
	 * @return OptionEntity[]
	 */
	public static function getPlanOptions( cbpaidProduct $plan, ?cbpaidSomething $subscription = null, ?PriceEntity $selectedPlanPrice = null ): array
	{
		if ( ! self::hasPlanOptions( $plan ) ) {
			return [];
		}

		$options			=	[];

		foreach ( new Registry( $plan->getParam( 'options_options', '', 'integrations' ) ) as $option ) {
			/** @var ParamsInterface $option */
			$name			=	$option->getString( 'name', '' );

			if ( $name === '' ) {
				continue;
			}

			$options[$name]	=	new OptionEntity( $plan, $option, $subscription, $selectedPlanPrice );
		}

		return $options;
	}

	/**
	 * Returns an array of options that are missing from selection and are marked required
	 *
	 * @param cbpaidProduct            $plan
	 * @param null|cbpaidSomething     $subscription
	 * @return OptionEntity[]
	 */
	public static function getMissingOptions( cbpaidProduct $plan, ?cbpaidSomething $subscription = null ): array
	{
		if ( ! self::hasPlanOptions( $plan ) ) {
			return [];
		}

		$options					=	self::getPlanOptions( $plan, $subscription );
		$selected					=	[];

		foreach ( self::getSelectedValues( $plan, $subscription ) as $value ) {
			$option					=	$value->getOption()->getName();

			if ( \in_array( $option, $selected, true ) ) {
				continue;
			}

			$selected[]				=	$option;
		}

		$missing					=	[];

		foreach ( $options as $name => $option ) {
			if ( ! $option->isRequired() ) {
				continue;
			}

			if ( ! \in_array( $name, $selected, true ) ) {
				$missing[$name]		=	$option;
			}
		}

		return $missing;
	}

	/**
	 * Returns an array of options that have been selected for the subscription, basket, or new subscription
	 *
	 * @param cbpaidProduct        $plan
	 * @param null|cbpaidSomething $subscription
	 * @param null|PriceEntity     $selectedPlanPrice
	 * @return ValueEntity[]
	 */
	public static function getSelectedValues( cbpaidProduct $plan, ?cbpaidSomething $subscription = null, ?PriceEntity $selectedPlanPrice = null ): array
	{
		if ( ! self::hasPlanOptions( $plan ) ) {
			return [];
		}

		// This is necessary for handle cases like renewing the plan without changing any options as in that situation we do want to fallback to existing
		if ( Application::Input()->getBool( 'changeplanoptions.' . $plan->getInt( 'id', 0 ), false ) ) {
			$values		=	Application::Input()->subTree( 'planoptions.' . $plan->getInt( 'id', 0 ) )->asArray();

			if ( $subscription && ( ! $plan->getParam( 'options_changeable', 1, 'integrations' ) ) ) {
				return self::validateValues( self::getPlanOptions( $plan, $subscription, $selectedPlanPrice ), $subscription->getParams( 'integrations' )->subTree( 'planoptions' )->asArray() );
			}

			return self::validateValues( self::getPlanOptions( $plan, $subscription, $selectedPlanPrice ), $values );
		}

		if ( $subscription ) {
			return self::validateValues( self::getPlanOptions( $plan, $subscription, $selectedPlanPrice ), $subscription->getParams( 'integrations' )->subTree( 'planoptions' )->asArray() );
		}

		return [];
	}

	/**
	 * Returns the calculated first_rate and rate based off selected options
	 *
	 * @param cbpaidProduct        $plan
	 * @param null|cbpaidSomething $subscription
	 * @param null|PriceEntity     $selectedPlanPrice
	 * @return array
	 */
	public static function getSelectedPrice( cbpaidProduct $plan, ?cbpaidSomething $subscription = null, ?PriceEntity $selectedPlanPrice = null ): array
	{
		if ( ! self::hasPlanOptions( $plan ) ) {
			return [];
		}

		$selected		=	self::getSelectedValues( $plan, $subscription, $selectedPlanPrice );

		if ( ! $selected ) {
			return [];
		}

		$firstRate		=	0.0;
		$rate			=	0.0;

		foreach ( $selected as $value ) {
			$firstRate	+=	$value->getFirstRate();
			$rate		+=	$value->getRate();
		}

		return [ 'first_rate' => $firstRate, 'rate' => $rate ];
	}

	/**
	 * @param cbpaidPaymentBasket $paymentBasket
	 * @param cbpaidSomething     $subscription
	 * @return PaidEntity[]
	 */
	public static function getPaidBasketValues( cbpaidPaymentBasket $paymentBasket, cbpaidSomething $subscription ): array
	{
		$integrationParams	=	$paymentBasket->getParams( 'integrations' );

		if ( ! $integrationParams->has( 'planoptionspaid' ) ) {
			return [];
		}

		$selected			=	[];

		foreach ( $integrationParams->subTree( 'planoptionspaid.' . $subscription->getPlan()->getInt( 'id', 0 ) )->asArray() as $option ) {
			$values			=	( $option['values'] ?? [] );

			if ( ! $values ) {
				continue;
			}

			$title			=	( $option['title'] ?? '' );

			foreach ( $values as $value ) {
				$selected[]	=	new PaidEntity( $title, $value, $subscription );
			}
		}

		return $selected;
	}

	/**
	 * @param cbpaidPaymentBasket $paymentBasket
	 * @param cbpaidSomething     $subscription
	 * @return null|PriceEntity
	 */
	public static function getBasketPlanPrice( cbpaidPaymentBasket $paymentBasket, cbpaidSomething $subscription ): ?PriceEntity
	{
		$integrationParams	=	$paymentBasket->getParams( 'integrations' );

		if ( ! $integrationParams->has( 'planprices' ) ) {
			return null;
		}

		$plan				=	$subscription->getPlan();
		$price				=	$integrationParams->getString( 'planprices.' . $plan->getInt( 'id', 0 ), '' );

		if ( $price === '' ) {
			return null;
		}

		return ( self::getPlanPrices( $plan, $subscription )[$price] ?? null );
	}

	/**
	 * @param cbpaidPaymentBasket $paymentBasket
	 * @param cbpaidSomething     $subscription
	 * @param null|PriceEntity    $selectedPlanPrice
	 * @return ValueEntity[]
	 */
	public static function getBasketValues( cbpaidPaymentBasket $paymentBasket, cbpaidSomething $subscription, ?PriceEntity $selectedPlanPrice = null ): array
	{
		$integrationParams	=	$paymentBasket->getParams( 'integrations' );

		if ( ! $integrationParams->has( 'planoptions' ) ) {
			return [];
		}

		$plan				=	$subscription->getPlan();
		$selectedOptions	=	$integrationParams->subTree( 'planoptions.' . $plan->getInt( 'id', 0 ) )->asArray();

		return self::validateValues( self::getPlanOptions( $plan, $subscription, $selectedPlanPrice ), $selectedOptions );
	}

	/**
	 * Returns the calculated first_rate and rate based off basket options
	 *
	 * @param cbpaidPaymentBasket $paymentBasket
	 * @param cbpaidSomething     $subscription
	 * @param null|PriceEntity    $selectedPlanPrice
	 * @return array
	 */
	public static function getBasketPrice( cbpaidPaymentBasket $paymentBasket, cbpaidSomething $subscription, ?PriceEntity $selectedPlanPrice = null ): array
	{
		$selected		=	self::getBasketValues( $paymentBasket, $subscription, $selectedPlanPrice );

		if ( ! $selected ) {
			return [];
		}

		$firstRate		=	0.0;
		$rate			=	0.0;

		foreach ( $selected as $value ) {
			$firstRate	+=	$value->getFirstRate();
			$rate		+=	$value->getRate();
		}

		return [ 'first_rate' => $firstRate, 'rate' => $rate ];
	}

	/**
	 * Gets the base plan price based off configuration
	 * This will apply plan price override behavior if enabled
	 *
	 * @param cbpaidProduct        $plan
	 * @param null|cbpaidSomething $subscription
	 * @return array
	 */
	public static function getBasePlanPrice( cbpaidProduct $plan, ?cbpaidSomething $subscription = null ): array
	{
		$prices					=	[	'first_rate'	=>	(float) sprintf( '%.2f', $plan->first_rate ),
										'rate'			=>	(float) sprintf( '%.2f', $plan->rate ),
									];

		if ( ! $plan->getParam( 'options_price_enabled', 0, 'integrations' ) ) {
			return $prices;
		}

		$params					=	[	'first_rate'	=>	$plan->getParam( 'options_price_first_rate', '', 'integrations' ),
										'rate'			=>	$plan->getParam( 'options_price_rate', '', 'integrations' ),
									];

		foreach ( $params as $name => $override ) {
			if ( $override === '' ) {
				continue;
			}

			if ( $subscription ) {
				$price			=	$subscription->getPersonalized( $override, false, false, null, false );
			} else {
				$price			=	$plan->getPersonalized( $override, Application::MyUser()->getUserId(), false, false, null, false );
			}

			if ( $price === '' ) {
				continue;
			}

			$parts				=	[];

			preg_match( '#([+*/-])?([\d.]+)(%)?$#', $price, $parts );

			$amount				=	(float) ( $parts[2] ?? 0.0 );

			if ( ( $parts[3] ?? '' ) === '%' ) {
				$amount			=	( $prices[$name] * ( $amount / 100 ) );
			}

			switch ( ( $parts[1] ?? '' ) ) {
				case '+':
					$amount		=	( $prices[$name] + $amount );
					break;
				case '-':
					$amount		=	( $prices[$name] - $amount );
					break;
				case '*':
					$amount		=	( $prices[$name] * $amount );
					break;
				case '/':
					$amount		=	( $prices[$name] / $amount );
					break;
			}

			$prices[$name]		=	$amount;
		}

		return $prices;
	}

	/**
	 * Sets the base plan price based off configuration
	 * Completely ignored if plan price override behavior is not enabled
	 *
	 * @param cbpaidProduct        $plan
	 * @param null|cbpaidSomething $subscription
	 * @return void
	 */
	public static function setBasePlanPrice( cbpaidProduct $plan, ?cbpaidSomething $subscription = null ): void
	{
		if ( ! $plan->getParam( 'options_price_enabled', 0, 'integrations' ) ) {
			return;
		}

		self::setPlanOverrides( $plan, self::getBasePlanPrice( $plan, $subscription ) );
	}

	/**
	 * Sets plan overrides whether it's a static price from options or PriceEntity
	 * This is purposely a centralized function so we can globally adjust override behavior if needed
	 *
	 * @param cbpaidProduct     $plan
	 * @param array|PriceEntity $price
	 * @return void
	 */
	public static function setPlanOverrides( cbpaidProduct $plan, $price ): void
	{
		if ( ! $price ) {
			return;
		}

		if ( $price instanceof PriceEntity ) {
			$plan->setOverride( 'first_different', $price->getFirstDifferent() );
			$plan->setOverride( 'first_rate', $price->getFirstRate() );
			$plan->setOverride( 'first_validity', $price->getFirstValidity() );
			$plan->setOverride( 'first_calstart', $price->getFirstCalStart() );
			$plan->setOverride( 'rate', $price->getRate() );
			$plan->setOverride( 'validity', $price->getValidity() );
			$plan->setOverride( 'calstart', $price->getCalStart() );
		} else {
			if ( $plan->getInt( 'first_different', 0 ) ) {
				$plan->setOverride( 'first_rate', $price['first_rate'] );
			}

			$plan->setOverride( 'rate', $price['rate'] );
		}
	}

	/**
	 * @param OptionEntity[] $options
	 * @param array          $values
	 * @return ValueEntity[]
	 */
	private static function validateValues( array $options, array $values ): array
	{
		$valid					=	[];

		foreach ( $values as $name => $value ) {
			if ( ! \array_key_exists( $name, $options ) ) {
				continue;
			}

			if ( \is_array( $value ) ) {
				foreach ( $value as $multipleValue ) {
					if ( $multipleValue === '' ) {
						continue;
					}

					if ( ! $options[$name]->hasValue( $multipleValue ) ) {
						continue;
					}

					$valid[]	=	$options[$name]->getValue( $multipleValue );
				}

				continue;
			}

			if ( $value === '' ) {
				continue;
			}

			if ( ! $options[$name]->hasValue( $value ) ) {
				continue;
			}

			$valid[]			=	$options[$name]->getValue( $value );
		}

		return $valid;
	}
}
