<?php
/**
* @version $Id: cbsubs.mailer.php 1608 2012-12-29 04:12:52Z beat $
* @package CBSubs (TM) Community Builder Plugin for Paid Subscriptions (TM)
* @subpackage Plugin for Paid Subscriptions
* @copyright (C) 2007-2022 and Trademark of Lightning MultiCom SA, Switzerland - www.joomlapolis.com - and its licensors, all rights reserved
* @license http://www.gnu.org/licenses/old-licenses/gpl-2.0.html GNU/GPL version 2
*/

use CB\Database\Table\UserTable;
use CBLib\Application\Application;
use CBLib\Database\DatabaseDriverInterface;
use CBLib\Database\Table\TableInterface;
use CBLib\Registry\ParamsInterface;
use CBLib\Language\CBTxt;

/** ensure this file is being included by a parent file */
if ( ! ( defined( '_VALID_CB' ) || defined( '_JEXEC' ) || defined( '_VALID_MOS' ) ) ) { die( 'Direct Access to this location is not allowed.' ); }

global $_CB_framework, $_PLUGINS;

/** @noinspection PhpIncludeInspection */
include_once( $_CB_framework->getCfg( 'absolute_path' ) . '/components/com_comprofiler/plugin/user/plug_cbpaidsubscriptions/cbpaidsubscriptions.class.php');

/**
 * Mailer settings model class
 */
class cbpaidMailerAutomessage extends cbpaidTable {
	public $id;								// sql:int(11)
	public $name;							// sql:varchar(64)
	public $mail_type;						// sql:varchar(8)  email,pm
	public $emailsubject;					// sql:mediumtext
	public $emailbody;						// sql:mediumtext
	public $emailapplytemplate;				// sql:tinyint(4) default="1"
	public $emailhtml;						// sql:tinyint(4) default="0"
	public $emailcc;						// sql:varchar(255)
	public $emailbcc;						// sql:varchar(255)
	public $emailfromname;					// sql:varchar(255)
	public $emailfromemail;					// sql:varchar(255)
	public $emailattachments;				// sql:text
	public $priority;						// sql:int(11)
	public $trigger_date_diff_a;			// sql:varchar(21)
	public $trigger_date_diff_b;			// sql:varchar(21)
	public $trigger_date_reference;			// sql:varchar(21)
	public $plans_applied_to;				// sql:varchar(1024)
	public $plans_status;					// sql:varchar(64)
	public $max_uses_total;					// sql:int(11)
	public $max_uses_per_customer;			// sql:int(11)
	public $max_uses_per_customer_interval;	// sql:varchar(19)
	public $buyer_geo_zone_id;				// sql:int(20)
	public $applies_to_business_consumer;	// sql:char(1) default="A" B,C
	public $cond_1_operator;				// sql:varchar(6)
	public $cond_1_plans_required;			// sql:varchar(1024)
	public $cond_1_plans_status;			// sql:varchar(40)
	public $cond_1_purchase_ok;				// sql:tinyint(4)
	public $cond_1_subscription_conditions;			// sql:tinyint(4)
	public $cond_1_subscription_autorecurring;		// sql:tinyint(4)	// 0: do not care, 1: yes, 2: no
	public $cond_1_subscription_regular_recurrings_used_min;		// sql:int(11)
	public $cond_1_subscription_regular_recurrings_used_max;		// sql:int(11)
	public $cond_1_basket_conditions;				// sql:tinyint(4)
	public $cond_1_basket_currencies;				// sql:varchar(1200)
	public $cond_1_basket_gatewayaccounts;			// sql:varchar(128)
	public $cond_1_basket_payment_methods;			// sql:varchar(256)
	public $cond_1_basket_payment_types;			// sql:varchar(256)
	public $cond_1_basket_autorecurring;			// sql:tinyint(4)	// 0: do not care, 1: yes, 2: no
	public $cond_1_basket_taxed;					// sql:tinyint(4)	// 0: do not care, 1: yes, 2: no
	public $cond_1_basket_address_country_codes;	// sql:varchar(1400)
	public $cond_1_date_1;					// sql:varchar(20)
	public $cond_1_date_cbfield_1;			// sql:int(11)
	public $cond_1_value_1;					// sql:varchar(1024)
	public $cond_1_dates_diff_a;			// sql:varchar(21)
	public $cond_1_dates_diff_b;			// sql:varchar(21)
	public $cond_1_date_2;					// sql:varchar(20)
	public $cond_1_date_cbfield_2;			// sql:int(11)
	public $cond_1_value_2;					// sql:varchar(1024)
	public $cond_2_operator;				// sql:varchar(6)
	public $cond_2_plans_required;			// sql:varchar(1024)
	public $cond_2_plans_status;			// sql:varchar(40)
	public $cond_2_purchase_ok;				// sql:tinyint(4)
	public $cond_2_subscription_conditions;			// sql:tinyint(4)
	public $cond_2_subscription_autorecurring;		// sql:tinyint(4)	// 0: do not care, 1: yes, 2: no
	public $cond_2_subscription_regular_recurrings_used_min;		// sql:int(11)
	public $cond_2_subscription_regular_recurrings_used_max;		// sql:int(11)
	public $cond_2_basket_conditions;				// sql:tinyint(4)
	public $cond_2_basket_currencies;				// sql:varchar(1200)
	public $cond_2_basket_gatewayaccounts;			// sql:varchar(128)
	public $cond_2_basket_payment_methods;			// sql:varchar(256)
	public $cond_2_basket_payment_types;			// sql:varchar(256)
	public $cond_2_basket_autorecurring;			// sql:tinyint(4)	// 0: do not care, 1: yes, 2: no
	public $cond_2_basket_taxed;					// sql:tinyint(4)	// 0: do not care, 1: yes, 2: no
	public $cond_2_basket_address_country_codes;	// sql:varchar(1400)
	public $cond_2_date_1;					// sql:varchar(20)
	public $cond_2_date_cbfield_1;			// sql:int(11)
	public $cond_2_value_1;					// sql:varchar(1024)
	public $cond_2_dates_diff_a;			// sql:varchar(21)
	public $cond_2_dates_diff_b;			// sql:varchar(21)
	public $cond_2_date_2;					// sql:varchar(20)
	public $cond_2_date_cbfield_2;			// sql:int(11)
	public $cond_2_value_2;					// sql:varchar(1024)
	public $published;						// sql:tinyint(4)
	public $start_date;						// sql:datetime null="false"
	public $stop_date;						// sql:datetime null="false"
	public $storemessagebody;				// sql:tinyint(4)
	public $applies_to_blocked_users;		// sql:int(11)
	public $viewaccesslevel	=	1;			// sql:int(11)
	public $usergroups;						// sql:varchar(1024)
	public $user_ids;						// sql:varchar(1024)
	public $ordering;						// sql:int(11)
	public $owner	=	0;					// sql:int(11)
	public $params;							// sql:text
	public $integrations;					// sql:text

	/**
	 * Constructor
	 *
	 * @param  DatabaseDriverInterface  $db
	 */
	public function __construct( &$db = null ) {
		parent::__construct( '#__cbsubs_mailer_automessages', 'id', $db );
		$this->_historySetLogger();
	}

	/**
	 * Singletons loader
	 *
	 * @param  boolean  $isPublished
	 * @return cbpaidMailerAutomessage[]
	 */
	public static function getInstances( $isPublished = true ) {
		static $cache				=	array();
		if ( ! isset( $cache[$isPublished] ) ) {
			$conditions				=	$isPublished ? array( 'published' => 1 ) : array();
			$me						=	new self();
			$cache[$isPublished]	=	$me->loadThisMatchingList( $conditions, array( 'priority' => 'ASC', 'ordering' => 'ASC' ) );
		}
		return $cache[$isPublished];
	}
	/**
	 * Singleton loader
	 *
	 * @param  int  $id
	 * @return cbpaidMailerAutomessage|boolean  False if not existent
	 */
	public static function getInstance( $id ) {
		$instances					=	self::getInstances();
		if ( isset( $instances[$id] ) ) {
			return $instances[$id];
		}
		return false;
	}
	/**
	 * loads and returns an array of geoZone ids to which a country/province/zip is part of
	 *
	 * @param  string  $buyerCountry
	 * @param  string  $buyerProvince
	 * @param  string  $buyerZip
	 * @return int[]
	 */
	private function _loadGeoZonesOfBuyer( $buyerCountry, $buyerProvince, $buyerZip ) {
		/* Query example:
			SELECT b.id FROM #__cbsubs_geo_zones b
			JOIN #__cbsubs_geo_zones_entries be ON
			     be.geo_zone_id = b.id
			 AND be.`country_iso_code2` = 'CH'
			 AND ( be.`province_iso_code` = '' OR be.`province_iso_code` = 'CH-VD' )
			 AND ( be.`zip_code_condition` = 0 OR ( be.`zip_code_condition` = 1 AND '1010' REGEXP be.`zip_code_regexp` ) OR ( be.`zip_code_condition` = 2 AND be.`zip_code_min` <= 1010 AND be.`zip_code_max` >= 1010 ) )
		 */
		$sql			=	'SELECT DISTINCT b.id FROM ' . $this->_db->NameQuote( '#__cbsubs_geo_zones' ) . ' AS b'
						.	"\n JOIN " . $this->_db->NameQuote( '#__cbsubs_geo_zones_entries' ) . ' AS be'
						.	"\n			  ON be.geo_zone_id = b.id"
						.	"\n 		 AND be." . $this->_db->NameQuote( 'country_iso_code2' ) . ' = ' . $this->_db->Quote( $buyerCountry )
						.	"\n 		 AND ( be." . $this->_db->NameQuote( 'province_iso_code' ) . ' = ' . $this->_db->Quote( '' )
						.	"\n 		    OR be." . $this->_db->NameQuote( 'province_iso_code' ) . ' = ' . $this->_db->Quote( $buyerProvince ) . ' )'
						.	"\n 		 AND ( be." . $this->_db->NameQuote( 'zip_code_condition' ) . ' = 0'
						.	"\n 		  OR ( be." . $this->_db->NameQuote( 'zip_code_condition' ) . ' = 1 AND ' . $this->_db->Quote( $buyerZip ) . ' REGEXP be.' . $this->_db->NameQuote( 'zip_code_regexp' ) . ' )'
						.	"\n 		  OR ( be." . $this->_db->NameQuote( 'zip_code_condition' ) . ' = 2'
						.	"\n 		 						 AND be." . $this->_db->NameQuote( 'zip_code_min' ) . ' <= ' . (int) $buyerZip
						.	"\n 		 						 AND be." . $this->_db->NameQuote( 'zip_code_max' ) . ' >= ' . (int) $buyerZip . ' ) )';
		$this->_db->setQuery( $sql );
		return $this->_db->loadResultArray();
	}

	/**
	 * loads a cache and returns an array of geoZone ids to which a country/province/zip is part of
	 *
	 * @param  string  $buyerCountry
	 * @param  string  $buyerProvince
	 * @param  string  $buyerZip
	 * @return int[]
	 */
	protected function getGeoZonesOfBuyer( $buyerCountry, $buyerProvince, $buyerZip ) {
		static $geoZones	=	array();
		if ( ! isset( $geoZones[$buyerCountry][$buyerProvince][$buyerZip] ) ) {
			$geoZones[$buyerCountry][$buyerProvince][$buyerZip]		=	$this->_loadGeoZonesOfBuyer( $buyerCountry, $buyerProvince, $buyerZip );
		}
		return $geoZones[$buyerCountry][$buyerProvince][$buyerZip];
	}
	/**
	 * Checks if $viewaccesslevel is included in view access levels of $userId
	 *
	 * @param  int      $userId
	 * @param  int      $viewaccesslevel
	 * @return boolean
	 */
	protected static function _checkUserViewAccessLevels( $userId, $viewaccesslevel ) {
		return Application::User( (int) $userId )->canViewAccessLevel( (int) $viewaccesslevel );
	}
	/**
	 * Checks if any of $userId's groups is within $allowedGroups
	 *
	 * @param  int    $userId
	 * @param  int[]  $allowedGroups
	 * @return boolean
	 */
	protected static function _checkUserGroupsInAllowedGroups( $userId, $allowedGroups ) {
		global $_CB_framework;

		$user				=	CBuser::getUserDataInstance( $userId );

		$usersGids			=	$user->gids;
		if ( ( ! $userId ) && ! $usersGids ) {
			$usersGids		=	$_CB_framework->acl->mapGroupNamesToValues( array( 'Public' ) );
		}

		if ( $user->id ) {
			// add registered pseudo-group:
			array_unshift( $usersGids, -1 );
		}

		// add users pseudo-group:
		array_unshift( $usersGids, -2 );

		$allowedGroupsAndChildren	=	array();
		foreach ( $allowedGroups as $grp ) {
			$allowedGroupsAndChildren	=	array_merge( $allowedGroupsAndChildren, $_CB_framework->acl->get_group_parent_ids( $grp ) );
		}
		return ( count( array_intersect( $allowedGroupsAndChildren, $usersGids ) ) > 0 );
	}
	/**
	 * Checks if promotion applies for the maximum uses in total as well as for $userId
	 *
	 * @param  int      $userId
	 * @return boolean
	 */
	protected function checkIfMaxDiscountUsesNotReached( $userId ) {
		if ( $this->max_uses_total ) {
			if ( ( cbpaidMailerMailQueue::countMailsInTable( $this->id ) + cbpaidMailerMessage::countMailsInTable( $this->id ) ) >= $this->max_uses_total ) {
				return false;
			}
		}
		if ( $this->max_uses_per_customer ) {
			if ( $this->max_uses_per_customer_interval ) {
				$sinceDateTime	=	$this->_intervalAgo( $this->max_uses_per_customer_interval );
			} else {
				$sinceDateTime	=	null;
			}
			if ( ( cbpaidMailerMailQueue::countMailsInTable( $this->id, $userId, $sinceDateTime ) + cbpaidMailerMessage::countMailsInTable( $this->id, $userId, $sinceDateTime ) ) >= $this->max_uses_per_customer ) {
				return false;
			}
		}
		return true;
	}
	/**
	 * Finds the UTC start time back from now by $interval
	 *
	 * @param  string       $interval  E.g. '0001-00-00 00:00:00'
	 * @return string|null             E.g. '2011-02-03 04:05:06'
	 */
	protected function _intervalAgo( $interval )
	{
		if ( $interval == '0000-00-00 00:00:00' ) {
			return null;
		}

		$systemTimeZone	=	new DateTimeZone( cbpaidTimes::getInstance()->systemTimeZone() );
		$startDate		=	new DateTime( '@' . cbpaidTimes::getInstance()->startTime(), $systemTimeZone );
		$startDate->setTimezone( $systemTimeZone );

		$dateInterval	=	cbpaidTimes::getInstance()->dateInterval( $interval );
		$startDate->sub( $dateInterval );
		$expiryTime		=	$startDate->getTimestamp();

		return cbpaidTimes::getInstance()->getUtcDateOfTime( $expiryTime );
	}
	/**
	 * checks in a cache if buyer country/province/zip is part of $this->buyer_geo_zone_id
	 *
	 * @param  string  $buyerCountry
	 * @param  string  $buyerProvince
	 * @param  string  $buyerZip
	 * @return boolean
	 */
	protected function checkBuyerGeoZone( $buyerCountry, $buyerProvince, $buyerZip ) {
		return ( $this->buyer_geo_zone_id == 0 ) || in_array( $this->buyer_geo_zone_id, $this->getGeoZonesOfBuyer( $buyerCountry, $buyerProvince, $buyerZip ) );
	}
	/**
	 * checks if the payment basket is_business '1' or '0' matches the $this->applies_to_business_consumer A,B,C rule
	 *
	 * @param  string|null  $is_business
	 * @return boolean
	 */
	protected function checkAppliesToBusinessCondition( $is_business ) {
		return ( $this->applies_to_business_consumer == 'A' ) || ( ( $is_business === '1' ) && ( $this->applies_to_business_consumer == 'B' ) ) || ( ( $is_business !== '1' ) && ( $this->applies_to_business_consumer == 'C' ) );
	}
	/**
	 * Checks if $this promotion applies to a given $planId
	 *
	 * @param  int      $planId
	 * @return boolean
	 */
	protected function checkIfAppliesToPlanId( $planId ) {
		return ( ( $planId == null) || ( $this->plans_applied_to == '' ) || in_array( $planId, explode( '|*|', $this->plans_applied_to ) ) );
	}
	/**
	 * Checks if $this promotion applies based on $userId 's existing subscriptions and CB fields conditions 1 and 2.
	 *
	 * @param  int                       $userId
	 * @param  array                     $resultTexts    (returned appended)
	 * @param  cbpaidPaymentBasket|null  $paymentBasket
	 * @return boolean
	 */
	protected function checkActiveConditions( $userId, &$resultTexts, $paymentBasket ) {
		if ( $this->cond_1_operator ) {
			$r							=	cbpaidCondition::checkConditionsOfObject( $this, $userId, $resultTexts, $paymentBasket );
		} else {
			$r							=	true;
		}
		return $r;
	}
	/**
	 * Checks if the published status and published dates match $now time
	 *
	 * @param  int  $now  Unix-Time
	 * @return boolean
	 */
	protected function checkPublishedAndDates( $now ) {
		if ( $this->published ) {
			$nullDate			=	$this->_db->getNullDate( 'date' );
			if ( substr( $this->start_date, 0, 10 ) != $nullDate ) {
				$nowDate		=	cbpaidTimes::getInstance()->getUtcDateOfTime( $now );
				if ( $this->start_date > $nowDate ) {
					return false;
				}
			}
			if ( substr( $this->stop_date, 0, 10 ) != $nullDate ) {
				if ( ! isset( $nowDate ) ) {
					$nowDate	=	cbpaidTimes::getInstance()->getUtcDateOfTime( $now );
				}
				if ( $this->stop_date <= $nowDate ) {
					return false;
				}
			}
			return true;
		}
		return false;
	}
	/**
	 * Checks if promotion view access level is included in view access levels of $userId
	 *
	 * @param  int      $userId
	 * @return boolean
	 */
	protected function checkViewAccessLevel( $userId ) {
		return self::_checkUserViewAccessLevels( $userId, $this->viewaccesslevel );
	}
	/**
	 * Checks if promotion applies to groups to which $userId belongs
	 *
	 * @param  int      $userId
	 * @return boolean
	 */
	protected function checkUsergroups( $userId ) {
		return self::_checkUserGroupsInAllowedGroups( $userId, explode( '|*|', $this->usergroups ) );
	}
	/**
	 * Checks if promotion applies to groups to which $userId belongs
	 *
	 * @param  int      $userId
	 * @return boolean
	 */
	protected function checkUserId( $userId ) {
		return ( $this->user_ids === '' ) || ( $userId && in_array( (string) $userId, explode( ',', $this->user_ids ) ) );
	}
	/**
	 * Checks if promotion applies to the enabled/blocked state of the $userId (if guest returns true)
	 *
	 * @param  int      $userId
	 * @return boolean
	 */
	protected function checkUserEnabled( $userId ) {
		if ( $userId == 0 ) {
			return true;
		}

		$user				=	CBuser::getUserDataInstance( $userId );

		if ( ! $user) {
			return false;
		}

		switch ( $this->applies_to_blocked_users ) {
			case 1:
				// Only to enabled users:
				return ! $user->block;
			case 2:
				// Only to blocked users:
				return (bool) $user->block;
			case 3:
				// To all users:
				return true;
		}
		return false;
	}
	/**
	 * Checks if the owner of the promo corresponds to the $owner
	 *
	 * @param  int     $owner
	 * @return boolean
	 */
	protected function checkOwner( $owner ) {
		return ( $this->owner == 0 ) || ( $this->owner == $owner );
	}
	/**
	 * Check if a promotion is applicable, and if net returns the reason in $resultTexts
	 *
	 * @param  int                      $userId
	 * @param  int|null                 $planId
	 * @param  boolean                  $is_business
	 * @param  string                   $buyerCountry
	 * @param  string                   $buyerProvince
	 * @param  string                   $buyerZip
	 * @param  int                      $sellerOwnerId
	 * @param  cbpaidPaymentBasket|null $paymentBasket
	 * @param  int                      $now
	 * @param  boolean                  $isPublished
	 * @param  array                    $resultTexts  (returned appended)
	 * @return boolean
	 */
	protected function checkConditionsApplicable( $userId, $planId, $is_business, $buyerCountry, $buyerProvince, $buyerZip, $sellerOwnerId, $paymentBasket, $now, $isPublished, &$resultTexts ) {
		/** @noinspection PhpUnusedLocalVariableInspection */
		$r =	 (	(	( ( ! $isPublished ) || $this->checkPublishedAndDates( $now ) )						||(($resultTexts[$this->id][] =	CBTxt::T("Promotion is not active at this time.") ) && false			))
		&&	(	$this->checkIfAppliesToPlanId( $planId )													||(($resultTexts[$this->id][] =	CBTxt::T("Promotion does not apply to this item.") ) && false			))
		&&	(	$this->checkIfMaxDiscountUsesNotReached( $userId )											|| ($resultTexts[$this->id][] =	CBTxt::T("Maximum uses of this promotion have been reached.")			))
		&&	(	$this->checkBuyerGeoZone( $buyerCountry, $buyerProvince, $buyerZip )						|| ($resultTexts[$this->id][] =	CBTxt::T("Promotion does not apply to your geographic region.")			))
		&&	(	$this->checkAppliesToBusinessCondition( $is_business )										|| ($resultTexts[$this->id][] =	CBTxt::T("Promotion does not apply to your business customer status.")	))
		&&		$this->checkActiveConditions( $userId, $resultTexts[$this->id], $paymentBasket )
		&&	(	$this->checkViewAccessLevel( $userId )														|| ($resultTexts[$this->id][] =	CBTxt::T("Promotion does not apply to your view access level.")		))
		&&	(	$this->checkUsergroups( $userId )															|| ($resultTexts[$this->id][] =	CBTxt::T("Promotion does not apply to your user group.")				))
		&&	(	$this->checkUserId( $userId )																|| ($resultTexts[$this->id][] =	CBTxt::T("Promotion does not apply to your login.")						))
		&&	(	$this->checkUserEnabled( $userId )															|| ($resultTexts[$this->id][] =	CBTxt::T("Promotion does not apply to blocked users.")						))
		&&	(	$this->checkOwner( $sellerOwnerId )															|| ($resultTexts[$this->id][] =	CBTxt::T("Promotion does not apply to this merchant.")					))
		);
		return ( ! isset( $resultTexts[$this->id] ) );
	}

	/**
	 * Gives the column of the trigger
	 * @return string
	 */
	protected function _columnOfTrigger()
	{
		return $this->trigger_date_reference;
	}

	/**
	 * Gives the table or class of the trigger depending on params
	 *
	 * @param boolean  $returnClassName  TRUE: Return class-name instead of table-name
	 * @param boolean  $isSubscription   OUT: (IN: FALSE): is a subscription date ?
	 * @param boolean  $isUserDate       OUT: (IN: FALSE): is a user profile date ?
	 * @return string|boolean|null
	 */
	protected function _tableOfTrigger( $returnClassName = false, &$isSubscription = false, &$isUserDate = false ) {
		switch ( $this->trigger_date_reference ) {
			case 'registerDate':
			case 'lastvisitDate':
				$table			=	$returnClassName ? 'cbpaidMailerUser' : '#__users';
				$isUserDate		=	true;
				break;
			case 'lastupdatedate':
				$table			=	$returnClassName ? 'cbpaidMailerComprofiler' : '#__comprofiler';
				$isUserDate		=	true;
				break;
			case 'time_initiated':
			case 'time_updated':
			case 'time_completed':
				$table			=	$returnClassName ? 'cbpaidPaymentBasket' : '#__cbsubs_payment_baskets';
				break;
			case 'subscription_date':
			case 'last_renewed_date':
			case 'expiry_date':
			case 'payment_date':
				$table			=	null;
				$isSubscription	=	true;
				break;
			default:
				return false;
		}
		return $table;
	}
	/**
	 * Converts CB +/- SQL date interval to real SQL interval: + -> DATE_SUB, - -> DATE_ADD
	 *
	 * @param  string  $duration   e.g. '-0000-01-15 00:00:00'  =  1 month and 15 days before
	 * @return string              Quoted string, UTC-date string, e.g. "'2015-06-18 12:34:56'"
	 */
	protected function _sqlDateToInterval( $duration ) {

		list( $sign, $y, $c, $d, $h, $m, $s )	=	sscanf( $duration, '%1s%d-%d-%d %d:%d:%d' );

		$dateInterval	=	new DateInterval( 'P' . (int) $y . 'Y' . (int) $c . 'M' . (int) $d . 'D' . 'T' . (int) $h . 'H' . (int) $m . 'M' . (int) $s . 'S' );

		$startTime		=	cbpaidTimes::getInstance()->startTime();

		$utcTimeZone	=	new DateTimeZone( 'UTC' );
		$date			=	new DateTime( '@' . $startTime );
		$date->setTimezone( $utcTimeZone );

		if ( $sign == '-' ) {
			$date->add( $dateInterval );
		} else {
			$date->sub( $dateInterval );
		}

		return $this->_db->Quote( $date->format( $this->_db->getDateFormat() ) );
	}
	/**
	 * Gets the date of when it is too late
	 *
	 * @param  string  $date
	 * @param  string  $interval
	 * @return string
	 */
	protected function _getTooLateTime ( $date, $interval ) {
		list( $p, $y, $c, $d, $h, $m, $s )			=	sscanf( $interval, '%1s%d-%d-%d %d:%d:%d' );
		$F				=	( $p == '-' ? -1 : 1 );
		$startTime		=	cbpaidTimes::getInstance()->strToTime( $date );
		/** @noinspection PhpWrongStringConcatenationInspection */
		$expiryTime		=	gmmktime( gmdate('H', $startTime )+($h*$F), gmdate('i', $startTime )+($m*$F), gmdate('s', $startTime )+($s*$F),
									  gmdate('m', $startTime )+($c*$F), gmdate('d', $startTime )+($d*$F), gmdate('Y', $startTime )+($y*$F));
		$expiryTime		=	$expiryTime + (2 * 86400 );		// Add 2 days so that we have time to process the mail queue.
		return cbpaidTimes::getInstance()->getUtcDateOfTime( $expiryTime );
	}
	/**
	 * Builds a cbSqlQueryPart for CONDITION
	 *
	 * @param  string  $table
	 * @param  string  $col
	 * @param  string  $operator
	 * @param  string  $textValue
	 * @return cbSqlQueryPart
	 */
	protected function _conditionToSql( $table, $col, $operator, $textValue ) {
		$sql							=	new cbSqlQueryPart();
		$sql->tag						=	'column';
		$sql->name						=	$col;
		$sql->table						=	$table;
		$sql->type						=	'sql:field';
		$sql->operator					=	$operator;
		$sql->value						=	$textValue;
		$sql->valuetype					=	'const:string';
		$sql->searchmode				=	'is';
		return $sql;
	}
	/**
	 * Builds a cbSqlQueryPart for a date field compared to SQL UTC_TIMESTAMP()
	 *
	 * @param  string  $table
	 * @param  string  $col
	 * @param  string  $operator
	 * @return cbSqlQueryPart
	 */
	protected function _dateToSql( $table, $col, $operator ) {
		// $this->validate( $field, $user, $col, $value, $postdata, $reason );
		$sql							=	new cbSqlQueryPart();
		$sql->tag						=	'column';
		$sql->name						=	$col;
		$sql->table						=	$table;
		$sql->type						=	'sql:field';
		$sql->operator					=	$operator;
		$sql->value						=	'UTC_TIMESTAMP()';
		$sql->valuetype					=	'const:formula';
		$sql->searchmode				=	'phrase';
		return $sql;
	}
	/**
	 * Builds a cbSqlQueryPart for a JOIN with a date field compared to SQL UTC_TIMESTAMP()
	 *
	 * @param  string  $table
	 * @param  string  $mainTableName
	 * @param  string  $col
	 * @param  string  $operator_a
	 * @param  string  $trigger_date_diff_a
	 * @param  string  $operator_b
	 * @param  string  $trigger_date_diff_b
	 * @return cbSqlQueryPart
	 */
	protected function _joinDataAddedToSql( $table, $mainTableName, $col, $operator_a, $trigger_date_diff_a, $operator_b, $trigger_date_diff_b ) {
		$sqlLeft					=	new cbSqlQueryPart();
		$sqlLeft->tag				=	'data';
		$sqlLeft->name				=	$col;
		$sqlLeft->table				=	$table;
		$sqlLeft->type				=	'sql:field';

		$sqlJoinkeys				=	new cbSqlQueryPart();
		$sqlJoinkeys->tag			=	'joinkeys';
		$sqlJoinkeys->type			=	'inner';
		$sqlJoinkeys->operator		=	'AND';
		$sqlLeft->addChildren( array( $sqlJoinkeys ) );

		$sqlJoin1					=	new cbSqlQueryPart();
		$sqlJoin1->tag				=	'column';
		$sqlJoin1->name				=	'id';
		$sqlJoin1->table			=	$table;
		$sqlJoin1->type				=	'sql:field';
		$sqlJoin1->operator			=	'=';
		$sqlJoin1->value			=	'user_id';
		$sqlJoin1->valuetype		=	'sql:field';
		$sqlJoin1->valuetable		=	$mainTableName;

		$sqlJoin2					=	new cbSqlQueryPart();
		$sqlJoin2->tag				=	'column';
		$sqlJoin2->name				=	$col;
		$sqlJoin2->table			=	$table;
		$sqlJoin2->type				=	'sql:field';
		$sqlJoin2->operator			=	$operator_a;
		$sqlJoin2->value			=	'UTC_TIMESTAMP()';
		$sqlJoin2->valuetype		=	'const:formula';
		$intervalSQLFunction		=	$this->_sqlDateToInterval( $trigger_date_diff_a );
		if ( $intervalSQLFunction ) {
			$sqlJoin2->value		=	$intervalSQLFunction;
		}

		$sqlJoinkeys->addChildren( array( $sqlJoin1, $sqlJoin2 ) );

		if ( $trigger_date_diff_b ) {
			$sqlJoin3					=	new cbSqlQueryPart();
			$sqlJoin3->tag				=	'column';
			$sqlJoin3->name				=	$col;
			$sqlJoin3->table			=	$table;
			$sqlJoin3->type				=	'sql:field';
			$sqlJoin3->operator			=	$operator_b;
			$sqlJoin3->value			=	'UTC_TIMESTAMP()';
			$sqlJoin3->valuetype		=	'const:formula';
			$intervalSQLFunction		=	$this->_sqlDateToInterval( $trigger_date_diff_b );
			if ( $intervalSQLFunction ) {
				$sqlJoin3->value		=	$intervalSQLFunction;
			}

			$sqlJoinkeys->addChildren( array( $sqlJoin3 ) );
		}

		$sqlRight					=	new cbSqlQueryPart();
		$sqlRight->tag				=	'data';
		$sqlRight->type				=	'const:datetime';
		$sqlRight->value			=	'0000-00-00 00:00:00';
		$sqlRight->valuetype		=	'const:datetime';

		$sql						=	new cbSqlQueryPart();
		$sql->tag					=	'where';
		$sql->type					=	'sql:operator';
		$sql->operator				=	'<>';
		$sql->addChildren( array( $sqlLeft, $sqlRight ) );
		return $sql;
	}
/*
	protected function OLD_joinDataAddedToSql( $table, $col, $operator, $trigger_date_diff ) {
		$sqlLeft					=	new cbSqlQueryPart();
		$sqlLeft->tag				=	'data';
		$sqlLeft->name				=	$col;
		$sqlLeft->table				=	$table;
		$sqlLeft->type				=	'sql:field';
		$sqlLeft->key				=	'id';
		$sqlLeft->value				=	'user_id';
		$sqlLeft->valuetype			=	'sql:field';

		$sqlRight					=	new cbSqlQueryPart();		// warning: temporary in CB 1.4-
		$sqlRight->tag				=	'data';
		// $sqlRight->name			=	'now';
		$sqlRight->type				=	'const:formula';
		$sqlRight->value			=	'UTC_TIMESTAMP()';
		// $sqlRight->valuetype		=	'const:formula';

		list( $interval, $func )	=	$this->_sqlDateToInterval( $trigger_date_diff );
		if ( $interval ) {
			$sqlRight->value		=	$func . '( UTC_TIMESTAMP(), INTERVAL ' . $interval . ')';
		}

		$sql						=	new cbSqlQueryPart();
		$sql->tag					=	'where';
		$sql->type					=	'sql:operator';
		$sql->operator				=	$operator;
		$sql->addChildren( array( $sqlLeft, $sqlRight ) );
		return $sql;
	}
*/
	/**
	 * Add an INTERVAL date to UTC_TIMESTAMP()
	 *
	 * @param  string  $table
	 * @param  string  $col
	 * @param  string  $operator
	 * @param  string  $trigger_date_diff
	 * @return cbSqlQueryPart
	 */
	protected function _dateAddedToSql( $table, /** @noinspection PhpUnusedParameterInspection */ $col, $operator, $trigger_date_diff ) {
		$intervalSQLFunction		=	$this->_sqlDateToInterval( $trigger_date_diff );
		$sql						=	$this->_dateToSql( $table, $this->_columnOfTrigger(), $operator );
		if ( $intervalSQLFunction ) {
			$sql->value				=	$intervalSQLFunction;
		}
		return $sql;
	}

	/**
	 * Adds basic dates conditions to the SQL search
	 *
	 * @param  cbSqlQueryPart  $searches
	 * @param  string              $mainTableName
	 * @param  object              $sampleObjectOfMainTable
	 * @return boolean
	 */
	protected function _addDateConditions( $searches, $mainTableName, $sampleObjectOfMainTable ) {
		$table						=	$this->_tableOfTrigger();
		if ( FALSE && $table ) {
			$sql					=	$this->_joinDataAddedToSql( $table, $mainTableName, $this->_columnOfTrigger(), '<=', $this->trigger_date_diff_a, '>=', $this->trigger_date_diff_b );
			$searches->addChildren( array( $sql ) );
			$queryExecutable		=	true;
		} else {
			$queryExecutable		=	property_exists( $sampleObjectOfMainTable, $this->_columnOfTrigger() );
			if ( $queryExecutable ) {
				$sql				=	$this->_dateAddedToSql( $mainTableName, $this->_columnOfTrigger(), '<=', $this->trigger_date_diff_a );
				$searches->addChildren( array( $sql ) );
				if ( $this->trigger_date_diff_b ) {
					$sql			=	$this->_dateAddedToSql( ( $table ? $table : $mainTableName ), $this->_columnOfTrigger(), '>=', $this->trigger_date_diff_b );
					$searches->addChildren( array( $sql ) );
				}
			}
		}
		return $queryExecutable;
	}
	/**
	 * Builds and sends e-mail
	 *
	 * @param  UserTable             $user
	 * @param  cbpaidSomething|null  $subscription or cbpaidPaymentBasket
	 * @param  int                   $now
	 * @param  cbpaidProduct         $plan
	 * @return boolean
	 */
	protected function addToQueue( $user, $subscription, $now, $plan = null ) {
		$cbUser								=	CBuser::getInstance( $user->id );

		if ( ! $cbUser ) {
			return false;
		}

		$subscriptionTrigger				=	false;
		$isUserDate							=	false;
		$this->_tableOfTrigger( false, $subscriptionTrigger, $isUserDate );

		$trigger_date_exists				=	( $isUserDate ? isset( $user->{$this->_columnOfTrigger()} ) : isset( $subscription->{$this->_columnOfTrigger()} ) );
		if ( ! $trigger_date_exists ) {
			// Typically wrong subscription type...:
			return false;
		}
		$trigger_date_used					=	( $isUserDate ? $user->{$this->_columnOfTrigger()} : $subscription->{$this->_columnOfTrigger()} );

		$conditions							=	array(	'automessage_id'	=>	(int) $this->id,
														'user_id'			=>	(int) $user->id,
														'plan_id'			=>	$subscriptionTrigger ? (int) $subscription->plan_id : 0,
														'subscription_id'	=>	$subscriptionTrigger ? (int) $subscription->id : 0,
														'trigger_date_used'	=>	$trigger_date_used
													 );
		$mqE								=	new cbpaidMailerMailQueue();
		if ( ! $mqE->loadThisMatching( $conditions ) ) {
			$msE							=	new cbpaidMailerMessage();
			if ( ! $msE->loadThisMatching( $conditions ) ) {

				// Email can be queued:

				$savedLanguage				=	CBTxt::setLanguage( $user->getUserLanguage() );

				// Apply template:
				$params						=	cbpaidApp::settingsParams();
				if ( $this->emailapplytemplate === '0' ) {
					$template				=	'';
				} elseif ( $this->mail_type == 'email' ) {
					if ( $this->emailhtml ) {
						$template			=	$params->getString( 'integration_cbsubsmailer_email_template_html', '' );
					} else {
						$template			=	$params->getString( 'integration_cbsubsmailer_email_template_text', '' );
					}
				} elseif ( $this->mail_type == 'pm' ) {
					$template				=	$params->getString( 'integration_cbsubsmailer_pm_template', '' );
				} else {
					$template				=	'';
				}
				if ( strpos( $template, '[MESSAGE_CONTENT]' ) !== false ) {
					//TODO: $unsubscribeLink
					$emailbody				=	str_replace( '[MESSAGE_CONTENT]', $this->emailbody, $template );
				} else {
					$emailbody				=	$this->emailbody;
				}

				if ( $plan ) {
					if ( $subscription ) {
						$extraStrings		=	$subscription->substitutionStrings( $this->emailhtml );
					} else {
						$extraStrings		=	$plan->substitutionStrings( $this->emailhtml, $user->id, true );
					}
				} elseif ( is_a( $subscription, 'cbpaidPaymentBasket' ) ) {
					// Payment basket:
					$extraStrings			=	$subscription->substitutionStrings( $this->emailhtml );
				} else {
					$extraStrings			=	array( 'EMAILADDRESS' => $user->email );
				}
				// Add the date on which the trigger triggered:
				$extraStrings['TRIGGER_SOURCE_DATE']	=	cbpaidTimes::getInstance()->cbFormatDateInOfficialTz( $trigger_date_used );
				$emailbody					=	$cbUser->replaceUserVars( $emailbody,				false, false, $extraStrings, true );
				if ( ! $this->emailhtml ) {
					$emailbody				=	str_replace( array( '<br>', '<br/>', '<br />' ), array( "\r\n", "\r\n", "\r\n" ), $emailbody );
					$emailbody				=	strip_tags( $emailbody );
				}

				$emailTo					=	$this->getParam( 'emailtoemail', null );

				if ( $emailTo ) {
					$emailTo				=	trim( $cbUser->replaceUserVars( $emailTo, false, false, array(), false ) );
				}

				if ( ! $emailTo ) {
					$emailTo				=	$user->email;
				}

				$mq							=	new cbpaidMailerMailQueue();
				$mq->emailfromemail			=	trim( $cbUser->replaceUserVars( $this->emailfromemail,	false, false, array(), false ) );
				$mq->emailfromname			=	trim( $cbUser->replaceUserVars( $this->emailfromname,	false, false, array(), false ) );
				$mq->emailtoemail			=	$emailTo;
				$mq->emailcc				=	trim( $cbUser->replaceUserVars( $this->emailcc,			false, false, array(), false ) );
				$mq->emailbcc				=	trim( $cbUser->replaceUserVars( $this->emailbcc,		false, false, array(), false ) );
				$mq->emailsubject			=	trim( $cbUser->replaceUserVars( $this->emailsubject,	false, false, $extraStrings, true ) );
				$mq->emailbody				=	$emailbody;
				$mq->emailattachments		=	trim( $this->emailattachments );
				$mq->emailhtml				=	$this->emailhtml;
				$mq->automessage_id			=	$this->id;
				$mq->mail_type				=	$this->mail_type;
				$mq->priority				=	$this->priority;
				$mq->user_id				=	$user->id;
				$mq->plan_id				=	$subscriptionTrigger ? (int) $subscription->plan_id : 0;
				$mq->subscription_id		=	$subscriptionTrigger ? (int) $subscription->id : 0;
				$mq->state					=	'A';
				$mq->trials_made			=	0;
				$mq->trigger_date_reference	=	$this->trigger_date_reference;
				$mq->trigger_date_used		=	$trigger_date_used;
				$mq->time_to_mail			=	cbpaidTimes::getInstance()->getUtcDateOfTime( $now );
				$mq->time_too_late			=	( $this->trigger_date_diff_b && $mq->trigger_date_used && ( $mq->trigger_date_used != $this->_db->getNullDate() ) ) ? $this->_getTooLateTime( $mq->trigger_date_used, $this->trigger_date_diff_b ) : null;
				$mq->store();
				unset( $mq );

				CBTxt::setLanguage( $savedLanguage );

				return true;
			}
		}
		return false;
	}
	/**
	 * Processes the auto-mailer
	 *
	 * @param  boolean  $isPublished
	 * @return int                    Processed mails
	 */
	public function processAutoMailer( $isPublished = true ) {
		global $_CB_framework, $_CB_database;

		$countProcessed				=	0;
		$now						=	$_CB_framework->now();
		if ( ( ! $isPublished ) || $this->checkPublishedAndDates( $now ) ) {
			$objectClass			=	$this->_tableOfTrigger( true );
			if ( $objectClass ) {

				if ( $objectClass == 'cbpaidPaymentBasket' ) {

					// jos_payment_baskets based:

					$dummyUser						=	new cbpaidPaymentBasket();
					$dummyUserTbl					=	$dummyUser->getTableName();
					$dummyUserDb					=	$dummyUser->getDbo();
					$dummyUserForCols				=	$dummyUser;
					$userIdCol						=	'*';

				} else {

					// jos_users or jos_comprofiler dates based:

					$dummyUser						=	new $objectClass( $_CB_database );
					if ( $dummyUser instanceof TableInterface ) {
						$dummyUserTbl				=	$dummyUser->getTableName();
						$dummyUserDb				=	$dummyUser->getDbo();
					} else {
						$dummyUserTbl				=	$dummyUser->_tbl;
						$dummyUserDb				=	$dummyUser->_db;
					}
					$dummyUserForCols				=	new UserTable();
					$userIdCol						=	'id';

				}

				$searches							=	new cbSqlQueryPart();
				$searches->tag						=	'where';
				$searches->type						=	'sql:operator';
				$searches->operator					=	'AND';
				if ( $this->_addDateConditions( $searches, $dummyUserTbl, $dummyUserForCols ) ) {

					if ( $objectClass == 'cbpaidPaymentBasket' ) {
						switch ( $this->trigger_date_reference ) {
							case 'time_initiated':
								$statusToSearch		=	'NotInitiated';
								break;
							case 'time_updated':
								$statusToSearch		=	'Pending';
								break;
							case 'time_completed':
								$statusToSearch		=	'Completed';
								break;
							default:
								return 0;
						}

						$searches->addChildren( array( $this->_conditionToSql( $dummyUserForCols->getTableName(), 'payment_status', '=', $statusToSearch ) ) );
					}

					$conditions						=	array( 'dates_condition' => $searches );
					$offset							=	0;
					$limit							=	0;
					$dummyUser->setMatchingQuery( array( $userIdCol ), $conditions, array( 'id' => 'DESC' ), $offset, $limit );
					if ( $userIdCol == '*' ) {
						$objects					=	$dummyUserDb->loadObjectList( 'user_id', get_class( $dummyUser ) );
						$userIds					=	array_keys( $objects );
					} else {
						$objects					=	array();
						$userIds					=	$dummyUserDb->loadResultArray();
					}
					if ( count( $userIds ) > 0 ) {
						CBuser::advanceNoticeOfUsersNeeded( $userIds );
						foreach ( $userIds as $userId ) {
							/** @var cbpaidUserWithSubsFields $user */
							$user					=	CBuser::getUserDataInstance( $userId );
							$cbpaidUser				=	cbpaidUserExtension::getInstance( $userId );
							if ( $user && $user->id ) {
								set_time_limit( 60 );

								if ( ( $objectClass == 'cbpaidPaymentBasket' ) && isset( $objects[$userId] ) ) {
									$paymentBasket	=	$objects[$userId];
 								} else {
									$paymentBasket	=	null;
								}

								$resultTexts		=	array();
								if ( $this->checkConditionsApplicable( $userId, null, $cbpaidUser->getInvoiceAddressField( 'cb_subs_inv_payer_business_name' ), $cbpaidUser->getInvoiceAddressField( 'cb_subs_inv_address_country' ), $cbpaidUser->getInvoiceAddressField( 'cb_subs_inv_address_state' ), $cbpaidUser->getInvoiceAddressField( 'cb_subs_inv_address_zip' ), 0, $paymentBasket, $now, true, $resultTexts ) ) {
									if ( $this->addToQueue( $user, isset( $objects[$userId] ) ? $objects[$userId] : null , $now ) ) {
										++$countProcessed;
									}
								}
							}
						}
					}
				}


			} else {

				// subscription-based:

				$plansMgr				=	cbpaidPlansMgr::getInstance();
				$plans					=	$plansMgr->loadPublishedPlans( null, true, 'any', null );
				if ( $this->plans_applied_to ) {
					$plansNumbers		=	explode( '|*|', $this->plans_applied_to );
					cbArrayToInts( $plansNumbers );
				} else {
					$plansNumbers		=	array_keys( $plans );
				}
				if ( $this->plans_status ) {
					$plansStatus		=	explode( '|*|', $this->plans_status );
					if ( count( $plansStatus ) == 1 ) {
						$plansStatus	=	$this->plans_status;
					}
				} else {
					$plansStatus		=	null;
				}

				foreach ( $plansNumbers as $pId ) {
					if ( isset( $plans[$pId] ) ) {
						// $plan = new cbpaidProduct();
						$dummySub						=	$plans[$pId]->newSubscription();
						// $dummySub = new cbpaidSomething();
						$searches						=	new cbSqlQueryPart();
						$searches->tag					=	'where';
						$searches->type					=	'sql:operator';
						$searches->operator				=	'AND';
						if ( $this->_addDateConditions( $searches, $dummySub->getTableName(), $dummySub ) ) {
							$conditions					=	array( 'dates_condition' => $searches );
							$offset							=	0;
							$limit							=	0;
							$correspondingSubscriptions	=	$dummySub->loadAllSomethingsOfUser( null, $plansStatus, $conditions, $offset, $limit );
							foreach ( $correspondingSubscriptions as $subscription ) {
								// $subscription = NEW cbpaidSomething();
								$user					=	CBuser::getUserDataInstance( $subscription->user_id );
								if ( $user && $user->id ) {
									set_time_limit( 60 );
									$cbpaidUser			=	cbpaidUserExtension::getInstance( $subscription->user_id );
									$paymentBasket		=	null;
									$resultTexts		=	null;
									/** @var cbpaidUserWithSubsFields $user */
									if ( $this->checkConditionsApplicable( $subscription->user_id, $subscription->plan_id, $cbpaidUser->getInvoiceAddressField( 'cb_subs_inv_payer_business_name' ), $cbpaidUser->getInvoiceAddressField( 'cb_subs_inv_address_country' ), $cbpaidUser->getInvoiceAddressField( 'cb_subs_inv_address_state' ), $cbpaidUser->getInvoiceAddressField( 'cb_subs_inv_address_zip' ), 0, $paymentBasket, $now, true, $resultTexts ) ) {
										if ( $this->addToQueue( $user, $subscription, $now, $plans[$pId] ) ) {
											++$countProcessed;
										}
									}
								}
							}
						}
					}
				}

			}
		}
		return $countProcessed;
	}
}

/**
 * Messages Abstract Model for Queue and Sent messages
 */
abstract class cbpaidMailerMessagesTable extends cbpaidTable {
	public $id;							// sql:int(11)
	public $state;						// sql:char(1) default="I"nvalid, "A"ctive, "F"ailed
	public $automessage_id;				// sql:int(11)
	public $plan_id;					// sql:int(11)
	public $user_id;					// sql:int(11)
	public $trigger_date_reference;		// sql:varchar(21)
	public $trigger_date_used;			// sql:varchar(21)
	public $subscription_id;			// sql:int(11) unsigned="true"
	public $mail_type;					// sql:varchar(8)
	public $emailtoname;				// sql:varchar(255)
	public $emailtoemail;				// sql:varchar(255)
	public $emailsubject;				// sql:mediumtext
	public $emailbody;					// sql:mediumtext
	public $emailhtml;					// sql:tinyint(4) default="0"
	public $emailcc;					// sql:varchar(255)
	public $emailbcc;					// sql:varchar(255)
	public $emailfromname;				// sql:varchar(255)
	public $emailfromemail;				// sql:varchar(255)
	public $emailattachments;			// sql:text
	public $integrations;				// sql:varchar(255)
	/**
	 * Count mails in the database table
	 *
	 * @param  int   $automessage_id
	 * @param  int   $userId
	 * @param  array $conditions
	 * @return int
	 */
	protected function _countMailsInTable( $automessage_id, $userId = 0, $conditions = array() ) {
		$conditions[]		=	'automessage_id = ' . (int) $automessage_id;
		$conditions[]		=	'state = "A"';
		if ( $userId ) {
			$conditions[]	=	'user_id = ' . (int) $userId;
		}
		return $this->countRows( implode( ' AND ', $conditions ) );
	}
}

/**
 * Messages Model for Queued Message
 */
class cbpaidMailerMailQueue extends cbpaidMailerMessagesTable {
	//	public $id;						// sql:int(11)
	public $time_to_mail;				// sql:datetime null="true"
	public $time_too_late;				// sql:datetime null="true"
	public $priority;					// sql:int(11)
	public $trials_made;				// sql:int(11)
	//	public $state;					// sql:char(1) default="I"nvalid, "A"ctive, "F"ailed
	//	public $automessage_id;			// sql:int(11)
	//	public $plan_id;				// sql:int(11)
	//	public $user_id;				// sql:int(11)
	//	public $trigger_date_reference;	// sql:// varchar(21)
	//	public $trigger_date_used;		// sql:varchar(21)
	//	public $subscription_id;		// sql:int(11) unsigned="true"
	//	public $mail_type;				// sql:varchar(8)
	//	public $emailtoname;			// sql:varchar(255)
	//	public $emailtoemail;			// sql:varchar(255)
	//	public $emailsubject;			// sql:mediumtext
	//	public $emailbody;				// sql:mediumtext
	//	public $emailhtml;				// sql:tinyint(4) default="0"
	//	public $emailcc;				// sql:varchar(255)
	//	public $emailbcc;				// sql:varchar(255)
	//	public $emailfromname;			// sql:varchar(255)
	//	public $emailfromemail;			// sql:// varchar(255)
	//	public $emailattachments;		// sql:text
	//	public $integrations;			// sql:varchar(255)
	/**
	 * Constructor
	 *
	 * @param  DatabaseDriverInterface  $db
	 */
	public function __construct( &$db = null ) {
		parent::__construct( '#__cbsubs_mailer_mailqueue', 'id', $db );
		// No need here: $this->_historySetLogger();
	}
	/**
	 * Get the instances of top priority in queue
	 *
	 * @param  int    $limit
	 * @param  int    $now
	 * @return cbpaidMailerMailQueue[]
	 */
	public static function getInstancesTopPriority( $limit, $now ) {
		global $_PLUGINS;

		static $pluginsNotLoaded	=	true;
		if ( $pluginsNotLoaded ) {
			// Loads plugins for sending events below:
			$_PLUGINS->loadPluginGroup( 'user' );
			$_PLUGINS->loadPluginGroup('user/plug_cbpaidsubscriptions/plugin');
			$pluginsNotLoaded		=	false;
		}

		$nowDate	=	cbpaidTimes::getInstance()->getUtcDateOfTime( $now );
		$loader		=	new self();
		$conditions	=	array(	array( 'state', '=', 'A' ),
								array( 'time_to_mail', '<=', $nowDate ),
								array( 'OR' => array(
													  array( 'time_too_late', '>',  $nowDate ),
													  array( 'time_too_late', 'IS NULL',  null )
								) )
						);
		$ordering	=	array(	'priority'		=> 'DESC',
								'time_to_mail'	=>	'ASC' );
		return $loader->loadThisMatchingList($conditions, $ordering, 0, $limit );
	}
	/**
	 * Counts the mails in the table
	 *
	 * @param  int       $automessage_id
	 * @param  int       $userId
	 * @param  int|null  $sinceUTCDateTime
	 * @return int
	 */
	public static function countMailsInTable( $automessage_id, $userId = 0, $sinceUTCDateTime = null ) {
		$me					=	new self();
		$conditions			=	array();
		if ( $sinceUTCDateTime ) {
			global $_CB_database;
			$conditions[]	=	'time_to_mail >= ' . $_CB_database->Quote( $sinceUTCDateTime );
		}
		return $me->_countMailsInTable( $automessage_id, $userId, $conditions );
	}
	/**
	 * Sends a message
	 *
	 * @param  int     $maxRetrials
	 * @param  int     $retrialHours
	 * @return boolean
	 */
	public function sendMail( $maxRetrials, $retrialHours ) {
		global $_PLUGINS;

		$subject					=	$this->emailsubject;
		$body						=	$this->emailbody;

		if ( $this->mail_type == 'email' ) {
			global $ueConfig;

			if ( $this->emailtoemail != '' ) {
				$mailTo				=	preg_split( '/ *, */', $this->emailtoemail );
			} else {
				return false;
			}

			if ( $this->emailtoname != '' ) {
				$mailToName			=	preg_split( '/ *, */', $this->emailtoname );
			} else {
				$mailToName			=	null;
			}

			if ( $this->emailcc != '' ) {
				$mailCC				=	preg_split( '/ *, */', $this->emailcc );
			} else {
				$mailCC				=	null;
			}

			if ( $this->emailbcc != '' ) {
				$mailBCC			=	preg_split( '/ *, */', $this->emailbcc );
			} else {
				$mailBCC			=	null;
			}

			if ( $this->emailattachments != '' ) {
				$mailAttachments	=	preg_split( '/ *, */', $this->emailattachments );
			} else {
				$mailAttachments	=	null;
			}

			if ( $this->emailfromemail != '' ) {
				$mailFrom			=	$this->emailfromemail;
				$mailReplyTo		=	$mailFrom;
			} else {
				$mailFrom			=	$ueConfig['reg_email_from'];
				$mailReplyTo		=	$ueConfig['reg_email_replyto'];
			}

			if ( $this->emailfromname != '' ) {
				$mailFromName		=	$this->emailfromname;
			} else {
				$cbNotification		=	new cbNotification();
				$mailFromName		=	$cbNotification->defaultFromName();

			}
			$mailReplyToName		=	$mailFromName;

			// Trigger CBSubs event: (plugins already loaded above)
			$_PLUGINS->trigger( 'onCPayBeforeMailerEmailMessageSent', array( $this->user_id, $this->automessage_id, &$mailFrom, &$mailFromName, &$mailToName, &$mailTo, &$subject, &$body, $this->emailhtml, &$mailCC, &$mailBCC, &$mailAttachments, &$mailReplyTo, &$mailReplyToName ) );

			if ( $mailTo && ( $subject || $body ) ) {
				//TODO: this function misses ToName!:
				cbimport( 'cb.tabs' );
				$error				=	null;
				$result				=	comprofilerMail( $mailFrom, $mailFromName, $mailTo, $subject, $body, $this->emailhtml, $mailCC, $mailBCC, $mailAttachments, $mailReplyTo, $mailReplyToName, array(), $error );
				if ( ! $result ) {
					// Log $error into CBSubs History logs
					$logObject		=	new cbpaidHistory( $this->_db );
					$logObject->logError( 3, CBTxt::T('Mailer mail sending error:') . ' ' . $error, null );
				}
			} else {
				$result				=	false;
			}

		} elseif ( $this->mail_type == 'pm' ) {

			// Trigger CBSubs event: (plugins already loaded above)
			$_PLUGINS->trigger( 'onCPayBeforeMailerPrivateMessageSent', array( $this->user_id, $this->automessage_id, &$subject, &$body ) );

			if ( $subject || $body ) {
				global $_CB_PMS;
				$resultArray		=	$_CB_PMS->sendPMSMSG( $this->user_id, 0, $subject, $body, true );
				$result				=	( ( count( $resultArray ) > 0 ) ? $resultArray[0] : false );
			} else {
				$result				=	null;
			}

		} else {
			$result					=	false;
		}
		if ( $result ) {
			// Trigger CBSubs event: (plugins already loaded above)
			$_PLUGINS->trigger( 'onCPayAfterMailerMessageSent', array( $this->user_id, $this->automessage_id, $subject, $body ) );

		} else {

			// Re-schedule retrial:
			++$this->trials_made;
			$this->time_to_mail		=	cbpaidTimes::getInstance()->getUtcDateOfTime( time() + ( 3600 * (int) $retrialHours ) );
			if ( $this->time_to_mail >= $this->time_too_late ) {
				$this->trials_made	=	$maxRetrials;
			}
			if ( $this->trials_made >= $maxRetrials ) {
				$this->state		=	'F';		// Failed
			}
		}
		return $result;
	}
}

/**
 * Messages Model for Sent Message
 */
class cbpaidMailerMessage extends cbpaidMailerMessagesTable {
	//	public $id;						// sql:int(11)
	public $time_completed;				// sql:datetime null="true"
	//	public $state;					// sql:char(1) default="I"nvalid, "A"ctive, "F"ailed
	//	public $automessage_id;			// sql:int(11)
	//	public $plan_id;				// sql:int(11)
	//	public $user_id;				// sql:int(11)
	//	public $trigger_date_reference;	// sql:// varchar(21)
	//	public $trigger_date_used;		// sql:varchar(21)
	//	public $subscription_id;		// sql:int(11) unsigned="true"
	//	public $mail_type;				// sql:varchar(8)
	//	public $emailtoname;			// sql:varchar(255)
	//	public $emailtoemail;			// sql:varchar(255)
	//	public $emailsubject;			// sql:mediumtext
	//	public $emailbody;				// sql:mediumtext
	//	public $emailhtml;				// sql:tinyint(4) default="0"
	//	public $emailcc;				// sql:varchar(255)
	//	public $emailbcc;				// sql:varchar(255)
	//	public $emailfromname;			// sql:varchar(255)
	//	public $emailfromemail;			// sql:// varchar(255)
	//	public $emailattachments;		// sql:text
	//	public $integrations;			// sql:varchar(255)
	/**
	 * Constructor
	 *
	 * @param  DatabaseDriverInterface  $db
	 */
	public function __construct( &$db = null ) {
		parent::__construct( '#__cbsubs_mailer_sentmails', 'id', $db );
		// No need here too: $this->_historySetLogger();
	}
	/**
	 * Counts the mails in the table
	 *
	 * @param  int       $automessage_id
	 * @param  int       $userId
	 * @param  int|null  $sinceUTCDateTime
	 * @return int
	 */
	public static function countMailsInTable( $automessage_id, $userId = 0, $sinceUTCDateTime = null ) {
		$me					=	new self();
		$conditions			=	array();
		if ( $sinceUTCDateTime ) {
			global $_CB_database;
			$conditions[]	=	'time_completed >= ' . $_CB_database->Quote( $sinceUTCDateTime );
		}
		return $me->_countMailsInTable( $automessage_id, $userId, $conditions );
	}
	/**
	 * Log $mailSent final $success state
	 *
	 * @param  cbpaidMailerMailQueue  $mailSent
	 * @param  boolean                $success
	 */
	public function logSentMessage( $mailSent, $success ) {
		$ignore						=	'id state';
		// Log also emailbody ? :
		$automailer					=	cbpaidMailerAutomessage::getInstance( $mailSent->automessage_id );
		if ( $automailer ) {
			if ( $automailer->storemessagebody != 1 ) {
				$ignore				.=	' emailbody';
			}
		}
		$this->bind( $mailSent, $ignore, false );
		$this->time_completed		=	cbpaidTimes::getInstance()->getUtcDateOfTime();
		$this->state				=	( $success ? 'A' : 'F' );
		$this->store( true );
	}
}

/**
 * Controller class for Mailer
 */
class cbpaidMailerAutoMessagesProcessor {
	/**
	 * Processes Mailer Rules for each Mailer
	 *
	 * @return string  TEXT to display
	 */
	public static function processRules( ) {
		$countProcessed				=	0;
		$automailers				=	cbpaidMailerAutomessage::getInstances( true );
		foreach ( $automailers as $mailer ) {
			$countProcessed			+=	$mailer->processAutoMailer();
		}
		$html						=	CBTxt::T("Processed [PROCESSED_COUNT] messages into the email queue.", null, array( '[PROCESSED_COUNT]' => $countProcessed ) );
		return $html;
	}
}

/**
 * Mailer sender class
 */
class cbpaidMailerSender {
	/**
	 * Sends mails from the queue
	 *
	 * @param  int  $limit
	 * @param  int  $maxRetrials
	 * @param  int  $retrialHours
	 * @return null|string
	 */
	public static function mailFromQueue( $limit, $maxRetrials, $retrialHours ) {
		global $_CB_framework;
		$now							=	$_CB_framework->now();
		$mailsInQueue					=	cbpaidMailerMailQueue::getInstancesTopPriority( $limit, $now );
		$countMailedSuccess				=	0;
		$countMailedFailed				=	0;
		foreach ( $mailsInQueue as $mailToSend ) {
			set_time_limit( 60 );
			if ( $mailToSend->sendMail( $maxRetrials, $retrialHours ) ) {
				$mailToSend->delete();
				$mailedMessage			=	new cbpaidMailerMessage();
				$mailedMessage->logSentMessage( $mailToSend, true );
				++$countMailedSuccess;
			} else {
				if ( $mailToSend->trials_made >= $maxRetrials ) {
					$mailToSend->delete();
					$mailedMessage		=	new cbpaidMailerMessage();
					$mailedMessage->logSentMessage( $mailToSend, false );
				} else {
					$mailToSend->store();
				}
				++$countMailedFailed;
			}
		}
		if ( $countMailedFailed ) {
			$return	=	CBTxt::T("Sent [SUCCESS_MESSAGES_COUNT] messages successfully but failed sending [FAILED_MESSAGES_COUNT] messages.", null, array( '[SUCCESS_MESSAGES_COUNT]' => $countMailedSuccess, '[FAILED_MESSAGES_COUNT]' => $countMailedFailed ) );
		} elseif ( $countMailedSuccess ) {
			$return	=	 CBTxt::T("Sent [SUCCESS_MESSAGES_COUNT] messages successfully.", null, array( '[SUCCESS_MESSAGES_COUNT]' => $countMailedSuccess ) );
		} else {
			$return	=	 null;
		}
		return $return;
	}
}

/**
 * Model class for User record
 */
class cbpaidMailerUser extends cbpaidTable {
	/**
	 * Constructor
	 *
	 * @param  DatabaseDriverInterface  $db
	 */
	public function __construct( &$db = null ) {
		parent::__construct( '#__users', 'id', $db );
	}
}
/**
 * Model class for Comprofiler record
 */
class cbpaidMailerComprofiler extends cbpaidTable {
	/**
	 * Constructor
	 *
	 * @param  DatabaseDriverInterface  $db
	 */
	public function __construct( &$db = null ) {
		parent::__construct( '#__comprofiler', 'id', $db );
	}
}
/**
* Paid Subscriptions Mailer Class
*/
class getcbsubsmailerTab extends cbTabHandler {
	/**
	 * WARNING: UNCHECKED ACCESS! On purpose unchecked access for M2M operations
	 * Generates the HTML to display for a specific component-like page for the tab. WARNING: unchecked access !
	 *
	 * @param  UserTable        $user      the user being displayed
	 * @param  ParamsInterface  $params    Params for the task
	 * @param  array            $postdata  _POST data for saving edited tab content as generated with getEditTab
	 * @return mixed                       either string HTML for tab content, or false if ErrorMSG generated
	 */
	public function executeTask( /** @noinspection PhpUnusedParameterInspection */ $user, $params, $postdata ) {
		$baseClass				=	cbpaidApp::getBaseClass();

		$do						=	$baseClass->_getReqParam( 'mailerdo' );
		switch ($do ) {
			case 'cronprocessmailerrules':
			case 'processmailerrules':
				$html			=	$this->processEmailRules( $params, ( $do == 'processmailerrules' ) );
				break;

			case 'cronsendmailqueue':
			case 'sendmailqueue':
				$html			=	$this->sendMailQueue( $params, ( $do == 'sendmailqueue' ) );
				break;

			case 'cronprocesssendmailqueue':
				$html			=	$this->processEmailRules( $params, false );
				$html			.=	' ';
				$html			.=	$this->sendMailQueue( $params, false );
				break;

			default:
				$html			=	CBTxt::Th( 'UE_NOT_AUTHORIZED', 'You are not authorized to view this page!' );
				break;
		}

		return $html;
	}
	/**
	 * Process emails/PMs into sending queue
	 *
	 * @param  ParamsInterface  $params       Params for the task
	 * @param  boolean          $interactive  True: human interface, False: cron-job
	 * @return string                         HTML
	 */
	protected function processEmailRules( /** @noinspection PhpUnusedParameterInspection */ $params, $interactive ) {
		$html					=	cbpaidMailerAutoMessagesProcessor::processRules();
		if ( ( $html == null ) && $interactive ) {
			$html				=	 CBTxt::T("No messages added in messages queue." );
		}
		return $html;
	}
	/**
	 * Send emails/PMs depending on parasm
	 *
	 * @param  ParamsInterface  $params       Params for the task
	 * @param  boolean          $interactive  True: human interface, False: cron-job
	 * @return string                         HTML
	 */
	protected function sendMailQueue( $params, $interactive ) {
		$mailsPerCron			=	(int) $params->get( 'integration_cbsubsmailer_mailspercron', 100 );
		$maxRetrials			=	(int) $params->get( 'integration_cbsubsmailer_maxretrials', 3 );
		$retrialHours			=	(int) $params->get( 'integration_cbsubsmailer_retrialhours', 24 );
		$html					=	cbpaidMailerSender::mailFromQueue( $mailsPerCron, $maxRetrials, $retrialHours );
		if ( ( $html == null ) && $interactive ) {
			$html				=	 CBTxt::T("No messages in messages queue to be sent." );
		}
		return $html;
	}
	/**
	 * USED by XML interface ONLY !!! Renders url for the send mail cron
	 *
	 * @param  string           $value   Variable value ( 'sendmailqueue' )
	 * @param  ParamsInterface  $params
	 * @return string                    HTML to display
	 */
	public function renderMailQueueUrl( $value, $params ) {
		$baseClass				=	cbpaidApp::getBaseClass();
		$url	=	'index.php?option=com_comprofiler&amp;task=pluginclass&amp;plugin=cbpaidsubscriptions&amp;' . $baseClass->_getPagingParamName('act') . '=cbsubsclass&amp;' . $baseClass->_getPagingParamName('class') . '=mailer' . '&user=62'
				.	'&amp;' . $baseClass->_getPagingParamName( 'mailerdo' ) . '=' . htmlspecialchars( $value ) . '&amp;key='
				.	md5( $params->getString( 'license_number', '' ) );
		$url	=	cbSef( $url, true, 'raw' );
		return '<a href="' . $url . '" target="_blank">' . $url . '</a>';
	}
}
