<?php
/**
* Community Builder (TM)
* @version $Id: $
* @package CommunityBuilder
* @copyright (C) 2004-2019 www.joomlapolis.com / Lightning MultiCom SA - and its licensors, all rights reserved
* @license http://www.gnu.org/licenses/old-licenses/gpl-2.0.html GNU/GPL version 2
*/

namespace CB\Plugin\FamilyPlans\Trigger;

use CB\Database\Table\TabTable;
use CB\Database\Table\UserTable;
use CB\Plugin\FamilyPlans\Helper;
use CB\Plugin\FamilyPlans\Table\FamilyTable;
use CB\Plugin\FamilyPlans\Table\PseudoDonationTable;
use CB\Plugin\FamilyPlans\Table\PseudoMerchandiseTable;
use CB\Plugin\FamilyPlans\Table\PseudoSubscriptionTable;
use CB\Plugin\FamilyPlans\TemplateHandler;
use CBLib\Application\Application;
use CBLib\Registry\ParamsInterface;
use CBLib\Registry\GetterInterface;
use CBLib\Language\CBTxt;

defined('CBLIB') or die();

class SubscriptionsTrigger extends \cbPluginHandler
{
	/**
	 * @param string                      $do
	 * @param TabTable                    $tab
	 * @param UserTable                   $user
	 * @param int                         $ui
	 * @param array                       $postdata
	 * @param \CBplug_cbpaidsubscriptions $plugin
	 * @return string
	 *
	 * @throws \Exception
	 */
	public function showSharedSubscriptions( $do, $tab, $user, $ui, $postdata, $plugin )
	{
		global $_CB_framework, $_CB_database;

		switch ( $do ) {
			case 'sharedsubscriptions':
				$userId			=	$this->input( 'user', 0, GetterInterface::INT );

				if ( ! $userId ) {
					$user		=	\CBuser::getMyUserDataInstance();
				} else {
					$user		=	\CBuser::getUserDataInstance( $userId );
				}

				if ( ! $user->get( 'id', 0, GetterInterface::INT ) ) {
					return null;
				}

				if ( ( Application::MyUser()->getUserId() != $user->get( 'id', 0, GetterInterface::INT ) ) && ( ! \cbpaidApp::authoriseAction( 'cbsubs.usersubscriptionmanage' ) ) ) {
					return null;
				}

				$action			=	$this->input( 'post/act', null, GetterInterface::STRING );

				switch ( $action ) {
					case 'delete':
					case 'cancel':
						cbSpoofCheck( 'plugin' );

						$ids		=	cbToArrayOfInt( $this->input( 'post/ids', array(), GetterInterface::RAW ) );

						if ( ! $ids ) {
							$_CB_framework->enqueueMessage( CBTxt::T( 'Please select a shared subscription.' ), 'error' );
							break;
						}

						$success	=	false;

						foreach ( $ids as $id ) {
							$row	=	new FamilyTable();

							$row->load( $id );

							if ( ! $row->get( 'id', 0, GetterInterface::INT ) ) {
								continue;
							}

							$planName			=	$row->getSubscription()->getPersonalized( 'name', false );

							if ( $action == 'delete' ) {
								if ( $row->get( 'user', 0, GetterInterface::INT ) ) {
									$toFrom		=	\CBuser::getInstance( $row->get( 'user', 0, GetterInterface::INT ), false )->getField( 'formatname', null, 'html', 'none', 'list', 0, true );
								} else {
									$toFrom		=	htmlspecialchars( $row->get( 'to', null, GetterInterface::STRING ) );
								}
							} else {
								$toFrom			=	\CBuser::getInstance( $row->get( 'user_id', 0, GetterInterface::INT ), false )->getField( 'formatname', null, 'html', 'none', 'list', 0, true );
							}

							if ( $row->getError() || ( ! $row->canDelete() ) ) {
								if ( $action == 'delete' ) {
									$_CB_framework->enqueueMessage( CBTxt::T( 'SHARED_SUB_PLAN_TO_FAILED_DELETE', 'Shared subscription [plan] to [to] failed to delete! Error: [error]', array( '[plan]' => $planName, '[to]' => $toFrom, '[error]' => $row->getError() ) ), 'error' );
								} else {
									$_CB_framework->enqueueMessage( CBTxt::T( 'SHARED_SUB_PLAN_FROM_FAILED_CANCEL', 'Shared subscription [plan] from [from] failed to cancel! Error: [error]', array( '[plan]' => $planName, '[from]' => $toFrom, '[error]' => $row->getError() ) ), 'error' );
								}

								continue;
							}

							if ( $row->getError() || ( ! $row->delete() ) ) {
								if ( $action == 'delete' ) {
									$_CB_framework->enqueueMessage( CBTxt::T( 'SHARED_SUB_PLAN_TO_FAILED_DELETE', 'Shared subscription [plan] to [to] failed to delete! Error: [error]', array( '[plan]' => $planName, '[to]' => $toFrom, '[error]' => $row->getError() ) ), 'error' );
								} else {
									$_CB_framework->enqueueMessage( CBTxt::T( 'SHARED_SUB_PLAN_FROM_FAILED_CANCEL', 'Shared subscription [plan] from [from] failed to cancel! Error: [error]', array( '[plan]' => $planName, '[from]' => $toFrom, '[error]' => $row->getError() ) ), 'error' );
								}

								continue;
							}

							$success	=	true;
						}

						if ( $success ) {
							// Report success if at least something was deleted or cancelled:
							if ( $action == 'delete' ) {
								$_CB_framework->enqueueMessage( CBTxt::T( 'Successfully deleted shared subscriptions!' ) );
							} else {
								$_CB_framework->enqueueMessage( CBTxt::T( 'Successfully cancelled shared subscriptions!' ) );
							}
						}
						break;
				}

				/** @var $viewer \cbpaidSharedListView */
				$viewer			=	TemplateHandler::getViewer( null, 'sharedlist' );

				$prefix			=	'shared_';
				$limit			=	30;
				$limitstart		=	(int) $_CB_framework->getUserStateFromRequest( $prefix . 'limitstart{com_comprofiler}', $prefix . 'limitstart', 0 );
				$search			=	$_CB_framework->getUserStateFromRequest( $prefix . 'search{com_comprofiler}', $prefix . 'search', '' );
				$type			=	$this->input( 'type', null, GetterInterface::STRING );
				$where			=	null;

				if ( $search ) {
					if ( $type == 'from' ) {
						$where	.=	"\n AND ( f." . $_CB_database->NameQuote( 'to' ) . " LIKE " . $_CB_database->Quote( '%' . $_CB_database->getEscaped( $search, true ) . '%', false )
								.	" OR u." . $_CB_database->NameQuote( 'name' ) . " LIKE " . $_CB_database->Quote( '%' . $_CB_database->getEscaped( $search, true ) . '%', false )
								.	" OR u." . $_CB_database->NameQuote( 'email' ) . " LIKE " . $_CB_database->Quote( '%' . $_CB_database->getEscaped( $search, true ) . '%', false ) . " )";
					} else {
						$where	.=	"\n AND ( u." . $_CB_database->NameQuote( 'name' ) . " LIKE " . $_CB_database->Quote( '%' . $_CB_database->getEscaped( $search, true ) . '%', false )
								.	" OR u." . $_CB_database->NameQuote( 'email' ) . " LIKE " . $_CB_database->Quote( '%' . $_CB_database->getEscaped( $search, true ) . '%', false ) . " )";
					}
				}

				$searching		=	( $where ? true : false );

				$query			=	"SELECT COUNT(*)"
								.	"\n FROM " . $_CB_database->NameQuote( '#__cbsubs_family' ) . " AS f"
								.	"\n INNER JOIN " . $_CB_database->NameQuote( '#__cbsubs_subscriptions' ) . " AS s"
								.	" ON s." . $_CB_database->NameQuote( 'id' ) . " = f." . $_CB_database->NameQuote( 'sub_id' )
								.	"\n LEFT JOIN " . $_CB_database->NameQuote( '#__users' ) . " AS u";
				if ( $type == 'from' ) {
					$query		.=	" ON u." . $_CB_database->NameQuote( 'id' ) . " = f." . $_CB_database->NameQuote( 'user' )
								.	"\n WHERE f." . $_CB_database->NameQuote( 'user_id' ) . " = " . $user->get( 'id', 0, GetterInterface::INT );
				} else {
					$query		.=	" ON u." . $_CB_database->NameQuote( 'id' ) . " = f." . $_CB_database->NameQuote( 'user_id' )
								.	"\n WHERE f." . $_CB_database->NameQuote( 'user' ) . " = " . $user->get( 'id', 0, GetterInterface::INT );
				}
				$query			.=	$where;
				$_CB_database->setQuery( $query );
				$total			=	(int) $_CB_database->loadResult();

				$pageNav		=	new \cbPageNav( $total, $limitstart, $limit );

				$pageNav->setInputNamePrefix( $prefix );
				$pageNav->setStaticLimit( true );

				if ( Application::MyUser()->getUserId() != $user->get( 'id', 0, GetterInterface::INT ) ) {
					$pageNav->setBaseURL( $_CB_framework->pluginClassUrl( array( 'plugin' => 'cbpaidsubscriptions', 'do' => 'sharedsubscriptions', 'type' => $type, 'user' => $user->get( 'id', 0, GetterInterface::INT ) ), false ) );
				} else {
					$pageNav->setBaseURL( $_CB_framework->pluginClassUrl( array( 'plugin' => 'cbpaidsubscriptions', 'do' => 'sharedsubscriptions', 'type' => $type ), false ) );
				}

				$query			=	"SELECT f.*"
								.	"\n FROM " . $_CB_database->NameQuote( '#__cbsubs_family' ) . " AS f"
								.	"\n INNER JOIN " . $_CB_database->NameQuote( '#__cbsubs_subscriptions' ) . " AS s"
								.	" ON s." . $_CB_database->NameQuote( 'id' ) . " = f." . $_CB_database->NameQuote( 'sub_id' )
								.	"\n LEFT JOIN " . $_CB_database->NameQuote( '#__users' ) . " AS u";
				if ( $type == 'from' ) {
					$query		.=	" ON u." . $_CB_database->NameQuote( 'id' ) . " = f." . $_CB_database->NameQuote( 'user' )
								.	"\n WHERE f." . $_CB_database->NameQuote( 'user_id' ) . " = " . $user->get( 'id', 0, GetterInterface::INT );
				} else {
					$query		.=	" ON u." . $_CB_database->NameQuote( 'id' ) . " = f." . $_CB_database->NameQuote( 'user_id' )
								.	"\n WHERE f." . $_CB_database->NameQuote( 'user' ) . " = " . $user->get( 'id', 0, GetterInterface::INT );
				}
				$query			.=	$where
								.	"\n ORDER BY f." . $_CB_database->NameQuote( 'sent' ) . " DESC";
				$_CB_database->setQuery( $query, $pageNav->limitstart, $pageNav->limit );
				$rows			=	$_CB_database->loadObjectList( null, '\CB\Plugin\FamilyPlans\Table\FamilyTable', array( $_CB_database ) );

				$viewer->setModel( $rows );

				if ( $type == 'from' ) {
					return $viewer->drawSharedFromList( $user, $pageNav, $searching );
				} else {
					return $viewer->drawSharedToList( $user, $pageNav, $searching );
				}
				break;
			case 'sharedsubscription':
				$userId			=	$this->input( 'user', 0, GetterInterface::INT );

				if ( ! $userId ) {
					$user		=	\CBuser::getMyUserDataInstance();
				} else {
					$user		=	\CBuser::getUserDataInstance( $userId );
				}

				if ( ! $user->get( 'id', 0, GetterInterface::INT ) ) {
					return null;
				}

				if ( ( Application::MyUser()->getUserId() != $user->get( 'id', 0, GetterInterface::INT ) ) && ( ! \cbpaidApp::authoriseAction( 'cbsubs.usersubscriptionmanage' ) ) ) {
					return null;
				}

				$subId			=	$this->input( 'subscription', 0, GetterInterface::INT );

				if ( ! $subId ) {
					return null;
				}

				$subscription	=	Helper::getSubscription( $user->get( 'id', 0, GetterInterface::INT ), $subId );

				if ( ( ! $subscription ) || ( ! $subscription->getPlan()->getParam( 'familyplans_share', 0, 'integrations' ) ) ) {
					return null;
				}

				$action			=	$this->input( 'post/act', null, GetterInterface::STRING );

				switch ( $action ) {
					case 'share':
						cbSpoofCheck( 'plugin' );

						$row	=	new FamilyTable();

						$row->set( 'user_id', $subscription->get( 'user_id', 0, GetterInterface::INT ) );
						$row->set( 'sub_id', $subscription->get( 'id', 0, GetterInterface::INT ) );
						$row->set( 'to', $this->input( 'post/to', null, GetterInterface::STRING ) );

						if ( $row->getError() || ( ! $row->check() ) ) {
							$_CB_framework->enqueueMessage( CBTxt::T( 'SHARED_SUB_FAILED_SAVE', 'Subscription failed to share! Error: [error]', array( '[error]' => $row->getError() ) ), 'error' );
							break;
						}

						if ( $row->getError() || ( ! $row->store() ) ) {
							$_CB_framework->enqueueMessage( CBTxt::T( 'SHARED_SUB_FAILED_SAVE', 'Subscription failed to share! Error: [error]', array( '[error]' => $row->getError() ) ), 'error' );
							break;
						}

						$_CB_framework->enqueueMessage( CBTxt::T( 'Successfully shared subscription!' ) );
						break;
					case 'delete':
					case 'resend':
						cbSpoofCheck( 'plugin' );

						$ids		=	cbToArrayOfInt( $this->input( 'post/ids', array(), GetterInterface::RAW ) );

						if ( ! $ids ) {
							$_CB_framework->enqueueMessage( CBTxt::T( 'Please select a shared subscription.' ), 'error' );
							break;
						}

						if ( ( $action == 'resend' ) && ( $subscription->get( 'status', null, GetterInterface::STRING ) != 'A' ) ) {
							$_CB_framework->enqueueMessage( CBTxt::T( 'Can not resend shared subscriptions for an inactive subscription.' ), 'error' );
							break;
						}

						$success	=	false;

						foreach ( $ids as $id ) {
							$row	=	new FamilyTable();

							$row->load( $id );

							if ( ( ! $row->get( 'id', 0, GetterInterface::INT ) )
								 || ( $row->get( 'user_id', 0, GetterInterface::INT ) != $subscription->get( 'user_id', 0, GetterInterface::INT ) )
								 || ( $row->get( 'sub_id', 0, GetterInterface::INT ) != $subscription->get( 'id', 0, GetterInterface::INT ) )
							) {
								continue;
							}

							if ( $action == 'delete' ) {
								if ( $row->getError() || ( ! $row->canDelete() ) ) {
									$_CB_framework->enqueueMessage( CBTxt::T( 'SHARED_SUB_TO_FAILED_DELETE', 'Shared subscription to [to] failed to delete! Error: [error]', array( '[to]' => htmlspecialchars( $row->get( 'to', null, GetterInterface::STRING ) ), '[error]' => $row->getError() ) ), 'error' );
									continue;
								}

								if ( $row->getError() || ( ! $row->delete() ) ) {
									$_CB_framework->enqueueMessage( CBTxt::T( 'SHARED_SUB_TO_FAILED_DELETE', 'Shared subscription to [to] failed to delete! Error: [error]', array( '[to]' => htmlspecialchars( $row->get( 'to', null, GetterInterface::STRING ) ), '[error]' => $row->getError() ) ), 'error' );
									continue;
								}

								$success	=	true;
							} else {
								if ( $row->get( 'user', 0, GetterInterface::INT ) ) {
									// Already accepted so skip the resend:
									continue;
								}

								if ( ! $row->send( true ) ) {
									$_CB_framework->enqueueMessage( CBTxt::T( 'SHARED_SUB_TO_FAILED_RESEND', 'Shared subscription to [to] failed to resend! Error: [error]', array( '[to]' => htmlspecialchars( $row->get( 'to', null, GetterInterface::STRING ) ), '[error]' => $row->getError() ) ), 'error' );
									continue;
								}

								$success	=	true;
							}
						}

						if ( $success ) {
							// Report success if at least something was deleted or resent:
							if ( $action == 'delete' ) {
								$_CB_framework->enqueueMessage( CBTxt::T( 'Successfully deleted shared subscriptions!' ) );
							} else {
								$_CB_framework->enqueueMessage( CBTxt::T( 'Successfully resent shared subscriptions!' ) );
							}
						}
						break;
				}

				/** @var $viewer \cbpaidSharedListView */
				$viewer			=	TemplateHandler::getViewer( null, 'sharedlist' );

				$prefix			=	'shared_';
				$limit			=	30;
				$limitstart		=	(int) $_CB_framework->getUserStateFromRequest( $prefix . 'limitstart{com_comprofiler}', $prefix . 'limitstart', 0 );
				$search			=	$_CB_framework->getUserStateFromRequest( $prefix . 'search{com_comprofiler}', $prefix . 'search', '' );
				$where			=	null;

				if ( $search ) {
					$where		.=	"\n AND ( f." . $_CB_database->NameQuote( 'to' ) . " LIKE " . $_CB_database->Quote( '%' . $_CB_database->getEscaped( $search, true ) . '%', false )
								.	" OR u." . $_CB_database->NameQuote( 'name' ) . " LIKE " . $_CB_database->Quote( '%' . $_CB_database->getEscaped( $search, true ) . '%', false )
								.	" OR u." . $_CB_database->NameQuote( 'email' ) . " LIKE " . $_CB_database->Quote( '%' . $_CB_database->getEscaped( $search, true ) . '%', false ) . " )";
				}

				$searching		=	( $where ? true : false );

				$query			=	"SELECT COUNT(*)"
								.	"\n FROM " . $_CB_database->NameQuote( '#__cbsubs_family' ) . " AS f"
								.	"\n LEFT JOIN " . $_CB_database->NameQuote( '#__users' ) . " AS u"
								.	" ON u." . $_CB_database->NameQuote( 'id' ) . " = f." . $_CB_database->NameQuote( 'user' )
								.	"\n WHERE f." . $_CB_database->NameQuote( 'user_id' ) . " = " . $subscription->get( 'user_id', 0, GetterInterface::INT )
								.	"\n AND f." . $_CB_database->NameQuote( 'sub_id' ) . " = " . $subscription->get( 'id', 0, GetterInterface::INT )
								.	$where;
				$_CB_database->setQuery( $query );
				$total			=	(int) $_CB_database->loadResult();

				$pageNav		=	new \cbPageNav( $total, $limitstart, $limit );

				$pageNav->setInputNamePrefix( $prefix );
				$pageNav->setStaticLimit( true );

				if ( Application::MyUser()->getUserId() != $user->get( 'id', 0, GetterInterface::INT ) ) {
					$pageNav->setBaseURL( $_CB_framework->pluginClassUrl( array( 'plugin' => 'cbpaidsubscriptions', 'do' => 'sharedsubscription', 'user' => $user->get( 'id', 0, GetterInterface::INT ), 'subscription' => $subscription->get( 'id', 0, GetterInterface::INT ) ), false ) );
				} else {
					$pageNav->setBaseURL( $_CB_framework->pluginClassUrl( array( 'plugin' => 'cbpaidsubscriptions', 'do' => 'sharedsubscription', 'subscription' => $subscription->get( 'id', 0, GetterInterface::INT ) ), false ) );
				}

				$query			=	"SELECT f.*"
								.	"\n FROM " . $_CB_database->NameQuote( '#__cbsubs_family' ) . " AS f"
								.	"\n LEFT JOIN " . $_CB_database->NameQuote( '#__users' ) . " AS u"
								.	" ON u." . $_CB_database->NameQuote( 'id' ) . " = f." . $_CB_database->NameQuote( 'user' )
								.	"\n WHERE f." . $_CB_database->NameQuote( 'user_id' ) . " = " . $subscription->get( 'user_id', 0, GetterInterface::INT )
								.	"\n AND f." . $_CB_database->NameQuote( 'sub_id' ) . " = " . $subscription->get( 'id', 0, GetterInterface::INT )
								.	$where
								.	"\n ORDER BY f." . $_CB_database->NameQuote( 'sent' ) . " DESC";
				$_CB_database->setQuery( $query, $pageNav->limitstart, $pageNav->limit );
				$rows			=	$_CB_database->loadObjectList( null, '\CB\Plugin\FamilyPlans\Table\FamilyTable', array( $_CB_database ) );

				$viewer->setModel( $rows );

				return $viewer->drawSharedList( $user, $subscription, $pageNav, $searching );
				break;
			case 'sharedcancel':
				$userId			=	$this->input( 'user', 0, GetterInterface::INT );

				if ( ! $userId ) {
					$user		=	\CBuser::getMyUserDataInstance();
				} else {
					$user		=	\CBuser::getUserDataInstance( $userId );
				}

				if ( ! $user->get( 'id', 0, GetterInterface::INT ) ) {
					return null;
				}

				if ( ( Application::MyUser()->getUserId() != $user->get( 'id', 0, GetterInterface::INT ) ) && ( ! \cbpaidApp::authoriseAction( 'cbsubs.usersubscriptionmanage' ) ) ) {
					return null;
				}

				$subId			=	$this->input( 'subscription', 0, GetterInterface::INT );

				if ( ! $subId ) {
					return null;
				}

				$row			=	new FamilyTable();

				$row->load( array( 'sub_id' => $subId, 'user' => $user->get( 'id', 0, GetterInterface::INT ) ) );

				$returnUrl		=	$_CB_framework->userProfileUrl( $user->get( 'id', 0, GetterInterface::INT ), false, 'getcbpaidsubscriptionsTab' );

				if ( ! $row->get( 'id', 0, GetterInterface::INT ) ) {
					cbRedirect( $returnUrl, CBTxt::T( 'Shared subscription does not exist!' ), 'error' );
				}

				if ( $row->getError() || ( ! $row->canDelete() ) ) {
					cbRedirect( $returnUrl, CBTxt::T( 'SHARED_SUB_FAILED_CANCEL', 'Shared subscription failed to cancel! Error: [error]', array( '[error]' => $row->getError() ) ), 'error' );
				}

				if ( $row->getError() || ( ! $row->delete() ) ) {
					cbRedirect( $returnUrl, CBTxt::T( 'SHARED_SUB_FAILED_CANCEL', 'Shared subscription failed to cancel! Error: [error]', array( '[error]' => $row->getError() ) ), 'error' );
				}

				cbRedirect( $returnUrl, CBTxt::T( 'Successfully cancelled shared subscription!' ) );
				break;
			case 'sharedaccept':
				$user			=	\CBuser::getMyUserDataInstance();

				if ( $user->get( 'id', 0, GetterInterface::INT ) ) {
					$returnUrl	=	$_CB_framework->userProfileUrl( $user->get( 'id', 0, GetterInterface::INT ), false, 'getcbpaidsubscriptionsTab' );
				} else {
					$returnUrl	=	'index.php';
				}

				$code			=	$this->input( 'code', null, GetterInterface::STRING );

				if ( ! $code ) {
					cbRedirect( $returnUrl, CBTxt::T( 'Shared subscription does not exist!' ), 'error' );
				}

				$row			=	new FamilyTable();

				$row->load( array( 'code' => $code ) );

				if ( ( ! $row->get( 'id', 0, GetterInterface::INT ) ) || $row->get( 'user', 0, GetterInterface::INT ) ) {
					cbRedirect( $returnUrl, CBTxt::T( 'Shared subscription does not exist!' ), 'error' );
				}

				if ( ! $user->get( 'id', 0, GetterInterface::INT ) ) {
					// Add the shared plan code to session so we don't have to keep passing it around in POST data:
					Application::Session()->set( 'cbsubs.shared.' . $row->getSubscription()->getPlanAttribute( 'id' ), $code );

					cbRedirect( $_CB_framework->pluginClassUrl( array( 'plugin' => 'cbpaidsubscriptions', 'do' => 'displayplans', 'plans' => $row->getSubscription()->getPlanAttribute( 'id' ) ), false ), CBTxt::T( 'Please login or register to accept this shared subscription.' ) );
				}

				if ( $row->getError() || ( ! $row->canAccept( $user->get( 'id', 0, GetterInterface::INT ) ) ) ) {
					cbRedirect( $returnUrl, CBTxt::T( 'SHARED_SUB_FAILED_ACCEPT', 'Shared subscription failed to accept! Error: [error]', array( '[error]' => $row->getError() ) ), 'error' );
				}

				if ( $row->getError() || ( ! $row->accept( $user ) ) ) {
					cbRedirect( $returnUrl, CBTxt::T( 'SHARED_SUB_FAILED_ACCEPT', 'Shared subscription failed to accept! Error: [error]', array( '[error]' => $row->getError() ) ), 'error' );
				}

				cbRedirect( $returnUrl, CBTxt::T( 'Successfully accepted shared subscription!' ) );
				break;
			case 'sharedreject':
				$user			=	\CBuser::getMyUserDataInstance();

				if ( $user->get( 'id', 0, GetterInterface::INT ) ) {
					$returnUrl	=	$_CB_framework->userProfileUrl( $user->get( 'id', 0, GetterInterface::INT ), false, 'getcbpaidsubscriptionsTab' );
				} else {
					$returnUrl	=	'index.php';
				}

				$code			=	$this->input( 'code', null, GetterInterface::STRING );

				if ( ! $code ) {
					cbRedirect( $returnUrl, CBTxt::T( 'Shared subscription does not exist!' ), 'error' );
				}

				$row			=	new FamilyTable();

				$row->load( array( 'code' => $code ) );

				if ( ( ! $row->get( 'id', 0, GetterInterface::INT ) ) || $row->get( 'user', 0, GetterInterface::INT ) ) {
					cbRedirect( $returnUrl, CBTxt::T( 'Shared subscription does not exist!' ), 'error' );
				}

				if ( $row->getError() || ( ! $row->delete() ) ) {
					cbRedirect( $returnUrl, CBTxt::T( 'SHARED_SUB_FAILED_REJECT', 'Shared subscription failed to reject! Error: [error]', array( '[error]' => $row->getError() ) ), 'error' );
				}

				if ( ! $user->get( 'id', 0, GetterInterface::INT ) ) {
					Application::Session()->unsetEntry( 'cbsubs.shared.' . $row->getSubscription()->getPlanAttribute( 'id' ) );
				}

				$subject		=	$row->getSubscription()->getPlan()->getParam( 'familyplans_notify_reject_sub', null, 'integrations' );
				$message		=	$row->getSubscription()->getPlan()->getParam( 'familyplans_notify_reject_msg', null, 'integrations' );

				Helper::sendNotification( $row, $row->get( 'user_id', 0, GetterInterface::INT ), $subject, $message );

				cbRedirect( $returnUrl, CBTxt::T( 'Successfully rejected shared subscription!' ) );
				break;
		}

		return null;
	}

	/**
	 * @param \cbpaidUsersubscriptionRecord[] $subscriptions
	 * @param string                          $status
	 * @param bool                            $ordered
	 * @param \cbpaidUserExtension            $extension
	 */
	public function getSharedSubscriptions( &$subscriptions, $status, $ordered, $extension )
	{
		global $_CB_database;

		if ( ! $extension->id ) {
			// Guests can't have or share subscriptions so skip this:
			return;
		}

		$query					=	"SELECT f.*"
								.	"\n FROM " . $_CB_database->NameQuote( '#__cbsubs_family' ) . " AS f"
								.	"\n INNER JOIN " . $_CB_database->NameQuote( '#__cbsubs_subscriptions' ) . " AS s"
								.	" ON s." . $_CB_database->NameQuote( 'id' ) . " = f." . $_CB_database->NameQuote( 'sub_id' )
								.	"\n WHERE f." . $_CB_database->NameQuote( 'user' ) . " = " . (int) $extension->id;
		if ( $status ) {
			$query				.=	"\n AND s." . $_CB_database->NameQuote( 'status' ) . " = " . $_CB_database->Quote( $status );
		}
		$_CB_database->setQuery( $query );
		/** @var FamilyTable[] $familySubscriptions */
		$familySubscriptions	=	$_CB_database->loadObjectList( null, '\CB\Plugin\FamilyPlans\Table\FamilyTable', array( $_CB_database ) );

		if ( ! $familySubscriptions ) {
			// This user was not shared any subscriptions so skip the subscription union:
			return;
		}

		// Make sure the subscriptions can still be shared as a safety net encase admin turned off sharing:
		$shared				=	array();

		foreach ( $familySubscriptions as $familySubscription ) {
			$subscription	=	$familySubscription->getSubscription();

			if ( ( ! $subscription ) || ( ! $subscription->getPlan()->getParam( 'familyplans_share', 0, 'integrations' ) ) ) {
				continue;
			}

			// Make sure they don't already have a subscription and if they do and it's inactive lets hide it:
			foreach ( $subscriptions as $i => $existingSubscription ) {
				if ( $existingSubscription->getPlanAttribute( 'id' ) != $subscription->getPlanAttribute( 'id' ) ) {
					continue;
				}

				// The user already has a subscription to this plan so lets suppress the shared subscription since users own subscription has priority:
				if ( $existingSubscription->get( 'status', null, GetterInterface::STRING ) == 'A' ) {
					continue 2;
				} else {
					unset( $subscriptions[$i] );
				}
			}

			$shared[]		=	$subscription;
		}

		if ( ! $shared ) {
			// There were shared subscriptions, but looks like they don't allow sharing anymore so lets stop here:
			return;
		}

		$subscriptions	=	( $subscriptions + $shared );
	}

	/**
	 * @param UserTable        $user
	 * @param string           $status
	 * @param int              $planId
	 * @param int              $replacedPlanId
	 * @param ParamsInterface  $integrationParams
	 * @param string           $cause
	 * @param string           $reason
	 * @param int              $now
	 * @param \cbpaidSomething $subscription
	 * @param int              $autorenewed
	 */
	public function sharedSubscriptionStateChanged( &$user, $status, $planId, $replacedPlanId, &$integrationParams, $cause, $reason, $now, &$subscription, $autorenewed )
	{
		global $_CB_database;

		if ( ( ! $subscription->getPlan()->getParam( 'familyplans_share', 0, 'integrations' ) )
			 || ( $user->get( 'id', 0, GetterInterface::INT ) != $subscription->get( 'user_id', 0, GetterInterface::INT ) )
		) {
			return;
		}

		$query		=	"SELECT *"
					.	"\n FROM " . $_CB_database->NameQuote( '#__cbsubs_family' )
					.	"\n WHERE " . $_CB_database->NameQuote( 'user_id' ) . " = " . $user->get( 'id', 0, GetterInterface::INT )
					.	"\n AND " . $_CB_database->NameQuote( 'sub_id' ) . " = " . $subscription->get( 'id', 0, GetterInterface::INT )
					.	"\n AND " . $_CB_database->NameQuote( 'user' ) . " > 0";
		$_CB_database->setQuery( $query );
		$rows		=	$_CB_database->loadObjectList( null, '\CB\Plugin\FamilyPlans\Table\FamilyTable', array( $_CB_database ) );

		/** @var FamilyTable[] $rows */
		foreach ( $rows as $row ) {
			$userId					=	$row->get( 'user', 0, GetterInterface::INT );

			if ( ( ! $userId ) || ( $userId == $user->get( 'id', 0, GetterInterface::INT ) ) ) {
				// Ignore passing events through to family member if the family member is the users self or doesn't exist:
				continue;
			}

			$family					=	\CBuser::getUserDataInstance( $userId );

			if ( ! $family->get( 'id', 0, GetterInterface::INT ) ) {
				// Ignore passing events through for non-existing users:
				continue;
			}

			$pseudoSubscription		=	$row->getSubscription();

			if ( ! $pseudoSubscription ) {
				// Unable to find the original subscription so just stop here (we don't want to use the one given to this function):
				break;
			}

			$pseudoSubscription->triggerIntegrations( $family, $cause, $replacedPlanId, null, $reason, $autorenewed );
		}
	}

	/**
	 * Override the plan button display
	 *
	 * @param \cbpaidSomething $subscription
	 * @param string          $childrenRendering
	 * @param string          $controlButtons
	 * @param bool            $showRenewButtons
	 * @param bool            $showUnsubscribeButtons
	 * @param int             $now
	 * @param UserTable       $user
	 */
	public function displaySubscription( &$subscription, &$childrenRendering, &$controlButtons, $showRenewButtons, $showUnsubscribeButtons, $now, $user )
	{
		if ( ( ! $user->get( 'id', 0, GetterInterface::INT ) ) || ( ! $subscription->getPlan()->getParam( 'familyplans_share', 0, 'integrations' ) ) ) {
			return;
		}

		if ( ( $subscription instanceof PseudoSubscriptionTable ) || ( $subscription instanceof PseudoMerchandiseTable ) || ( $subscription instanceof PseudoDonationTable ) ) {
			// The user has a subscription that allows sharing and they're NOT the owner of it so display the cancel button to allow leaving the family:

			/** @var $viewer \cbpaidSharedListView */
			$viewer			=	TemplateHandler::getViewer( null, 'sharedlist' );

			$controlButtons	=	$viewer->drawCancelButton( $user, $subscription );
		} else {
			// The user has a subscription that allows sharing and they're the owner of it so display the share button:

			/** @var $viewer \cbpaidSharedListView */
			$viewer			=	TemplateHandler::getViewer( null, 'sharedlist' );

			$controlButtons	=	$viewer->drawShareButton( $user, $subscription ) . $controlButtons;
		}
	}

	/**
	 * @param UserTable $user
	 * @param bool      $displayToMe
	 * @param bool      $itsmyself
	 * @param string    $htmlSubscriptionsAndUpgrades
	 * @param string    $htmlInvoicesLink
	 * @param string    $htmlPaymentsLink
	 * @param string    $tabTitleText
	 * @param string    $htmlTabDescription
	 * @return string
	 */
	public function displaySubscriptions( $user, $displayToMe, $itsmyself, &$htmlSubscriptionsAndUpgrades, &$htmlInvoicesLink, &$htmlPaymentsLink, &$tabTitleText, &$htmlTabDescription )
	{
		global $_CB_database;

		/** @var $viewer \cbpaidSharedListView */
		$viewer			=	TemplateHandler::getViewer( null, 'sharedlist' );
		$title			=	CBTxt::Th( 'Shared Subscriptions' );
		$return			=	null;

		$query			=	"SELECT COUNT(*)"
						.	"\n FROM " . $_CB_database->NameQuote( '#__cbsubs_family' ) . " AS f"
						.	"\n INNER JOIN " . $_CB_database->NameQuote( '#__cbsubs_subscriptions' ) . " AS s"
						.	" ON s." . $_CB_database->NameQuote( 'id' ) . " = f." . $_CB_database->NameQuote( 'sub_id' )
						.	"\n WHERE f." . $_CB_database->NameQuote( 'user_id' ) . " = " . $user->get( 'id', 0, GetterInterface::INT );
		$_CB_database->setQuery( $query );
		$fromUser		=	(int) $_CB_database->loadResult();

		if ( $fromUser ) {
			$return		.=	$viewer->drawSharedFromLink( $user, $title );
		}

		$query			=	"SELECT COUNT(*)"
						.	"\n FROM " . $_CB_database->NameQuote( '#__cbsubs_family' ) . " AS f"
						.	"\n INNER JOIN " . $_CB_database->NameQuote( '#__cbsubs_subscriptions' ) . " AS s"
						.	" ON s." . $_CB_database->NameQuote( 'id' ) . " = f." . $_CB_database->NameQuote( 'sub_id' )
						.	"\n WHERE f." . $_CB_database->NameQuote( 'user' ) . " = " . $user->get( 'id', 0, GetterInterface::INT );
		$_CB_database->setQuery( $query );
		$toUser			=	(int) $_CB_database->loadResult();

		if ( $toUser ) {
			$return		.=	$viewer->drawSharedToLink( $user, ( ! $fromUser ? $title : null ) );
		}

		return $return;
	}

	/**
	 * Alters the list of available plans if there's a shared subscription available
	 *
	 * @param \cbpaidProduct[] $plans
	 * @param UserTable|null   $user
	 * @param boolean          $published
	 * @param string           $forCause
	 * @param int|null         $owner
	 */
	public function loadPlans( &$plans, $user, $published, $forCause, $owner )
	{
		if ( $forCause != 'registration' ) {
			return;
		}

		$sharedPlans			=	array();

		// Override registration plan display and selection if a shared subscription is being accepted:
		foreach ( \cbpaidPlansMgr::getInstance()->loadPublishedPlans( null, true, 'any', null ) as $k => $plan ) {
			$row				=	$this->getShared( $plan );

			if ( ( ! $row ) || ( ! $row->canAccept() ) ) {
				continue;
			}

			$sharedPlans[$k]	=	$this->overridePlan( $plan );
		}

		if ( $sharedPlans ) {
			$plans				=	$sharedPlans;
		}
	}

	/**
	 * Override the price on plan display
	 *
	 * @param \cbpaidProduct $plan
	 * @param string         $childrenRendering
	 * @param string         $reason
	 * @param boolean        $drawOnlyAsContainer
	 * @param int            $userId
	 * @return null|string
	 */
	public function displayPlan( &$plan, &$childrenRendering, $reason, $drawOnlyAsContainer, $userId )
	{
		global $_CB_framework;

		if ( $userId ) {
			return null;
		}

		$row		=	$this->getShared( $plan );

		if ( ( ! $row ) || ( ! $row->canAccept( $userId ) ) ) {
			return null;
		}

		$plan		=	$this->overridePlan( $plan );

		$fromUser	=	\CBuser::getInstance( $row->get( 'user_id', 0, GetterInterface::INT ), false )->getField( 'formatname', null, 'html', 'none', 'list', 0, true );
		$rejectUrl	=	$_CB_framework->pluginClassUrl( array( 'plugin' => 'cbpaidsubscriptions', 'do' => 'sharedreject', 'code' => $row->get( 'code', null, GetterInterface::STRING ) ) );

		return CBTxt::Th( 'SUBSCRIPTION_SHARED_FROM_USER_CAN_REJECT', 'This is a shared subscription from [user]. If you do not want to accept this shared subscription <a href="[reject_url]">click here to reject it</a>.', array( '[user]' => $fromUser, '[reject_url]' => $rejectUrl ) );
	}

	/**
	 * Override the price display as we don't want shared subscriptions price to actually output
	 *
	 * @param \cbpaidProduct     $plan
	 * @param string             $ret
	 * @param array              $method
	 * @param array              $args
	 * @param int                $userId
	 * @param \cbpaidPaymentItem $paymentItem
	 */
	public function displayPrice( $plan, &$ret, $method, $args, $userId, $paymentItem )
	{
		$row	=	$this->getShared( $plan );

		if ( ( ! $row ) || ( ! $row->canAccept( $userId ) ) ) {
			return;
		}

		$ret	=	null;
	}

	/**
	 * Validates a shared subscription during registration
	 *
	 * @param null|string|\cbpaidProduct[] $chosenPlans
	 * @param UserTable                    $user
	 * @return null|bool
	 */
	public function sharedSubscriptionValidation( &$chosenPlans, $user )
	{
		global $_PLUGINS;

		if ( ( ! $chosenPlans ) || ( ! is_array( $chosenPlans ) ) ) {
			return null;
		}

		$userId						=	$user->get( 'id', 0, GetterInterface::INT );
		$sharedExists				=	false;
		$nonSharedExists			=	false;

		foreach ( $chosenPlans as $k => $chosenPlan ) {
			$row					=	$this->getShared( $chosenPlan );

			if ( $row ) {
				if ( ! $row->canAccept( $userId ) ) {
					// This shared subscription can not be accepted by this user so lets error and clear the cache:
					Application::Session()->unsetEntry( 'cbsubs.shared.' . $chosenPlan->get( 'id', 0, GetterInterface::INT ) );

					$_PLUGINS->raiseError();
					$_PLUGINS->_setErrorMSG( CBTxt::T( 'SHARED_SUB_FAILED_ACCEPT', 'Shared subscription failed to accept! Error: [error]', array( '[error]' => $row->getError() ) ) );

					return false;
				}

				$chosenPlans[$k]	=	$this->overridePlan( $chosenPlan );

				$sharedExists		=	true;
			} else {
				$nonSharedExists	=	true;
			}
		}

		if ( $sharedExists && $nonSharedExists ) {
			// If both shared and non-shared exist we need to block the registration since shared must only be accepted by itself:
			$_PLUGINS->raiseError();
			$_PLUGINS->_setErrorMSG( CBTxt::T( 'You can not accept a shared subscription and subscribe to another plan at the same time.' ) );

			return false;
		}

		return null;
	}

	/**
	 * Accepts a shared subscription during registration
	 *
	 * @param null|string|\cbpaidProduct[] $chosenPlans
	 * @param UserTable                    $user
	 * @return null|bool
	 */
	public function sharedSubscriptionRegistration( &$chosenPlans, $user )
	{
		global $_PLUGINS;

		if ( ( ! $chosenPlans ) || ( ! is_array( $chosenPlans ) ) ) {
			return null;
		}

		$userId					=	$user->get( 'id', 0, GetterInterface::INT );
		$sharedSubscription		=	null;

		foreach ( $chosenPlans as $k => $chosenPlan ) {
			$row				=	$this->getShared( $chosenPlan );

			if ( ! $row ) {
				continue;
			}

			if ( ! $row->canAccept( $userId ) ) {
				continue;
			}

			$chosenPlans[$k]	=	$this->overridePlan( $chosenPlan );

			$sharedSubscription	=	$row;
			break;
		}

		if ( ! $sharedSubscription ) {
			return null;
		}

		if ( ! $sharedSubscription->accept( $user ) ) {
			Application::Session()->unsetEntry( 'cbsubs.shared.' . $chosenPlan->get( 'id', 0, GetterInterface::INT ) );

			$_PLUGINS->raiseError();
			$_PLUGINS->_setErrorMSG( CBTxt::T( 'SHARED_SUB_FAILED_ACCEPT', 'Shared subscription failed to accept! Error: [error]', array( '[error]' => $sharedSubscription->getError() ) ) );

			return false;
		}

		return true;
	}

	/**
	 * Get the shared family object from shared session code by plan id
	 *
	 * @param \cbpaidProduct $plan
	 * @return null|FamilyTable
	 */
	private function getShared( $plan )
	{
		if ( ! $plan->getParam( 'familyplans_share', 0, 'integrations' ) ) {
			return null;
		}

		$code	=	Application::Session()->get( 'cbsubs.shared.' . $plan->get( 'id', 0, GetterInterface::INT ), null, GetterInterface::STRING );

		if ( ! $code ) {
			return null;
		}

		$row	=	new FamilyTable();

		$row->load( array( 'code' => $code ) );

		if ( ! $row->get( 'id', 0, GetterInterface::INT ) ) {
			Application::Session()->unsetEntry( 'cbsubs.shared.' . $plan->get( 'id', 0, GetterInterface::INT ) );

			return null;
		}

		return $row;
	}

	/**
	 * Overrides the plan to ensure access
	 *
	 * @param \cbpaidProduct $plan
	 * @return \cbpaidProduct
	 */
	private function overridePlan( $plan )
	{
		$plan->setOverride( 'exclusive', 1 );
		$plan->setOverride( 'first_rate', 0 );
		$plan->setOverride( 'rate', 0 );
		$plan->setOverride( 'pricedisplay', ' ' );
		$plan->setOverride( 'allow_registration', 1 );
		$plan->setOverride( 'propose_registration', 1 );
		$plan->setOverride( 'reg_conds', 0 );
		$plan->setOverride( 'allow_upgrade_to_this', 1 );
		$plan->setOverride( 'auto_compute_upgrade', 0 );
		$plan->setOverride( 'propose_upgrade', 1 );
		$plan->setOverride( 'cond_1_operator', 0 );
		$plan->setOverride( 'upgrade_conds', 0 );
		$plan->setOverride( 'allow_newsubscriptions', 1 );
		$plan->setOverride( 'allow_frontend', 1 );
		$plan->setOverride( 'viewaccesslevel', 1 );
		$plan->setOverride( 'access', 1 );

		return $plan;
	}
}
