<?php
/**
 * @version $Id: cbpaidWebservices.php 1541 2012-11-23 22:21:52Z beat $
 * @package CBSubs (TM) Community Builder Plugin for Paid Subscriptions (TM)
 * @subpackage Plugin for Paid Subscriptions
 * @copyright (C) 2007-2022 and Trademark of Lightning MultiCom SA, Switzerland - www.joomlapolis.com - and its licensors, all rights reserved
 * @license http://www.gnu.org/licenses/old-licenses/gpl-2.0.html GNU/GPL version 2
 */

/** ensure this file is being included by a parent file */
if ( ! ( defined( '_VALID_CB' ) || defined( '_JEXEC' ) || defined( '_VALID_MOS' ) ) ) { die( 'Direct Access to this location is not allowed.' ); }

/**
 * CBSubs HTTPS remote web-services requests class
 */
class cbpaidWebservices {
	/**
	 * Posts a POST form by https or http and gets result.
	 *
	 * @param  string  $urlNoHttpsPrefix  URL without https:// in front (but works also with http:// or https:// in front, but it's ignored.
	 * @param  array|string  $formvars          Variables in form to post
	 * @param  int     $timeout           Timeout of the access
	 * @param  string  $result            RETURNING: the fetched access
	 * @param  int     $status            RETURNING: the status of the access (e.g. 200 is normal)
	 * @param  string  $getPostType       'post' (default) or 'get'
	 * @param  string  $contentType       'normal' (default) or 'xml' ($formvars['xml']=xml content) or 'json' (application/json)
	 * @param  string  $acceptType        '* / *' (default) or 'application/xml' or 'application/json'; also handles accept-version header (e.g. * / * / v10)
	 * @param  boolean $https             SSL protocol (default: true)
	 * @param  int     $port              port number (default: 443)
	 * @param  string  $username          HTTP username authentication
	 * @param  string  $password          HTTP password authentication
	 * @param  boolean $allowHttpFallback Allow fallback to http if https not available locally (default: false)
	 * @param  string  $referer           referrer
	 * @param  array   $headers           HTTP headers
	 * @return int     $error             error-code of access (0 for ok)
	 */
	public static function httpsRequestGuzzle( $urlNoHttpsPrefix, $formvars, $timeout, &$result, &$status, $getPostType = 'post', $contentType='normal', $acceptType='*/*', $https = true, $port = 443, $username = '', $password = '', $allowHttpFallback = false, $referer = null, $headers = array() ) {
		$urlNoHttpsPrefix					=	preg_replace( '/^https?:\/\//', '', $urlNoHttpsPrefix );

		// Check if protocol version has been overridden otherwise default to HTTP 1.1
		$typeParts							=	explode( '/', $getPostType, 2 );
		$getPostType						=	( isset( $typeParts[0] ) ? $typeParts[0] : 'post' );
		$getPostVersion						=	( isset( $typeParts[1] ) ? (float) $typeParts[1] : 1.1 );

		try {
			if ( $port && ( ( $https && ( $port != 443 ) ) || ( ( ! $https ) && ( $port != 80 ) ) ) ) {
				$portPostfix				=	':' . (int) $port;
			} else {
				$portPostfix				=	'';
			}

			$url							=	( $https && ( ! $allowHttpFallback ) ? 'https://' : 'http://' ) . $urlNoHttpsPrefix . $portPostfix;

			$request						=	new \GuzzleHttp\Client();

			if ( $timeout ) {
				$headers['timeout']			=	(int) $timeout;
			}

			if ( $contentType == 'xml' ) {
				$headers['Content-Type']	=	'application/xml';
			} elseif ( $contentType == 'json' ) {
				$headers['Content-Type']	=	'application/json';
			} else {
				$headers['Content-Type']	=	'application/x-www-form-urlencoded';
			}

			if ( $username || $password ) {
				$headers['Authorization']	=	'Basic ' . base64_encode( $username . ':' . $password );
			}

			$acceptVersion					=	null;

			if ( $acceptType != '*/*' ) {
				$acceptParts				=	explode( '/', $acceptType );

				if ( count( $acceptParts ) > 1 ) {
					$acceptType				=	$acceptParts[0] . '/' . $acceptParts[1];
					$acceptVersion			=	$acceptParts[2];
				}

				if ( $acceptType != '*/*' ) {
					$headers['Accept']		=	$acceptType;
				}
			}

			if ( $acceptVersion ) {
				$headers['Accept-Version']	=	$acceptVersion;
			}

			if ( $referer ) {
				$headers['Referer']			=	$referer;
			}

			if ( $referer ) {
				$headers['Port']			=	$port;
			}

			$options						=	array();

			if ( $headers ) {
				$options['headers']			=	$headers;
			}

			if ( $getPostVersion != 1.1 ) {
				$options['version']			=	$getPostVersion;
			}

			switch ( strtoupper( $getPostType ) ) {
				case 'DELETE':
					if ( $formvars ) {
						$options['query']	=	$formvars;
					}

					$results				=	$request->delete( $url, $options );
					break;
				case 'GET':
					if ( $formvars ) {
						$options['query']	=	$formvars;
					}

					$results				=	$request->get( $url, $options );
					break;
				case 'POST':
				default:
					if ( $formvars ) {
						if ( ( cbGuzzleVersion() >= 6 ) && ( ! is_string( $formvars ) ) ) {
							$options['form_params']		=	$formvars;
						} else {
							$options['body']			=	$formvars;
						}
					}

					$results				=	$request->post( $url, $options );
					break;
			}

			$status							=	(int) $results->getStatusCode();

			if ( $status != 200 ) {
				$error						=	$results->getReasonPhrase();
			} else {
				$error						=	null;
			}

			$result							=	(string) $results->getBody();
		} catch( \GuzzleHttp\Exception\ClientException $e ) {
			if ( $e->hasResponse() ) {
				$status						=	(int) $e->getResponse()->getStatusCode();
				$error						=	$e->getResponse()->getReasonPhrase();
				$result						=	(string) $e->getResponse()->getBody();
			} else {
				$status						=	$e->getCode();
				$error						=	$e->getMessage();
			}
		}

		return $error;
	}
	/**
	 * Posts a POST form by https if available, otherwise by http and gets result.
	 * @deprecated 4.3, to be removed in 5.0, replaced by httpsRequestGuzzle() with same parameters
	 *
	 * @param  string  $urlNoHttpsPrefix  URL without https:// in front (but works also with http:// or https:// in front, but it's ignored.
	 * @param  array|string  $formvars          Variables in form to post
	 * @param  int     $timeout           Timeout of the access
	 * @param  string  $result            RETURNING: the fetched access
	 * @param  int     $status            RETURNING: the status of the access (e.g. 200 is normal)
	 * @param  string  $getPostType       'post' (default), 'get', or 'delete'; optionally override protocol version with type/VERSION (e.g. get/1.0, post/1.0)
	 * @param  string  $contentType       'normal' (default) or 'xml' ($formvars['xml']=xml content) or 'json' (application/json)
	 * @param  string  $acceptType        '* / *' (default) or 'application/xml' or 'application/json'
	 * @param  boolean $https             SSL protocol (default: true)
	 * @param  int     $port              port number (default: 443)
	 * @param  string  $username          HTTP username authentication
	 * @param  string  $password          HTTP password authentication
	 * @param  boolean $allowHttpFallback Allow fallback to http if https not available locally (default: false)
	 * @param  string  $referer           referrer
	 * @return int     $error             error-code of access (0 for ok)
	 */
	public static function httpsRequest( $urlNoHttpsPrefix, $formvars, $timeout, &$result, &$status, $getPostType = 'post', $contentType='normal', $acceptType='*/*', $https = true, $port = 443, $username = '', $password = '', $allowHttpFallback = false, $referer = null ) {
		$urlNoHttpsPrefix	=	preg_replace( '/^https?:\/\//', '', $urlNoHttpsPrefix );

		// Check if protocol version has been overridden otherwise default to HTTP 1.1
		$typeParts			=	explode( '/', $getPostType, 2 );
		$getPostType		=	( isset( $typeParts[0] ) ? $typeParts[0] : 'post' );
		$getPostVersion		=	( isset( $typeParts[1] ) ? $typeParts[1] : '1.1' );

		if ( is_callable( 'fsockopen' ) && ( ( ! $https ) || ( version_compare( phpversion(), '4.3.0', '>' ) && extension_loaded( 'openssl' ) && defined( 'OPENSSL_VERSION_TEXT' ) ) ) ) {

			// otherwise try fsockopen:

			if ( is_array( $formvars ) ) {
				$formUrl			=	array();
				foreach ( $formvars as $k => $v ) {
					$formUrl[$k]	=	urlencode( $k ) . '=' . urlencode( $v );
				}
				$formUrl			=	implode( '&', $formUrl );
			} else {
				$formUrl			=	$formvars;
			}
			$urlParts				=	explode( '/', $urlNoHttpsPrefix, 2 );

			$posturl				=	( $https ? 'ssl://' : 'tcp://' ) . $urlParts[0];

			if ( $getPostType == 'post' ) {
				$header				=	'POST';
			} elseif ( $getPostType == 'delete' ) {
				$header				=	'DELETE';
			} else {
				$header				=	'GET';
			}
			$header					.=	' /' . ( count( $urlParts ) == 2 ? $urlParts[1] : '' ) . " HTTP/$getPostVersion\r\n";
			$header					.=	'Host: ' . $urlParts[0] . "\r\n";
			if ( $username || $password ) {
				$header				.=	'Authorization: Basic ' . base64_encode( $username . ':' . $password ) . "\r\n";
			}
			$header					.=	'User-Agent: PHP Script' . "\r\n";
			if ( $referer ) {
				$header				.=	'Referer: ' . $referer . "\r\n";
			}
			if ( $contentType == 'xml' ) {
				$header				.=	'Content-Type: application/xml' . "\r\n";
			} elseif ( $contentType == 'json' ) {
				$header				.=	'Content-Type: application/json' . "\r\n";
			} else {
				$header				.=	'Content-Type: application/x-www-form-urlencoded' . "\r\n";			//TODO: 				$header				.=	'Content-Type: application/x-www-form-urlencoded;charset=\"utf-8\"' . "\r\n";
			}
			if ( $acceptType != '*/*' ) {
				$header				.=	'Accept: ' . $acceptType . "\r\n";
			}
			$header					.=	'Content-Length: ' . strlen( $formUrl ) . "\r\n";
			$header					.=	'Coonection: close' . "\r\n";	// HTTP 1.1
			$header					.=	"\r\n";
			$error					=	null;
			$errstr					=	null;
			$status					=	100;

			$fp						=	@fsockopen ( $posturl, $port, $error, $errstr, $timeout );
			if ( $fp ) {
				if ( is_callable( 'stream_set_timeout' ) ) {
					stream_set_timeout( $fp, $timeout );
				}
				$bytesWritten		=	@fwrite( $fp, $header . $formUrl );
				if ( $bytesWritten !== false ) {
					$response		=	array();
					$result			=	'';
					while (!feof($fp)) {
						$line		=	@fgets( $fp, 8192 );
						if ( trim( $line ) == '' ) {
							break;
						}
						$response[]	=	$line;
					}

					$isChunked		=	( in_array( "Transfer-Encoding: chunked\r\n", $response ) );

					while (!feof($fp)) {
						$chunk		=	@fread( $fp, 8192 );

						if ( $isChunked ) {
							if ( ( strlen( $result ) > 0 ) && ( $chunk === "\r\n" ) ) {
								$chunk	=	'';
							} elseif ( ! static::unchunkHttpResponse( $chunk ) ) {
								break;
							}
						}

						$result		.=	$chunk;
					}

					@fclose ( $fp );

					if ( count( $response ) > 0 ) {
						$parts				=	explode( ' ', $response[0], 3 );
						if ( count( $parts ) == 3 ) {
							if ( ( trim( $parts[2] ) == 'OK' ) || ( preg_match( '/^\s*\d{3}\s*/', $parts[1] ) ) ) {
								$status		=	(int) $parts[1];	// 200 hopefully.
							}
						}
					}

					return $error;
				} else {
					fclose( $fp );
				}
			}

		}
		{
			// then try using curl executable:

			cbimport( 'cb.snoopy' );

			$curl_found				=	false;
			$path					=	null;
			if(function_exists('is_executable')) {
				$paths = array( '/usr/bin/curl', '/usr/local/bin/curl', 'curl' );	// IN SNOOPY ALREADY: '/usr/local/bin/curl'
				foreach ($paths as $path) {
					if ( @is_executable( $path ) ) {
						$curl_found = true;
						break;
					}
				}
			}

			//		if ( $curl_found ) {		// we will do http as last resort without using curl if curl_found == false:

			$s					=	new CBSnoopy();
			$s->curl_path = $path;

			if ( ! is_array( $formvars ) ) {
				if ( $contentType == 'xml' ) {
					$formvars	=	array( 'xml' => $formvars );
				} else {
					$formarr	=	explode( '&', $formvars );
					$formvars	=	array();
					foreach ( $formarr as $v ) {
						$p		=	explode( '=', $v, 2 );
						if ( count( $p ) == 2 ) {
							$formvars[$p[0]]	=	urldecode( $p[1] );
						}
					}
				}
			}

			$s->read_timeout	=	$timeout;

			if ( $username || $password ) {
				$s->user		=	$username;
				$s->pass		=	$password;
			}

			if ($contentType == 'xml' ) {
				$s->set_submit_xml();
			} elseif ($contentType == 'json' ) {
				// available after CB 1.2.1 :
				if ( is_callable( array( $s, 'set_submit_json' ) ) ) {
					$s->set_submit_json();
				}
			}
			if ( $acceptType ) {
				$s->accept		=	$acceptType;
			}

			if ( ( (int) $port ) && ( ( $https && ( $port != 443 ) ) || ( ( ! $https ) && ( $port != 80 ) ) ) ) {
				$portPostfix	=	':' . (int) $port;
			} else {
				$portPostfix	=	'';
			}
			if ( (int) $port ){
				$s->port		=	$port;
			}
			$posturl		=	( ( $https && ( $curl_found || ! $allowHttpFallback ) ) ? 'https://' : 'http://' ) . $urlNoHttpsPrefix . $portPostfix;
			/* $return = */
			if ( $referer ) {
				$s->referer	=	$referer;
			}
			@$s->submit( $posturl, $formvars);
			$status = $s->status;
			$error = $s->error;
			$result = $s->results;

			//		}

		}
		return $error;
	}

	/**
	 * Function to decode HTTP 1.1 Transfer-Encoding: chunked
	 *
	 * @param  string  $str  String to decode (IN+OUT)
	 * @return boolean       True: A decoding has been performed, False: Wrong encoding or 0-length chunk (meaning end of persistant connection information
	 */
	private static function unchunkHttpResponse( &$str )
	{
		if ( ! is_string( $str ) or strlen( $str ) < 1 ) {
			return false;
		}

		$eol		=	"\r\n";
		$add		=	strlen($eol);
		$tmp		=	$str;
		$str		=	'';

		do {
			$tmp	=	ltrim( $tmp );
			$pos	=	strpos( $tmp, $eol );
			if ( $pos === false ) {
				return false;
			}
			$len	=	hexdec(substr( $tmp, 0, $pos ) );
			if ( $len == 0 ) {
				$str	=	'';
				return false;
			}
			if ( ! is_numeric( $len ) or $len < 0 ) {
				return false;
			}
			$str	.=	substr( $tmp, ( $pos + $add ), $len );
			$tmp	=	substr( $tmp, ( $len + $pos + $add ) );
			$check	=	trim( $tmp );
		} while( ! empty( $check ) );

		unset( $tmp );

		return true;
	}
}