<?php
/**
 * @version $Id: cbpaidSubscriptionsMgr.php 1608 2012-12-29 04:12:52Z beat $
 * @package CBSubs (TM) Community Builder Plugin for Paid Subscriptions (TM)
 * @subpackage Plugin for Paid Subscriptions
 * @copyright (C) 2007-2022 and Trademark of Lightning MultiCom SA, Switzerland - www.joomlapolis.com - and its licensors, all rights reserved
 * @license http://www.gnu.org/licenses/old-licenses/gpl-2.0.html GNU/GPL version 2
 */

use CBLib\Application\Application;
use CB\Database\Table\UserTable;
use CBLib\Registry\ParamsInterface;
use CBLib\Language\CBTxt;

/** ensure this file is being included by a parent file */
if ( ! ( defined( '_VALID_CB' ) || defined( '_JEXEC' ) || defined( '_VALID_MOS' ) ) ) { die( 'Direct Access to this location is not allowed.' ); }

/**
 * Class to manage subscriptions
 *
 */
class cbpaidSubscriptionsMgr {
	protected $_upgradesCache						=	array();

	/**
	 * Gets a single instance of the cbpaidSubscriptionsMgr class
	 *
	 * @return cbpaidSubscriptionsMgr
	 */
	public static function getInstance( ) {
		static $singleInstance						=	null;

		if ( $singleInstance === null ) {
			$singleInstance							=	new self();
		}
		return $singleInstance;
	}
	/**
	 * Returns an array with $accessPlans and their children as available for upgrades for $user at this time
	 *
	 * @param  int[]      $accessPlans
	 * @param  UserTable  $user
	 * @return int[]
	 */
	public function getUpgradablePlansWithChildrensForPlans( $accessPlans, $user ) {
		global $_CB_framework;

		// load active subscriptions into $activeSubscriptions and plans to which user can upgrade into $upgradePlans
		$activeSubscriptions						=	array();
		$upgradePlans								=	cbpaidSubscriptionsMgr::getInstance()->getUpgradeAndRenewalPossibilities( 1, ( $user ? $user->id : null ), $_CB_framework->now(), $activeSubscriptions, $accessPlans, 1, true );

		$accessPlansWithChildren					=	array();
		foreach ( $upgradePlans as $plan ) {
			if ( in_array( $plan->id, $accessPlans ) || in_array( $plan->get( 'parent' ), $accessPlans ) ) {
				$accessPlansWithChildren[]			=	$plan->id;
			}
		}
		return $accessPlansWithChildren;
	}
	/**
	 * Checks for upgrade or renewal possibilities
	 *
	 * @param  int                   $ui                     1=frontend, 2=backend
	 * @param  int                   $user_id
	 * @param  int                   $now                    system unix time
	 * @param  cbpaidUsersubscriptionRecord[]  $subscriptionsReturned  RETURNED: current subscriptions
	 *                                                               with ->status = 'A' for active ones and 'X' for expired ones. 'R' unpaid, 'C' cancelled.
	 * @param  array|null            $plansToShowOnly        array of specific plan numbers to show (so we add these plans if allowed and not spontaneous in frontend
	 * @param  int                   $subsAccess             0 has only read access, 1 has user access, 2 reserved for future Super-admin access
	 * @param  boolean               $plansToShowOnlyDoIncludeChildren  Include children with plansToShowOnly
	 * @return cbPaidProduct[]                               upgrade possibilities including _renewalDiscount in plan's currency
	 */
	public function getUpgradeAndRenewalPossibilities( $ui, $user_id, $now, &$subscriptionsReturned, $plansToShowOnly = null, $subsAccess = 1, $plansToShowOnlyDoIncludeChildren = false ) {
		global $_CB_database, $_CB_framework;

		if ( ! isset( $this->_upgradesCache[$user_id] ) ) {
			$quantity								=	1;			//TBD later !

			$paidUserExtension						=	cbpaidUserExtension::getInstance( $user_id );
			$subscriptions							=	$paidUserExtension->getUserSubscriptions( null, true );

			$user									=	CBuser::getUserDataInstance( (int) $user_id );
			$plansMgr								=	cbpaidPlansMgr::getInstance();
			$plans									=	$plansMgr->loadPublishedPlans( $user, true, 'any', null );		//TBD LATER: upgrades limiting owners

			$params									=	cbpaidApp::settingsParams();
			$enableFreeRegisteredUser				=	$params->get( 'enableFreeRegisteredUser', 1 );
			$createAlsoFreeSubscriptions			=	$params->get( 'createAlsoFreeSubscriptions', 0 );

			$noValidSubscriptionFound				=	true;
			$subscriptionsUpgradePlansIdsDiscount	=	array();	// array: [$k][$upgrade_plan->id]=discountedPrice  where $l is index in $subscriptions
			$activeExclusiveSubChild				=	array();	// array: [$parentPlanId] = true
			$notProposedParents						=	array();

			foreach ( $subscriptions as $k => $subscription ) {
				// for each user subscription:
				// 1. check if it's plan can be shown as an extra subscription possibility and/or upgrade,
				$subscription->checkRenewalUpgrade( $ui, $user, $quantity, $now, $subsAccess );

				// 2. don't propose subscription which can not be shown to the user
				if ( $subscription->_hideItsPlan && isset( $plans[$subscription->plan_id] ) ) {
					$plans[$subscription->plan_id]->_drawOnlyAsContainer =	true;
					// $notProposedParents[$subscriptions[$k]->plan_id]				=	true;
				}
				if (  ( $subscription->_hideThisSubscription || ! $subscription->checkIfValid( $now ) )
					&& ( isset( $plans[$subscription->plan_id] ) && ( $plans[$subscription->plan_id]->get( 'multiple') == 0 ) ) )
				{
					foreach ( $plans as $pk => $plan ) {
						// hidden or inactive subscription: do not display any of its children plans as upgrade possibility:
						if ( $plan->get( 'parent' ) == $subscription->plan_id ) {
							$plan->_drawOnlyAsContainer				=	true;
							$notProposedParents[$pk]            	=	true;
						}
					}
				}
				if ( $subscription->_hideThisSubscription ) {
					unset( $subscriptions[$k] );
				} elseif ( $subscription->checkIfValid( $now ) ) {
					// 3. all upgrade possibilities of this subscription
					$noValidSubscriptionFound						=	false;
					$subscriptionsUpgradePlansIdsDiscount[$subscription->id]	=	$subscription->_upgradePlansIdsDiscount;
					if ( $subscription->getPlanAttribute( 'exclusive' ) == 1 ) {
						$activeExclusiveSubChild[$subscription->getPlanAttribute( 'parent' )] =	true;
					}
				}
			}

			// add to each plan the subscriptions which can be upgraded: plan, subscription and price:
			foreach ( $plans as $plan ) {
				foreach ( $subscriptionsUpgradePlansIdsDiscount as $subscriptionId => $upgradePlansDiscount ) {
					$subscription						 =	$subscriptions[$subscriptionId];
					foreach ( $upgradePlansDiscount as $planId => $discountedPrice ) {
						if ( $plan->get( 'id' ) == $planId ) {
							$plan->_canUpgradeThisSubscription =	array( $subscription->plan_id, $subscription->id );
							$plan->_renewalDiscount            =	$discountedPrice;
						}
					}
				}
			}

			// finally remove all plans not allowed for upgrade and
			// also all exclusive plans which can't be upgraded to by no subscription
			// (already subscribed plans have already been removed by plan's _hideItsPlan instructions):
			// also memorize them as removed parent, so that children are not proposed either:
			foreach ( $plans as $plan ) {

				// remove plans not listed by default and not specifically selected:
				$resultTexts				=	array();
				if (   ( ! $plan->isPlanAllowingUpgradesToThis( $user_id, $resultTexts ) )
					|| ( ( ( $plan->get( 'propose_upgrade' ) != 1 ) && ( $ui != 2 ) ) && ! ( $plansToShowOnly && ( in_array( $plan->get( 'id' ), $plansToShowOnly ) || ( $plansToShowOnlyDoIncludeChildren && in_array( $plan->get( 'parent' ), $plansToShowOnly ) ) ) ) )
					|| ( ( $plan->get( 'exclusive' ) == 1 )
						&& ( $plan->get( 'multiple' ) == 0 )
						&& isset( $activeExclusiveSubChild[$plan->get( 'parent' )] )
						&& ( $plan->_canUpgradeThisSubscription === null ) ) )
				{
					// if ( $ui == 1 ) {	// when we are in frontend:
					if ( ! ( isset( $plan->_drawOnlyAsContainer ) && ( $plan->_drawOnlyAsContainer ) ) ) {
						$plan->_drawOnlyAsContainer =	true;
						$notProposedParents[$plan->get( 'id' )] =	true;
					}
				}
			}
			// very finally remove also children of non-authorized parent plans:
			// second case is that parent plan isn't published:
			foreach ( $plans as $plan ) {
				$parentPlanId										=	$plan->get( 'parent' );
				if ( $parentPlanId && ( isset( $notProposedParents[$parentPlanId] ) || ! isset( $plans[$parentPlanId] ) ) ) {
					$plan->_drawOnlyAsContainer =	true;
				}
			}


			// If no sbscriptions at all or no active/registered ones, and the corresponding setting allows it:
			// Find the first free lifetime one with Registered level:
			if ( ( ( count( $subscriptions ) == 0 ) || $noValidSubscriptionFound ) && $enableFreeRegisteredUser && ! $createAlsoFreeSubscriptions ) {
				$firstFreePlanId									=	null;
				$registeredUserGroup								=	$_CB_framework->getCfg( 'new_usertype' );
				foreach ( $plans as $v ) {
					if ( $v->isLifetimeValidity() && $v->isFree() && in_array( $v->get( 'usergroup' ), array( $registeredUserGroup, 0 ) ) ) {
						if ( $firstFreePlanId === null ) {
							$firstFreePlanId						=	$v->get( 'id' );
						}
						break;
					}
				}
				if ( $firstFreePlanId ) {
					$freeSub										=	new cbpaidUsersubscriptionRecord( $_CB_database );
					$freeSub->createSubscription( $user_id, $plans[$firstFreePlanId],  null, null, 'A', false );
					array_unshift( $subscriptions, $freeSub );
					$plans[$firstFreePlanId]->_drawOnlyAsContainer	=	true;
				}
			}
			$this->_upgradesCache[$user_id]							=	array( 'subscriptions' => &$subscriptions, 'plans' => &$plans );
		}
		$subscriptionsReturned										=	$this->_upgradesCache[$user_id]['subscriptions'];
		return $this->_upgradesCache[$user_id]['plans'];
	}
	/**
	 * Checks for the user's subscriptions,
	 * and if no subscription found (and free memberships not allowed):
	 * redirects or returns FALSE depending on $redirect
	 * Otherwise returns TRUE
	 *
	 * @param  string   $functionName  Name of calling function (CB API functions: getDisplayTab, getEditTab, onDuringLogin, getTabComponent, module: mod_subscriptions )
	 * @param  int      $userId        Check a specific user
	 * @param  boolean  $redirect      If should redirect in case of expiration
	 * @return boolean                 if $redirect == false: TRUE: membership valid, FALSE: membership not valid, otherwise: TRUE or REDIRECT !
	 */
	public function checkExpireMe( /** @noinspection PhpUnusedParameterInspection */ $functionName, $userId = null, $redirect = true ) {
		global $_CB_framework;

		static $notCalled			=	true;
		static $notMassExpired		=	true;
		if ( $notCalled && ( $_CB_framework->getUi() == 1 ) ) {

			if ( $userId == null ) {
				$params				=	cbpaidApp::settingsParams();
				if ( $notMassExpired && $params->get( 'massexpirymethod', 0 ) == 1 ) {
					// mass-expire 10 subscriptions at a time on the way if not exipring a particular user:
					$plansMgr		=	cbpaidPlansMgr::getInstance();
					$plansMgr->checkAllSubscriptions( 10 );
					$notMassExpired	=	false;
				}
				$userId				=	$_CB_framework->myId();
			}
			if ( $userId ) {
				// make sure to not check more than once:
				$notCalled			=	false;

				$null				=	null;

				$paidUserExtension	=	cbpaidUserExtension::getInstance( $userId );
				$subscriptionOk		=	$paidUserExtension->checkUserSubscriptions( false, $null, 'X', true );
				if ( ! $subscriptionOk ) {
					if ( $redirect ) {
						$this->_redirectExpiredMembership( $userId );
						return false;		// if we are already displaying the correct screen...
					} else {
						return false;
					}
				}
			}
		}
		return true;
	}
	/**
	 * Redirects expired user to the re-subscription screen.
	 * @access private
	 * @param  int  $userId
	 */
	protected function _redirectExpiredMembership( $userId )
	{
		global $_CB_framework;

		$params						=	cbpaidApp::settingsParams();

		$paidUserExtension			=	cbpaidUserExtension::getInstance( $userId );
		$expiredSubscriptions		=	$paidUserExtension->getUserSubscriptions( 'X' );	// check if there is any expired extensions for the text

		if ( count( $expiredSubscriptions ) > 0 ) {
			$textMessage			=	$params->get( 'subscriptionExpiredText', "Your membership has expired." );
			$expiredRedirectLink	=	$params->get( 'subscriptionExpiredRedirectLink' );
		} else {
			$textMessage			=	$params->get( 'subscriptionNeededText', "A membership is needed for access." );
			$expiredRedirectLink	=	$params->get( 'subscriptionNeededRedirectLink' );
		}

		if ( $expiredRedirectLink ) {
			$expiredRedirectLink	=	cbSef( $expiredRedirectLink, false );
		} else {
			$expiredRedirectLink	=	$_CB_framework->pluginClassUrl( array( 'plugin' => 'cbpaidsubscriptions', 'do' => 'accountexpired' ), false );
			if ( $userId ) {
				$_SESSION['cbsubs']['expireduser']	=	$userId;
			}
		}

		if ( ( Application::Input()->get( 'option' ) != 'com_comprofiler' ) || ( Application::Input()->get( 'view', Application::Input()->get( 'task' ) ) != 'pluginclass' ) || ( Application::Input()->get( 'plugin' ) != 'cbpaidsubscriptions' ) ) {
			cbRedirect( $expiredRedirectLink, CBTxt::T( $textMessage ), 'warning' );
		}
	}

	/**
	 * USED by XML interface ONLY !!! Renders url for the product
	 *
	 * @param  string           $value    Variable value ( 'massexpire' )
	 * @param  ParamsInterface  $params
	 * @return string                     HTML to display
	 */
	public function renderUrlOfAutoExpiry( $value, $params ) {
		$url	=	'index.php?option=com_comprofiler&amp;task=pluginclass&amp;plugin=cbpaidsubscriptions&amp;do=' . htmlspecialchars( $value ) . '&amp;key='
				.	md5( $params->getString( 'license_number', '' ) );
		$url	=	cbSef( $url, true, 'raw' );
		return '<a href="' . $url . '" target="_blank">' . $url . '</a>';
	}
} // end class cbpaidSubscriptionsMgr
