<?php
/**
 * @version $Id: cbpaidControllerPaychoices.php 1546 2012-12-02 23:16:25Z beat $
 * @package CBSubs (TM) Community Builder Plugin for Paid Subscriptions (TM)
 * @subpackage Plugin for Paid Subscriptions
 * @copyright (C) 2007-2022 and Trademark of Lightning MultiCom SA, Switzerland - www.joomlapolis.com - and its licensors, all rights reserved
 * @license http://www.gnu.org/licenses/old-licenses/gpl-2.0.html GNU/GPL version 2
 */

use CB\Database\Table\UserTable;
use CBLib\Registry\Registry;
use CBLib\Registry\GetterInterface;
use CBLib\Language\CBTxt;

/** ensure this file is being included by a parent file */
if ( ! ( defined( '_VALID_CB' ) || defined( '_JEXEC' ) || defined( '_VALID_MOS' ) ) ) { die( 'Direct Access to this location is not allowed.' ); }

/**
 * This class handles different payment methods
 *
 */
class cbpaidControllerPaychoices {
	/**
	 * Constructor
	 */
	public function __construct() {
	}
	/**
	 * returns the static cbpaidControllerPaychoices (and creates it if needed)
	 *
	 * @return cbpaidControllerPaychoices object
	 */
	public static function getInstance() {
		/** @var cbpaidControllerPaychoices */
		static $payMeansClass	=	null;

		if ( $payMeansClass === null ) {
			$payMeansClass	=	new self();
		}
		return $payMeansClass;
	}

	/**
	 * Gets all payment classes installed AND enabled
	 *
	 * @param  int          $owner
	 * @param  boolean      $enabled
	 * @param  string       $currency    Currency of payment that must be accepted
	 * @return cbpaidGatewayAccount[]  objects for the installed payment classes
	 */
	public function getPayAccounts( $owner = 0, $enabled = true, $currency = null )
	{
		$gatewayAccountsMgr			=	cbpaidGatewaysAccountsMgr::getInstance();
		$accounts					=	$gatewayAccountsMgr->loadEnabledAccounts( $owner, $enabled, $currency );
		return $accounts;
	}

	/**
	 * Gets a gateway account (regardless of enabled or not !)
	 *
	 * @param  int|null  $gateAccount
	 * @return cbpaidGatewayAccount|boolean  Object for the corresponding account or FALSE in case of error
	 */
	public function getPayAccount( $gateAccount )
	{
		if ( ! $gateAccount ) {
			$false				=	false;
			return $false;
		}
		$gatewayAccountsMgr			=	cbpaidGatewaysAccountsMgr::getInstance();
		/** @var cbpaidGatewayAccount $account */
		$account					=	$gatewayAccountsMgr->getObject( (int) $gateAccount );
		return $account;
	}

	/**
	 * Render the payment possibilities as radios, buttons, or URL for redirect of browser
	 *
	 * @param  cbpaidGatewaySelectorRadio[][]  $payChoicesHtmlArray
	 * @param  cbpaidPaymentBasket             $paymentBasket
	 * @param  string                          $redirectNow
	 * @param  string                          $chosenPaymentMethod
	 * @param  array                           $payChoicesHtmlRadiosArray  OUT
	 * @param  cbpaidGatewaySelector|null      $chosenPaymentSelector      OUT
	 * @return string[]
	 */
	private function _renderPayChoicesArray( $payChoicesHtmlArray, $paymentBasket, $redirectNow, $chosenPaymentMethod, &$payChoicesHtmlRadiosArray, &$chosenPaymentSelector ) {
		$ret										=	array();
		$chosenPaymentSelector						=	null;

		if ( ( $redirectNow == 'redirect' ) || ( $redirectNow === true ) ) {
			foreach ( $payChoicesHtmlArray as $drawParams ) {
				if ( is_array( $drawParams ) ) {
					if ( is_string( $drawParams[0] ) ) {
						// Redirect is possible: Just return URL:
						$ret						=	$drawParams[0];
					} else {
						// Redirect is wished but instead we got a button, as redirect was not possible (e.g. URL > 2k IE limit with Paypal encrypted payments):
						/** @var $renderer cbpaidBasketView */
						$renderer					=	cbpaidTemplateHandler::getViewer( null, 'basket' );
						$ret[]						=	$renderer->drawPaymentButton( $drawParams[0] );
					}
				}
			}
		} elseif ( $redirectNow == 'radios' ) {
			$payment_method_radios_template			=	cbpaidApp::settingsParams()->get( 'payment_method_radios_template', '' );
			/** @var $renderer cbpaidBasketView */
			$renderer								=	cbpaidTemplateHandler::getViewer( $payment_method_radios_template, 'basket' );
			$renderer->setModel( $paymentBasket );
			foreach ( $payChoicesHtmlArray as $gatewaySubMethods ) {
				if ( is_array( $gatewaySubMethods ) ) {
					foreach ( $gatewaySubMethods as $radioPaymentSelector ) {
						/** @var $radioPaymentSelector cbpaidGatewaySelectorRadio */
						$radioValue					=	$radioPaymentSelector->radioValue();
						$selected					=	( $chosenPaymentMethod === $radioValue );
						if ( $selected ) {
							$chosenPaymentSelector	=	$radioPaymentSelector;
						}
						$payChoicesHtmlRadiosArray[] =	array( $selected, $radioValue, $renderer->drawPaymentRadio( $radioPaymentSelector, $selected ) );
					}
				} elseif ( is_string( $gatewaySubMethods ) ) {
					$ret[]							=	$gatewaySubMethods;
				}
			}
		} else {
			/** @var $renderer cbpaidBasketView */
			$renderer								=	cbpaidTemplateHandler::getViewer( null, 'basket' );
			$renderer->setModel( $paymentBasket );
			foreach ( $payChoicesHtmlArray as $gatewaySubMethods ) {
				if ( is_array( $gatewaySubMethods ) ) {
					foreach ( $gatewaySubMethods as $paymentButton ) {
						$ret[]						=	$renderer->drawPaymentButton( $paymentButton );
					}
				} elseif ( is_string( $gatewaySubMethods ) ) {
					$ret[]							=	$gatewaySubMethods;
				}
			}
		}
		return $ret;
	}
	/**
	 * Gets payment methods and parameters available to pay this basket.
	 *
	 * @param  UserTable            $user
	 * @param  cbpaidPaymentBasket  $paymentBasket
	 * @param  string               $introText
	 * @param  string               $redirectNow   IN: null or 'display', OUT : 'buttons', 'radios', 'redirect'
	 * @return array
	 */
	public function getPaymentMethodsParams( $user, $paymentBasket, $introText, &$redirectNow ) {
		$params								=	cbpaidApp::settingsParams();
		$invoicingAddressQuery				=	$params->get( 'invoicing_address_query' );
		$basket_requiredterms				=	$params->get( 'basket_requiredterms' );

		$payChoicesHtmlArray				=	array();

		if ( $paymentBasket->mc_amount1 != 0 || $paymentBasket->mc_amount3 != 0 || $paymentBasket->mc_gross != 0 ) {
			$payment_method_selection_type	=	$params->get( 'payment_method_selection_type', 'buttons' );
			$payAccounts					=	$this->getPayAccounts( $paymentBasket->owner, true, $paymentBasket->mc_currency );
			$nbAccounts						=	count( $payAccounts );

			$redirectNowPossible			=	! ( ( $nbAccounts != 1 ) || $introText || ( $invoicingAddressQuery > 0 ) || ( $basket_requiredterms > 0 ) );
			if ( ( $redirectNow !== 'display' ) && $redirectNowPossible ) {
				$redirectNow				=	'redirect';
			} else {
				if ( $payment_method_selection_type == 'radios' ) {
					$redirectNow			=	'radios';
				} else {
					$redirectNow			=	'buttons';
				}
			}

			if ( $nbAccounts > 0 ) {
				foreach ( array_keys( $payAccounts ) as $k ) {
					$payClass				=	$payAccounts[$k]->getPayMean();
					$payChoicesHtmlArray[]	=	$payClass->getPaymentBasketProcess( $user, $paymentBasket, $payClass->gatewayApiVersion == '1.2.0' ? ( $redirectNow == 'redirect' ) : $redirectNow );
				}
			} else {
				$result						=	'<div class="alert alert-danger">' . sprintf( CBTxt::Th( "No payment gateway defined for the selling owner %s of these products." ), $paymentBasket->owner ) . "</div>\n";
				cbpaidApp::getBaseClass()->_setErrorMSG( $result );
			}

			$isAnyAutoRecurring				=	$paymentBasket->isAnyAutoRecurring();
			if ( ( $isAnyAutoRecurring != 1 ) && ( $paymentBasket->period1 ) && ( $paymentBasket->mc_amount1 == 0 ) )  {
				// Free trial button:
				$payChoicesHtmlArray[]		=	$this->getFreeTrialButton( $user, $paymentBasket, $redirectNow );
			}

		} else {
			// Free basket (after disounts): Free purchase button:
			$redirectNowPossible			=	! ( $introText || ( $invoicingAddressQuery > 0 ) || ( $basket_requiredterms > 0 ) );
			$redirectNow					=	$redirectNowPossible ? 'redirect' : 'buttons';

			// Free order button:
			$payChoicesHtmlArray[]			=	$this->getFreeTrialButton( $user, $paymentBasket, $redirectNow, 'order' );
		}
		return $payChoicesHtmlArray;
	}
	/**
	 * Give all rendering parameters for Free trial/Order now button
	 *
	 * @param  UserTable            $user
	 * @param  cbpaidPaymentBasket  $paymentBasket
	 * @param  string               $redirectNow    'buttons', 'radios', 'redirect'
	 * @param  string               $buttonType     'freetrial' or 'order'
	 * @return array
	 */
	private function getFreeTrialButton( $user, $paymentBasket, $redirectNow, $buttonType = 'freetrial' ) {
		cbpaidApp::import( 'processors.freetrial.freetrial' );
		$freetrial							=	new cbpaidGatewayAccountfreetrial();
		$payClass							=	$freetrial->getPayMean();
		$payClass->_button					=	$buttonType;
		return $payClass->getPaymentBasketProcess( $user, $paymentBasket, $redirectNow );
	}
	/**
	 * Give all rendering parameters for a payment button
	 * or error html
	 *
	 * @param  UserTable                   $user
	 * @param  cbpaidPaymentBasket         $paymentBasket
	 * @param  int                         $gatewayId
	 * @param  string                      $paymentType            'single', 'subscribe' or gateway-specific payment type
	 * @param  cbpaidGatewaySelector|null  $chosenPaymentSelector
	 * @return array|string
	 */
	private function getPayMethodButton( $user, $paymentBasket, $gatewayId, /** @noinspection PhpUnusedParameterInspection */ $paymentType, $chosenPaymentSelector ) {
		if ( is_numeric( $gatewayId ) ) {
			// A payment gateway has been choosen:
			$payAccounts					=	$this->getPayAccounts( $paymentBasket->owner, true );
			if ( isset( $payAccounts[$gatewayId] ) ) {
				$payClass					=	$payAccounts[$gatewayId]->getPayMean();
				$payChoices					=	$payClass->getPaymentBasketProcess( $user, $paymentBasket, $payClass->gatewayApiVersion == '1.2.0' ? false : 'buttons' );
				if ( is_array( $payChoices ) ) {
					foreach ( $payChoices as $paymentButton ) {
						if ( ( $paymentButton->paymentType == $chosenPaymentSelector->paymentType ) && ( $paymentButton->subMethod == $chosenPaymentSelector->subMethod ) ) {
							return array( $paymentButton );
						}
					}
				}
				return $payChoices;
			} else {
				return '<div class="alert alert-danger">' . CBTxt::Th("The chosen payment method is not available. Please choose another one.") . '</div>';
			}
		} elseif ( $gatewayId == 'freetrial' ) {
			// Free trial choice:
			return $this->getFreeTrialButton( $user, $paymentBasket, 'buttons' );
		} else {
			trigger_error( 'Unknown payment method', E_USER_WARNING );
			return null;
		}

	}
	/**
	 * Modify aspect of single payment button if there is only one to match the global parameter for this
	 *
	 * @param  cbpaidGatewaySelector[]  $payChoicePayButton
	 * @param  string                   $paymentType
	 * @return void
	 */
	private function modifyAspectPayMethodButton( &$payChoicePayButton, $paymentType = 'single' ) {
		if ( is_array( $payChoicePayButton ) && ( count( $payChoicePayButton ) == 1 ) ) {
			$params							=	cbpaidApp::settingsParams();
			$payment_button					=	$params->get( 'payment_button' );
			if ( $payment_button == 'paybutton' ) {
				$method						=	( $paymentType == 'subscribe' ? 'subscribe' : 'single' );
				$prmImg						=	'paybutton_' . $method . '_image';
				$prmCustImg					=	'paybutton_' . $method . '_custom_image';
				$customImage				=	trim( $params->get( $prmCustImg ) );
				if ( $customImage == '' ) {
					$customImage			=	cbpaidApp::renderCCImage( $params->get( $prmImg ), true );
				}
				if ( $customImage ) {
					$payChoicePayButton[0]->customImage		=	$customImage;
				}
				$customCSS					=	trim( $params->get( 'paybutton_' . $method . '_custom_css' ) );
				if ( $customCSS ) {
					$payChoicePayButton[0]->customCSS		=	$customCSS;
				}
			}
		}
	}
	/**
	 * Gets all allowed currencies as an array (including primary and secondary currency)
	 *
	 * @return array
	 * @todo TODO TODO add list of currencies for translations!!!
	 */
	public function getAllCurrencies( ) {
		$params								=	cbpaidApp::settingsParams();
		$currency_code						=	$params->get( 'currency_code', 'USD' );
		$secondary_currency_code			=	$params->get( 'secondary_currency_code' );
		$allowed_currencies					=	$params->get( 'allowed_currencies' );

		$allCurrencies						=	$allowed_currencies ? explode( '|*|', $allowed_currencies ) : array();

		if ( $secondary_currency_code && ! in_array( $secondary_currency_code, $allCurrencies ) ) {
			array_unshift( $allCurrencies, $secondary_currency_code );
		}
		if ( $currency_code && ! in_array( $currency_code, $allCurrencies ) ) {
			array_unshift( $allCurrencies, $currency_code );
		}
		return $allCurrencies;
	}
	/**
	 * Renders a <select> drop-down with currency
	 *
	 * @param  string  $selectedCurrency
	 * @param  string  $currencyInputName
	 * @return string
	 */
	private function drawCurrencySelect( $selectedCurrency, $currencyInputName ) {
		$allCurrencies						=	$this->getAllCurrencies();
		if ( ! in_array( $selectedCurrency, $allCurrencies ) ) {
			array_unshift( $allCurrencies, $selectedCurrency );
		}
		$values								=	array();
		foreach ( $allCurrencies as $currency ) {
			$values[]						=	moscomprofilerHTML::makeOption( $currency, CBTxt::T( $currency ) );
		}
		return moscomprofilerHTML::selectList( $values, $currencyInputName, 'class="ml-auto flex-grow-0 w-sm-100 rounded-0 form-control cpayCurrency cpayOrderCurrency" aria-label="' . htmlspecialchars( CBTxt::T( 'Currency' ) ) . '"', 'value', 'text', $selectedCurrency, 2 );
	}
	/**
	 * Renders a drop-down with currency
	 *
	 * @param  cbpaidPaymentBasket  $paymentBasket
	 * @return string
	 */
	private function displayCurrencySelector( $paymentBasket ) {
		list( $getParams, $formHiddens, $currencyInputName )	=	self::getCurrencyChangeFormParams( $paymentBasket );

		$htmlCurrency						=	$this->drawCurrencySelect( $paymentBasket->mc_currency, $currencyInputName );

		$methodsHTML						=	'<div class="cbregPaymentCurrencyChoice">'
											.		'<div class="input-group input-group-sm">'
											.			$htmlCurrency
											.			'<div class="w-sm-100 input-group-append">'
											.				'<button type="submit" id="cbregSelectCurrency" class="rounded-0 w-sm-100 btn btn-light border">' . CBTxt::Th("Change Currency") . '</button>'
											.			'</div>'
											.		'</div>'
											.	"</div>";

		$subscriptionsGUI					=	new cbpaidControllerUI();
		$result								=	'<div class="mb-2 cbregCurrencySelect">' . $subscriptionsGUI->drawForm( $methodsHTML, null, $formHiddens, $getParams ) . "</div>\n";
		$subscriptionsGUI->addcbpaidjsplugin();
		return $result;
	}
	/**
	 * Returns the URL and hidden params, and name of currency param for a form to change currency
	 *
	 * @param  cbpaidPaymentBasket  $paymentBasket
	 * @return array                ( url unsefed/unhtmlspecialchared, array( hidden form params ), name of currency input )
	 */
	public static function getCurrencyChangeFormParams( $paymentBasket ) {
		$getParams							=	$paymentBasket->getSetBasketPaymentMethodUrl( null, 'html', 'setbsktcurrency' );
		$ajaxGetParams						=	cbUnHtmlspecialchars( $paymentBasket->getSetBasketPaymentMethodUrl( null, 'raw', 'setbsktcurrency' ) );
		$formHiddens						=	array(	cbpaidApp::getBaseClass()->_getPagingParamName('act') => 'setbsktcurrency',
			'ajaxurl' => bin2hex( $ajaxGetParams ) );
		return array( $getParams, $formHiddens, 'currency' );
	}
	/**
	 * display basket and payment buttons or redirect for payment depending if multiple payment choices or intro text present:
	 *
	 * @param  UserTable            $user
	 * @param  cbpaidPaymentBasket  $paymentBasket
	 * @param  string               $introText
	 * @param  boolean              $ajax           TRUE if AJAX refresh inside #cbregPayMethodsChoice, FALSE: wraps in <div id="cbregPayMethodsChoice">
	 * @return string                               HTML  (or DOES REDIRECT if $redirectNow = ! ( ( $nbClasses != 1 ) || $introText ) == TRUE)
	 */
	public function getPaymentBasketPaymentForm( $user, $paymentBasket, $introText, $ajax = false ) {
		global $_PLUGINS;

		$result								=	null;
		$redirectNow						=	null;

		$integrationsResults				=	$_PLUGINS->trigger( 'onCbSubsBeforePaymentBasket', array( $paymentBasket, &$introText, &$redirectNow ) );
		foreach ( $integrationsResults as $intRes ) {
			if ( is_string( $intRes ) ) {
				$result						.=	$intRes;
			}
		}

		$params								=	cbpaidApp::settingsParams();
		$invoicingAddressQuery				=	$params->get( 'invoicing_address_query' );
		$basket_requiredterms				=	$params->get( 'basket_requiredterms' );
		$basket_requiredtermserror			=	$params->get( 'basket_requiredtermserror' );
		$payment_method_selection_type		=	$params->get( 'payment_method_selection_type', 'buttons' );
		$allow_select_currency				=	$params->get( 'allow_select_currency', '0' );

		$payChoicesArray					=	$this->getPaymentMethodsParams( $user, $paymentBasket, $introText, $redirectNow );

		$chosenPaymentMethod				=	$paymentBasket->gateway_account ? $paymentBasket->gateway_account . '-' . $paymentBasket->payment_type : '';		// cbGetParam( $_POST, 'payment_method' );

		$payChoicesHtmlRadiosArray			=	array();
		$chosenPaymentSelector				=	null;
		$payChoicesHtmlBottomArray			=	$this->_renderPayChoicesArray( $payChoicesArray, $paymentBasket, $redirectNow, $chosenPaymentMethod, $payChoicesHtmlRadiosArray, $chosenPaymentSelector );
		if ( $redirectNow == 'redirect' && is_string( $payChoicesHtmlBottomArray ) ) {
			cbRedirect( $payChoicesHtmlBottomArray );
		}

		$subscriptionsGUI					=	new cbpaidControllerUI();
		$subscriptionsGUI->addcbpaidjsplugin();

		if ( ( $payment_method_selection_type == 'radios') && ( $chosenPaymentMethod != '' ) && $chosenPaymentSelector ) {
			// Select button to draw:
			$payChoicePayButton				=	$this->getPayMethodButton( $user, $paymentBasket, $paymentBasket->gateway_account, $paymentBasket->payment_type, $chosenPaymentSelector );
			/** @var $chosenPaymentSelector cbpaidGatewaySelector */
			$this->modifyAspectPayMethodButton( $payChoicePayButton, $chosenPaymentSelector->paymentType );
			$dummy							=	null;
			$payChoicePayButtonHtmlArray	=	$this->_renderPayChoicesArray( array( $payChoicePayButton ), $paymentBasket, 'buttons', $chosenPaymentMethod, $payChoicesHtmlRadiosArray, $dummy );
			$payChoicesHtmlBottomArray		=	array_merge( $payChoicesHtmlBottomArray, $payChoicePayButtonHtmlArray );
		}

		// always add cancel link
		cbpaidApp::import( 'processors.cancelpay.cancelpay' );
		$cancelmethod						=	new cbpaidGatewayAccountcancelpay();
		$payClass							=	$cancelmethod->getPayMean();
		$basketCancelButton					=	$payClass->getPaymentBasketProcess( $user, $paymentBasket, 'buttons' );	// never redirectNow a cancel link :D !

		$basketHtml							=	$paymentBasket->displayBasket();

		if ( $allow_select_currency == 2 ) {
			$currencySelector				=	$this->displayCurrencySelector( $paymentBasket );
		} else {
			$currencySelector				=	null;
		}

		$cbUser								=	CBuser::getInstance( $paymentBasket->user_id, false );
		$extraStrings						=	array( 'payment_gateway'		=> $paymentBasket->gateway_account,
													   'payment_method'			=> $paymentBasket->payment_method,
													   'payment_type'			=> $paymentBasket->payment_type,
													   'payment_autorecurring'	=> $paymentBasket->isAnyAutoRecurringString() );		// "no": no, "yes": imposed by system or basket, "user-choice": chosen by user
		$txtConclusion						=	$cbUser->replaceUserVars( CBTxt::Th( $params->get('conclusion_text', 'Please click on the corresponding button below to pay:' ) ), true, false, $extraStrings, false ); // CBTxt::Th( 'Please click on the corresponding button below to pay:' )
		$txtFinal							=	$cbUser->replaceUserVars( CBTxt::Th($params->get('final_text') ), true, false, $extraStrings, false );

		$txtTerms							=	null;
		if ( $basket_requiredterms == 1 ) {
			global $_CB_database, $_CB_framework;

			$query							=	'SELECT ' . $_CB_database->NameQuote( 'params' )
											.	"\n FROM " .  $_CB_database->NameQuote( '#__comprofiler_fields' )
											.	"\n WHERE " . $_CB_database->NameQuote( 'name' ) . " = " . $_CB_database->Quote( 'acceptedterms' );
			$_CB_database->setQuery( $query );
			$tcParams						=	new Registry( $_CB_database->loadResult() );

			$cbUser							=	CBuser::getMyInstance();
			$termsOutput					=	$tcParams->get( 'terms_output', 'url' );
			$termsType						=	$cbUser->replaceUserVars( $tcParams->get( 'terms_type', 'TERMS_AND_CONDITIONS' ) );
			$termsDisplay					=	$tcParams->get( 'terms_display', 'modal' );
			$termsURL						=	cbSef( $cbUser->replaceUserVars( $tcParams->get( 'terms_url', null ) ), false );
			$termsText						=	$cbUser->replaceUserVars( $tcParams->get( 'terms_text', null ) );
			$termsWidth						=	$tcParams->get( 'terms_width', 400 );
			$termsHeight					=	$tcParams->get( 'terms_height', 200 );

			if ( ! $termsType ) {
				$termsType					=	CBTxt::T( 'TERMS_AND_CONDITIONS', 'Terms and Conditions' );
			}

			if ( ! $termsWidth ) {
				$termsWidth					=	400;
			}

			if ( ! $termsHeight ) {
				$termsHeight				=	200;
			}

			if ( ( ( $termsOutput == 'url' ) && $termsURL ) || ( ( $termsOutput == 'text' ) && $termsText ) ) {
				if ( $termsDisplay == 'iframe' ) {
					if ( is_numeric( $termsHeight ) ) {
						$termsHeight		.=	'px';
					}

					if ( is_numeric( $termsWidth ) ) {
						$termsWidth			.=	'px';
					}

					if ( $termsOutput == 'url' ) {
						$txtTerms			.=	'<div class="embed-responsive mb-2 cbTermsFrameContainer" style="padding-bottom: ' . htmlspecialchars( $termsHeight ) . ';">'
											.		'<iframe class="embed-responsive-item d-block border rounded cbTermsFrameURL" style="width: ' . htmlspecialchars( $termsWidth ) . ';" src="' . htmlspecialchars( $termsURL ) . '"></iframe>'
											.	'</div>';
					} else {
						$txtTerms			.=	'<div class="bg-light border rounded p-2 mb-2 cbTermsFrameText" style="height:' . htmlspecialchars( $termsHeight ) . ';width:' . htmlspecialchars( $termsWidth ) . ';overflow:auto;">' . $termsText . '</div>';
					}

					$txtTerms				.=	CBTxt::Th( 'TERMS_FIELD_I_AGREE_ON_THE_ABOVE_CONDITIONS', 'I Agree to the above [type].', array( '[type]' => $termsType ) );
				} else {
					$attributes				=	' class="cbTermsLink"';

					if ( ( $termsOutput == 'text' ) && ( $termsDisplay == 'window' ) ) {
						$termsDisplay		=	'modal';
					}

					if ( $termsDisplay == 'modal' ) {
						// Tooltip height percentage would be based off window height (including scrolling); lets change it to be based off the viewport height:
						$termsHeight		=	( substr( $termsHeight, -1 ) == '%' ? (int) substr( $termsHeight, 0, -1 ) . 'vh' : $termsHeight );

						if ( $termsOutput == 'url' ) {
							$tooltip		=	'<iframe class="d-block m-0 p-0 border-0 cbTermsModalURL" height="100%" width="100%" src="' . htmlspecialchars( $termsURL ) . '"></iframe>';
						} else {
							$tooltip		=	'<div class="cbTermsModalText" style="height:100%;width:100%;overflow:auto;">' . $termsText . '</div>';
						}

						$url				=	'javascript:void(0);';
						$attributes			.=	' ' . cbTooltip( $_CB_framework->getUi(), $tooltip, $termsType, array( $termsWidth, $termsHeight ), null, null, null, 'data-hascbtooltip="true" data-cbtooltip-modal="true"' );
					} else {
						$url				=	htmlspecialchars( $termsURL );
						$attributes			.=	' target="_blank"';
					}
					$txtTerms				=	CBTxt::Th( 'TERMS_FIELD_ACCEPT_URL_CONDITIONS', 'Accept <!--suppress HtmlUnknownTarget --><a href="[url]"[attributes]>[type]</a>', array( '[url]' => $url, '[attributes]' => $attributes, '[type]' => $termsType ) );
				}
			}
		} elseif ( $basket_requiredterms == 2 ) {
			$txtTerms					=	CBTxt::T( $params->get( 'basket_termsandconditions' ) );
		}

		if ($introText) {
			$result						.=	'<div class="mb-2 cbregIntro">' . CBTxt::Th( $introText ) . "</div>\n";
		}
		$result							.=	$basketHtml;

		$integrationsResults			=	$_PLUGINS->trigger( 'onCbSubsAfterPaymentBasket', array( $paymentBasket, &$result, &$txtTerms ) );
		$integrationHtml				=	null;

		foreach ( $integrationsResults as $intRes ) {
			if ( is_string( $intRes ) ) {
				$integrationHtml		.=	trim( $intRes );
			}
		}

		if ( ( $allow_select_currency == 2 ) && $integrationHtml ) {
			$result						.=	'<div class="row no-gutters cbregBasketActions">'
										.		'<div class="col-12 col-sm flex-sm-grow-1 order-2 order-sm-1">' . $integrationHtml . '</div>'
										.		'<div class="col-12 col-sm-auto flex-sm-shrink-1 order-1 order-sm-2 pl-sm-2">' . $currencySelector . '</div>'
										.	'</div>';
		} elseif ( $allow_select_currency == 2 ) {
			$result						.=	$currencySelector;
		} elseif ( $integrationHtml ) {
			$result						.=	$integrationHtml;
		}

		if ( $invoicingAddressQuery > 0 ) {
			$errorMsg					=	$paymentBasket->checkAddressComplete();
			if ( $errorMsg && ( $invoicingAddressQuery == 2 ) ) {
				$result					=	'';
				$introAddrNeeded		=	$params->get('invoicing_address_required_into_text');
				if ($introAddrNeeded) {
					$result				.=	'<div class="mb-2 cbregIntro">' . CBTxt::Th( $introAddrNeeded ) . "</div>\n";
				}
				$result					.=	$paymentBasket->renderInvoicingAddressForm( $user );	// $xmlController->handleAction( 'action', 'editinvoiceaddress' );
				return $result;
			} else {
				/* Removed to fix bug #6399 : Non-mandatory invoice address causing missing error:
				 * Later we could add a bit more nagging as a configuration option.
				 *	if ( $errorMsg ) {
				 *		cbpaidApp::getBaseClass()->_setErrorMSG( $errorMsg );
				 *	}
				 */
				$result					.=	'<div class="mb-2 cbregInvoicingAddress">'
					.	$paymentBasket->renderInvoicingAddressFieldset()
					.	'</div>';
			}
		}

		if ( $txtTerms ) {
			$accepted					=	( cbGetParam( $_POST, 'terms_accepted', 0 ) == 1 );
			$settings					=	'<div class="form-check form-check-inline cbSnglCtrlLbl cbregTermsAccept">'
										.		'<input type="checkbox" class="form-check-input required" name="terms_accepted" id="terms_accepted" value="1"' . ( $accepted ? ' checked="checked" disabled="disabled"' : '' ) . ' /> '
										.		'<label for="terms_accepted" class="form-check-label">'
										.			$txtTerms
										.		'</label>'
										.	'</div>';
			if ( ! $accepted ) {
				$settings				.=	'<span class="cb_button_wrapper">'
										.		'<button type="submit" id="cbTermsAccept" class="btn btn-sm btn-light border" title="' . htmlspecialchars( CBTxt::T( $basket_requiredtermserror ) ) . '">' . CBTxt::Th("Accept Terms") . '</button>'
										.	'</span>';
			}
			$getParams					=	$accepted ? '#' : $paymentBasket->getShowBasketUrl( false );
			$formHiddens				=	$accepted ? array( 'terms_accepted' => 1 ) : array();
			$result						.=	'<div class="mb-2 cbregTerms">' . $subscriptionsGUI->drawForm( $settings, null, $formHiddens, $getParams ) . "</div>\n";
		} else {
			$accepted					=	true;
		}

		if ( count( $payChoicesHtmlRadiosArray ) > 0 ) {
			$radios_intro_html			=	$cbUser->replaceUserVars( CBTxt::Th( $params->get( 'radios_intro_html', 'Choose your payment method:' ) ), true, false, $extraStrings, false ); // CBTxt::Th( 'Choose your payment method:' )
			$radios_conclusion_html		=	$cbUser->replaceUserVars( CBTxt::Th( $params->get( ( $chosenPaymentMethod != null ) ? 'radios_selected_conclusion_html' : 'radios_unselected_conclusion_html' ) ), true, false, $extraStrings, false );

			$getParams					=	$paymentBasket->getSetBasketPaymentMethodUrl( $user );
			$ajaxGetParams				=	cbUnHtmlspecialchars( $paymentBasket->getSetBasketPaymentMethodUrl( $user, 'raw' ) );
			$formHiddens				=	array(	cbpaidApp::getBaseClass()->_getPagingParamName('act') => 'setbsktpmtmeth', 'ajaxurl' => bin2hex( $ajaxGetParams ) );

			$htmlList					=	'<ul class="m-0 list-unstyled cbregPaymentMethodChoiceList">';

			foreach ( $payChoicesHtmlRadiosArray as $selHtmlArr ) {
				$formHiddens['payment_method']		=	$selHtmlArr[1];

				$htmlList				.=		'<li class="mb-2 cbregCCradioLi' . ( $selHtmlArr[0] ? ' cbregCCradioSelected' : null ) . '">'
										.			'<div class="rounded-0' . ( $selHtmlArr[0] ? ' border-secondary' : null ) . ' card">'
										.				'<div class="p-2 rounded-0 ' . ( $selHtmlArr[0] ? 'bg-secondary text-white' : 'border-bottom-0' ) . ' card-header cbregCCradioSelector">'
										.					$subscriptionsGUI->drawForm( $selHtmlArr[2], null, $formHiddens, $getParams )
										.				'</div>';

				if ( $selHtmlArr[0] ) {
					$htmlList			.=				'<div class="p-0 pl-3 pt-3 card-body cbregCCradioPay">';

					if ( $txtConclusion ) {
						$htmlList		.=					'<div class="mb-3 mr-3 cbregConcl">' . $txtConclusion . '</div>';
					}

					$htmlList			.=					'<div class="cbpayChoices">'
										.						implode ( '', $payChoicesHtmlBottomArray )
										.					'</div>'
										.				'</div>';
				}

				$htmlList				.=			'</div>'
										.		'</li>';
			}

			$htmlList					.=	'</ul>';

			$result						.=	( $radios_intro_html ? '<div class="mb-2 cbregPaymenMethodChoiceIntro">' . $radios_intro_html . '</div>' : '' )
										.	'<div class="mb-2 cbregPaymentChoices' . ( $txtTerms && ( ! $accepted ) ? ' hidden' : '' ) . '">'
										.		'<div class="cbregPaymentMethodsSelect">'
										.			'<div class="cbregPaymentMethodChoice ' . ( ( $chosenPaymentMethod != null ) ? 'cbregPMselected' : 'cbregPMunselected' ) . '">'
										.				$htmlList
										.				( $radios_conclusion_html ? '<div class="cbregPaymenMethodChoiceConclusion">' . $radios_conclusion_html . '</div>' : '' )
										.			'</div>'
										.		'</div>'
										.	'</div>';
		} else {
			if ( $txtConclusion ) {
				$result					.=	'<div class="mb-2 cbregConcl">' . $txtConclusion . '</div>';
			}

			$result						.=	'<div class="cbpayChoices' . ( $txtTerms && ( ! $accepted ) ? ' hidden' : '' ) . '">'
										.		implode ( '', $payChoicesHtmlBottomArray )
										.	'</div>';
		}

		$result							.=	$basketCancelButton;

		if ( $txtFinal ) {
			$result						.=	'<div class="mt-2 cbregFinalText">' . $txtFinal . '</div>';
		}

		$result							=	'<div class="cbpayBasketView">' . $result . '</div>';
		if ( ! $ajax ) {
			$result						=	'<div id="cbpayOrderContainer">'	// Needed for Javascript delegated binding
				.	$result
				.	'</div>';
		}
		return $result;
	}
}	// class cbpaidControllerPaychoices
